# Arch Perl library, Copyright (C) 2005 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::triggers;
use base 'AXP::Command';

use Arch::Name;
use Arch::TempFiles qw(temp_file_name);
use Arch::Util qw(load_file save_file run_tla);

use Data::Dumper;

sub infoline {
	"run commands for new revisions"
}

sub helptext {
	my $self = shift;
	my $prefix = $self->{prefix};

	return $self->SUPER::helptext . qq{
		This is an implementation of a new-arch-revision notification
		(sometimes also refered to as commit notification) system.  It
		enables to easily define a list of limits (versions, branches,
		categories or archives) to watch, and periodically check this
		list for new revisions.

		Every limit has an associated command that is run once for
		every new revision in that limit.  It is usually used to send
		an email containing a changelog-like entry for every new
		revision.

		Run '$prefix add LIMIT' to add new trigger,
		and '$prefix list' to list all defined triggers.
		Then place '$prefix process --quiet' to crontab.
	}
}

sub limits_option {
	(
		limits  => { sh => 'l', desc => "interpret no arguments as empty list of limits" },
	)
}

sub verbose_option {
	(
		verbose => { sh => 'v', desc => "be more verbose" },
	)
}

sub quiet_option {
	(
		quiet   => { sh => 'q', desc => "be quiet" },
	)
}

sub dump_perl_data ($) {
	local $Data::Dumper::Indent;
	local $Data::Dumper::Pair;
	local $Data::Dumper::Quotekeys;  
	local $Data::Dumper::Terse;
	$Data::Dumper::Indent = 1;
	$Data::Dumper::Pair = ' => ',
	$Data::Dumper::Quotekeys = 0;
	$Data::Dumper::Terse = 1;
	return Data::Dumper->Dump([ shift ]);
}

sub triggers_dir ($) {
	my $self = shift;
	return $self->{triggers_dir} ||= $self->setup_config_dir("triggers");
}

sub all_trigger_names ($) {
	my $self = shift;

	return $self->{all_trigger_names} if $self->{all_trigger_names};

	my $triggers_dir = $self->triggers_dir;
	opendir(DIR, $triggers_dir) || die "Can't opendir $triggers_dir: $!\n";

	# get a list of all valid non-empty versions
	my @versions = grep {
		my $dir = "$triggers_dir/$_";
		-d $dir && (-f "$dir/action" || rmdir($dir) && 0) &&
		do {
			s!%!/!;
			!/^\./ && (Arch::Name->is_valid($_, "archive+")
				|| warn("Ignoring non archive/version $_\n") && 0)
		}
	} readdir(DIR);
	closedir(DIR);
	return $self->{all_trigger_names} = \@versions;
}

sub get_trigger ($$;$) {
	my $self = shift;
	my $name = shift;
	my $is_new = shift || 0;

	die "No limit argument (number or arch name) is given, try --help\n"
		unless $name;

	my $all_trigger_names = $self->all_trigger_names($is_new);
	$name = $all_trigger_names->[$name - 1]
		if !$is_new && $name =~ /^\d+$/ && @$all_trigger_names >= $name && $name > 0;

	die "Invalid number argument ($name), try 'list'\n"
		if $name =~ /^\d+$/;
	die "No archive/version argument ($name), try 'list'\n"
		unless Arch::Name->is_valid($name, "archive+");
	die($is_new? "Trigger $name already exists, try 'list' or 'delete'\n": "No trigger $name found\n")
		unless $is_new xor { map { $_ => 1 } @$all_trigger_names }->{$name};

	my $dir = $self->triggers_dir;
	my $subdir = $name; $subdir =~ s!/!%!;	
	$dir .= "/$subdir";
	die($is_new? "Directory $dir already exists\n": "No expected directory $dir found\n")
		unless $is_new xor -d $dir;

	my $trigger = {
		dir => $dir,
		name => $name,
		vars => {},
		state => [],
		action => "",
		vars_file => "$dir/vars",
		state_file => "$dir/state",
		action_file => "$dir/action",
		is_slave => 0,
	};

	return $trigger;
}

sub create_trigger ($$) {
	my $self = shift;
	my $name = shift;

	my $trigger = $self->get_trigger($name, "new");
	mkdir($trigger->{dir}, 0777);

	return $trigger;
}

sub load_trigger ($$) {
	my $self = shift;
	my $name = shift;

	my $trigger = $self->get_trigger($name);
	my $vars_dump;

	load_file($trigger->{vars_file}, \$vars_dump);
	load_file($trigger->{state_file}, $trigger->{state});
	load_file($trigger->{action_file}, \$trigger->{action});

	$trigger->{vars} = eval $vars_dump;
	$trigger->{is_slave} = -l $trigger->{action_file};

	return $trigger;
}

sub save_trigger ($$) {
	my $self = shift;
	my $trigger = shift;

	save_file($trigger->{vars_file}, dump_perl_data($trigger->{vars}));
	save_file($trigger->{state_file}, $trigger->{state});
	save_file($trigger->{action_file}, \$trigger->{action})
		unless $trigger->{is_slave};
}

sub edit_trigger ($$) {
	my $self = shift;
	my $trigger = shift;

	my $editor = $ENV{EDITOR} || (-x "/usr/bin/nano"? 'nano': 'pico -bewz');
	my $tmp_file = temp_file_name("triggers");

	save_file($tmp_file, \$trigger->{action});
	my $status = !system("$editor $tmp_file");
	load_file($tmp_file, \$trigger->{action});

	return $status;
}

sub skip_revisions ($$) {
	my $self = shift;
	my $trigger = shift;

	my $name = $trigger->{name};
	my $tmp_file = temp_file_name();

	save_file($tmp_file, $trigger->{state});
	run_tla("abrowse --force --omit-empty --since", $tmp_file, "--snap", $tmp_file, $name)
		or die "abrowse failed with status $?\nwhile skipping existing $name revisions\n";
	load_file($tmp_file, $trigger->{state});
}

sub apply_vars ($$$) {
	my $self = shift;
	my $trigger = shift;
	my $vars = shift;

	while (my ($key, $value) = each %$vars) {
		$value ne ""
			? $trigger->{vars}->{$key} = $value
			: delete $trigger->{vars}->{$key}
	}
}

1;
