/*
 * IndexSearcherCache.java
 *
 * See package.html for licence
 *
 * Created on 2002. j???lius 15., 10:57
 */

package nz.net.catalyst.lucene.cache;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.lucene.index.IndexReader; 
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
/**
 * 
 * This is a <code>SearcherCache</code> that manages <code>IndexSearcher</code>s.
 * To create a cache one must specify the file system directory where the index
 * is located. If this directory was modified the searcher is recreated.<br/>
 *
 * This can be improved to manage searcher on every kind of <code>Directory</code>
 * (such as RAMDirectory). To do this one should add a <code>lastModified</code>
 * method to Directory
 *
 * Modified by Hamish Carpenter (hamish@catalyst.net.nz) in order to make the 
 * IndexSearcherCache a singleton class.  I have also added a debug boolean.
 *
 * @author  Peter Halacsy
 */
public class IndexSearcherCache extends SearcherCache {
    /** The directory where the index is located */
    protected File m_directory;
    private Directory directory;
    private boolean debug = false;
    private static final Map instances = new HashMap();     // path --> instance of this class
    
        /** Creates a new instance; for every path one instance is allowed
         *  we are storing the key as dfirecotry because there is some lucene magic that can
         *  make them a key.
         * Quote from lucene api docs:
         * <quote>    Directories are cached, so that, for a given canonical path, the same FSDirectory instance will always be returned. This permits synchronization on directories.</quote>
         * @param path Path to the index directory.  For example /lucene/index
         * @throws IOException If the index cannot be opened
         * @return A IndexSearcherCache for the required index.
         */
    public static IndexSearcherCache getInstance(File path) throws IOException {
        Directory directory = FSDirectory.getDirectory(path, false);
        
        synchronized(instances) {
            IndexSearcherCache isc = (IndexSearcherCache) instances.get(directory);
            if(isc == null) {
                // creating new
                isc = new IndexSearcherCache(path);
                instances.put(directory, isc);
            }
            return isc;
        }
    }
    
    /** singleton */
    private IndexSearcherCache(File path) throws IOException {
        this.m_directory = path;
    }    
    
    public void setDirectory(File directory) {
        m_directory = directory;
    }

    public File getDirectory() {
        return m_directory;
    }


    /** This is called by getSearcher to decide whether a new searcher instance
     * has to be created.
     * @return The age status of the searcher, true if the filesystem directory
     * has been modified since the cached searcher object was created.
     * @param searcher The searcher to test if old or not.
     */
    protected boolean isSearcherOld(CachedSearcher searcher) {
        if (debug) System.out.println("is searcher old: " + (m_directory.lastModified() > searcher.getCreationTime()));
        return m_directory.lastModified() > searcher.getCreationTime();
    }


    /** This method is called by getSearcher if fresh searcher is needed. It
     * creates a cached IndexSearcher.
     * @throws IOException If index cannot be opened.
     * @return A cached searcher
     */
    protected CachedSearcher createNewCachedSearcher() throws IOException {
        if (debug) System.out.println("new searcher created");
        return new CachedSearcher(new IndexSearcher(IndexReader.open(m_directory)));
    }
}
