/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.IOException;

import nz.net.catalyst.ELog;
import nz.net.catalyst.Log;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexWriter;

/**
 * Optimize the Lucene Index directory.
 * 
 * More information can be found in the Lucene Documentation.
 * 
 * Put simply, optimization, megres different segments of the index into
 * one.  This reduces the total number of files needed to be held open and 
 * decreases search time slightly.
 * 
 * This may take a long time on large indexes with multiple segments.
 * 
 * When testing a test index with 1 Million small documents an optimize took
 * 7 minutes.  This would be expected to be longer with larger documents and 
 * more fields indexed (and stored).
 */

public class Optimize implements IPackage, Constants {
  private final Transmission input;
  private Application application;

  Optimize(Transmission transmission) {
    input = transmission;
  }

  Transmission execute() {
    long queryStart = System.currentTimeMillis();
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());

    String appName = input.get(APPLICATION, NO_APP);

    try {
      application = Application.getAppOrDefault(appName);
    } catch (ApplicationMissingException e) {
      return error(e.getMessage());
    }

    input.setApplication(application);
    Analyzer analyzer = Application.getAnalyzer(input); // Don't care which one
    File luceneStoreDir = Application.getIndexDirectory(application);
    WriterControl writerControl = null;

    long beforeOpen = System.currentTimeMillis();
    long afterWriterOpen;
    long afterOptimize;
    boolean succeed = false;
	Log.log(ELog.DEBUG, "About to optimize index in: " + luceneStoreDir);
	
    try {
      writerControl = WriterControl.getWriterControl(luceneStoreDir, analyzer);
      IndexWriter writer = writerControl.getIndexWriter();
      afterWriterOpen = System.currentTimeMillis();
      writer.optimize();
      afterOptimize = System.currentTimeMillis();
      succeed = true;
	} catch (IOException e) {
	  String message = "Error while optimizing " + luceneStoreDir +
		": " +e.toString();
	  Log.log(ELog.ERROR, message);
	  return error(message);
    } catch (ArrayIndexOutOfBoundsException e) {
      String message = "Error while backing up (probable index corruption) " 
      					+ luceneStoreDir 
      					+ ": " 
      					+ e.toString();
      Log.log(ELog.ERROR, message);
      return error(message);
    } finally {
      if (writerControl != null) {
        try
        {
          // We must always free the IndexWriter for other users.
          writerControl.release();
        }
        catch (Throwable e)
        {
          String message = "Error while closing IndexWriter: " + e.toString();
          Log.log(ELog.ERROR, message);
          if (succeed)
            return error(message);
        }
      }
    }
    long afterWriterClose = System.currentTimeMillis();
    response.add("Open",  String.valueOf(afterWriterOpen - beforeOpen));
    response.add("Optimize", String.valueOf(afterOptimize - afterWriterOpen));
    response.add("Close",  String.valueOf(afterWriterClose - afterOptimize));
    return response;
  }

  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An INDEX-RESPONSE Transmission
   */

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    return response;
  }
}
