/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "date_time_utils.h"


int date_time_julian_day_get_current ()
// Returns the julian day for today.
{
  // Get the current date/time.
  time_t timevalue = time (0);
  tm date_struct;
  // Divide it up into its parts.
  localtime_r (&timevalue, &date_struct);
  GDateMonth month = GDateMonth (date_struct.tm_mon + 1);
  GDateYear year = date_struct.tm_year + 1900;
  // New GDate object using the current date.
  GDate *date;
  date = g_date_new_dmy (date_struct.tm_mday, month, year);
  // Get the Julian day out of it.
  int julian_day = g_date_get_julian (date);
  // Free memory.
  g_date_free (date);
  // Return Julian day.
  return julian_day;
}


ustring date_time_julian_human_readable (int julian_day, bool weekday)
// Returns a Julian day in human readable form.
{
  ustring returnvalue;
  GDate *date;
  date = g_date_new_julian (julian_day);
  gchar buf[1024];
  if (weekday)
    g_date_strftime(buf, 1000, "%A, %e %B %Y", date);
  else
    g_date_strftime(buf, 1000, "%e %B %Y", date);
  returnvalue = buf;
  g_date_free (date);
  return returnvalue;
}


int date_time_julian_day_get_parse (const ustring& date)
// This takes a human readable date, and tries to get the Julian day out of it.
{
  // Trim superfluous characters off.
  ustring cleaned_date = trim (date);
  // If the date has a time part, cut that off.
  size_t offposition = cleaned_date.find (":");
  if (offposition != string::npos) {
    cleaned_date.erase (offposition, cleaned_date.length() - offposition);
    offposition = cleaned_date.find_last_of(" ");
    if (offposition != string::npos) {
      cleaned_date.erase (offposition, cleaned_date.length() - offposition);
    }
  }
  // New GDate object.
  GDate *gdate;
  gdate = g_date_new ();
  // Parse the date.
  g_date_set_parse (gdate, cleaned_date.c_str());
  // Julian day to return.
  int julian_day;
  // Validate it. 
  if (g_date_valid (gdate)) {
    julian_day = g_date_get_julian (gdate);
  } else {
    // Not valid, so take today's Julian day.
    julian_day = date_time_julian_day_get_current ();
  }
  // Free memory.
  g_date_free (gdate);
  // Return result
  return julian_day;
}


int date_time_julian_day_get_parse (guint year, guint month, guint day)
{
  // New GDate object.
  GDate *gdate;
  gdate = g_date_new ();
  // Parse the date.
  g_date_set_day (gdate, day);
  g_date_set_month (gdate, GDateMonth (month));
  g_date_set_year (gdate, year);
  // Julian day to return.
  int julian_day;
  // Validate it. 
  if (g_date_valid (gdate)) {
    julian_day = g_date_get_julian (gdate);
  } else {
    // Not valid, so take today's Julian day.
    julian_day = date_time_julian_day_get_current ();
  }
  // Free memory.
  g_date_free (gdate);
  // Return result
  return julian_day;
}


void date_time_normal_get_year_month_day (guint32 julian_day, guint& year, guint& month, guint& day)
{
  // New GDate object.
  GDate *gdate;
  gdate = g_date_new ();
  // Set Julian day.
  g_date_set_julian (gdate, julian_day);
  // Return results.
  year = g_date_get_year (gdate);
  month = g_date_get_month (gdate);
  day = g_date_get_day (gdate);
  // Free memory.
  g_date_free (gdate);
}
