/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "formatter.h"
#include "fop.h"
#include "xep.h"
#include "xslformatter.h"
#include "gwrappers.h"
#include "directories.h"
#include "shell.h"
#include "session.h"


bool formatter_present (GtkWidget * window)
// Returns whether a script to run a formatter can be found.
// Gives message if not found.
{
  return xep_present (window);
}


int formatter_convert_to_pdf (const ustring& xslfo, const ustring& pdf, ProgressWindow& progresswindow, NoteCaller * footnotecaller, NoteCaller * crossreferencecaller)
// Converts file "xslfo" to file "pdf", and returns the result.
{
  // Remove any previously created pdf file, lest it would be shown in case of 
  // conversion errors.
  unlink (pdf.c_str());

  // Get the filename of the logfile.
  ustring logfile;
  logfile = gw_build_filename (directories_get_temp(), "formatter.log");

  // Formatting is done in three stages:
  // 1. Convert from the xslfo file to the xep intermediate format.
  // 2. If need be, modify that intermediate file.
  // 3. Convert from the xep intermediate file to pdf output.  
  ustring xep_filename = xslfo + ".xep";
  ustring formatter_command;  

  // Show progress.
  progresswindow.set_text ("Typesetting pages ...");
  progresswindow.set_iterate (0, 1, 100);
  // Start the first formatting process in the background.
  formatter_command = xep_convert_from_fo_to_xep (xslfo, xep_filename, logfile);
  // If we did not get a formatting command, return error.
  if (formatter_command.empty())  
    return -1;
  // Wait for the formatter to finish.
  while (program_is_running (formatter_command)) {
    progresswindow.iterate ();
    if (progresswindow.cancel) {
      // Kill formatter.
      ustring command = "killall -HUP";
      command.append (shell_quote_space (formatter_command));
      system (command.c_str());
      // Remove pdf file.
      unlink (pdf.c_str());
      // Return with 0 so as to give no error message.
      return 0;
    }
    g_usleep (1000000);
  }
  
  // Read the logfile made by the formatter
  ReadText rt1 (logfile, true, false);
  bool error1 = false;
  for (unsigned int i = 0; i < rt1.lines.size(); i++) {
    // Add to logfile.
    gw_message (rt1.lines[i]);
    // See if errors are there.
    if (rt1.lines[i].find ("ERROR") != string::npos) 
      error1 = true;
    if (rt1.lines[i].find ("error") != string::npos) 
      error1 = true;
  }
  // Signal when error is there.
  if (error1)
    return -1;

  // Do the middle stage: restarting footnote numbering each page.
  xep_convert_footnote_callers (xep_filename, footnotecaller, crossreferencecaller, progresswindow);
  
  // Do the last stage of the conversion.
  // Show progress.
  progresswindow.set_text ("Writing .pdf file ...");
  progresswindow.set_iterate (0, 1, 100);
  formatter_command = xep_convert_from_xep_to_pdf (xep_filename, pdf, logfile);
  // If we did not get a formatting command, return error.
  if (formatter_command.empty())  
    return -1;
  // Wait for the formatter to finish.
  while (program_is_running (formatter_command)) {
    progresswindow.iterate ();
    if (progresswindow.cancel) {
      // Kill formatter.
      ustring command = "killall -HUP";
      command.append (shell_quote_space (formatter_command));
      system (command.c_str());
      // Remove pdf file.
      unlink (pdf.c_str());
      // Return with 0 so as to give no error message.
      return 0;
    }
    g_usleep (1000000);
  }
  
  // Read the logfile made by the formatter
  ReadText rt2 (logfile, true, false);
  bool error2 = false;
  for (unsigned int i = 0; i < rt2.lines.size(); i++) {
    // Add to logfile.
    gw_message (rt2.lines[i]);
    // See if errors are there.
    if (rt2.lines[i].find ("ERROR") != string::npos) 
      error2 = true;
    if (rt2.lines[i].find ("error") != string::npos) 
      error2 = true;
  }
  // Signal when error is there.
  if (error2)
    return -1;

  // Check the pdf file is there.
  if (!g_file_test (pdf.c_str(), G_FILE_TEST_IS_REGULAR))
    return -1;
  
  return 0;
}
