/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef LOADSSIMULATIONDRIVER_H
#define LOADSSIMULATIONDRIVER_H

#include <QObject>
#include <QMap>
class QTimer;

//class declared in this file
class LoadsMovie;
class LoadsManager;
class Loads;

/** This class manages the display of a simulation (either an already made
 *  simulation opened in a single lml file by LoadsMovie, or a simple step
 *  given by a simulation motor).
 *
 *  It is just a utility class for handling the display of loads a given time,
 *  using a timer.
 *
 *	
 */
class LoadsSimulationDriver : public QObject {

  Q_OBJECT

public:
    /** Constructor.
      * Set the load manager, and (initial) current loads
      * (if the loads ptr is null, then all atoms initial positions will be stored; just in case
      * they have to move in the future)
      */
    LoadsSimulationDriver(LoadsManager *, Loads *);

    /// Destructor
    virtual ~LoadsSimulationDriver();

    /** update the display using the current loads (and refresh the 3D viewers if parameter is true)
     * @param force if true force display update independantly from refresh dt (default is false)
     */
    void updateDisplay(bool force);

    /// set tmin, tmax and dt to default
    void resetToDefault();

    /// set default tmin (using the current loads first event)
    void resetTMinToDefault();

    /// get tmin
    double getTMin() const;

    /// set the new tMin
    void setTMin(double);

    /// set default tmax (using the current loads last event)
    void resetTMaxToDefault();

    /// get tmax
    double getTMax() const;

    /// set the new tMax
    void setTMax(double);

    /// set the delta-t (default is 0.1)
    void setDt(double dt);

    /// get delta-t
    double getDt() const;

    /// get time
    double getTime() const;

    /// set the time to a different value ("jump" to a given time)
    void setTime(double);

    /// get delta-t between 3d view refresh
    double getRefreshDt() const;

    /// set delta-t between 3d view refresh (default 0.1)
    void setRefreshDt(double);

    /// get the next planed time when the 3D will be refreshed
    double getNextRefreshTime();

    /// set the current loads to be displayed
    void setLoads(Loads *);

    /// starts the timer that will repeatedly call play and emit doOneStep
    void startTimer();

    /// stop the timer
    void stopTimer();

    /// is the timer currently running
    bool isTimerActive();

    /// reset the 3D positions to the initial value (stored in the map during init), (do not update the display)
    void resetPositions();

    /// rewind time to tMin (do not update the display)
    void rewind();

    /// increase the timer interval by 5ms, thus calling doOneStep less often
    void slower();

    /// decrease the timer interval by 5ms, thus calling doOneStep more often
    void quicker();

    /// set the timer interval to maxSpeed (1000Hz)
    void maxSpeed();

signals:

    /// signal call when one step is to be performed
    void doOneStep();

public slots:

    /// Slot called when at each timer tick (i.e. after the same interval of time)
    void play();

private :
    /// private struct to store the initial positions
    struct Position {
      double x;
      double y;
      double z;
    };

    /// the load manager
    LoadsManager * myLM;

    /// the current loads
    Loads * currentLoads;

    /** init display and initial positions.
     *  stores all the initial position of the atoms implied in the loads.
     *  if current loads is NULL, then all the atoms will have their initial positions stored.
     */
    void init();

    /// initial positions of all the targets
    QMap<unsigned int, Position> initialPositionMap;

    /// the simulation timer
    QTimer *timer;

    /// simulation time
    double t;

    /// the delta-t used to change t
    double dt;

    /// first event time
    double tMin;

    /// last event time
    double tMax;

    /// the delta-t between two refresh of the 3D view
    double refreshDt;

    /// last time we updated the display
    double lastUpdateTime;

    /// last time the display was refresh
    double lastRefreshTime;

    /// get an initial position using the key (i.e. targetId)
    void getInitialPosition(unsigned int, double [3]);

    /// update the positions depending on the current loads (and refresh the DCs position if parameter is true)
    void updatePositions(bool force);

    /// update Atom data
    void updateAtomData();

    /// the reference pm positions (used to display the distances, ...)
    QMap<unsigned int, Position> referencePositionMap;

    /// timer interval in ms
    int interval;
};

inline void LoadsSimulationDriver::getInitialPosition(unsigned int targetId, double pos[3]) {
    QMap<unsigned int, Position>::const_iterator it = initialPositionMap.find(targetId);
    if (it == initialPositionMap.end())
        pos[0] = pos[1] = pos[2] = 0.0;
    else {
      pos[0] = it.value().x;
      pos[1] = it.value().y;
      pos[2] = it.value().z;
    }
}


#endif // LOADSSIMULATIONDRIVER_H


