/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "VtkImageComponentExtension.h"
#include "VtkImageComponent.h"
#include "RawImageComponent.h"
using namespace camitk;

// -- QT stuff
#include <QFile>
#include <QTextStream>
#include <QFileInfo>
#include <QMessageBox>
#include <QDir>

// vtk image writers
#include <vtkImageWriter.h>
#include <vtkJPEGWriter.h>
#include <vtkPNGWriter.h>
#include <vtkTIFFWriter.h>
#include <vtkBMPWriter.h>
#include <vtkPNMWriter.h>
#include <vtkMetaImageWriter.h>

// --------------- declare the plugin -------------------
Q_EXPORT_PLUGIN2(vtkimage, VtkImageComponentExtension);


// --------------- getName -------------------
QString VtkImageComponentExtension::getName() const {
    return "vtkImages Component";
}

// --------------- getDescription -------------------
QString VtkImageComponentExtension::getDescription() const {
    return "Manage any file type supported by vtk in <b>CamiTK</b>";
}

// --------------- getFileExtensions -------------------
QStringList VtkImageComponentExtension::getFileExtensions() const {
    QStringList ext;
    ext << "jpg" << "png" << "tiff" << "tif" << "bmp" << "pbm" << "pgm" << "ppm";
    ext << "mhd" << "mha" << "test";
    ext << "raw";
    return ext;
}

// --------------- open -------------------
Component * VtkImageComponentExtension::open(const QString & fileName) throw (AbortException) {
    try {
        if ( fileName.endsWith(".raw") )
            return new RawImageComponent( fileName );
        else
            return new VtkImageComponent( fileName );

    }
    catch (AbortException e) {
        throw e;
    }
}

// --------------- save -------------------
bool VtkImageComponentExtension::save(Component* component) const {
    ImageComponent* img = dynamic_cast<ImageComponent*>(component);
    if (!img)
        return false;
    else {
        QFileInfo fileInfo(component->getFileName());
        vtkSmartPointer<vtkImageData> image = img->getImageData();

        if (fileInfo.completeSuffix() == "raw") {
            //-- save as raw
            // vtk writer
            vtkSmartPointer<vtkImageWriter> writer;

            // write the image as raw data
            writer = vtkSmartPointer<vtkImageWriter>::New();
            writer->SetFileDimensionality(3);
            writer->SetFileName(fileInfo.absoluteFilePath().toStdString().c_str());
            writer->SetInput(image);

            try {
                writer->Write();
            }
            catch (...) {
                return false;
            }

            // Write info in a separted file
            QFile infoFile(fileInfo.absoluteDir().absolutePath() + fileInfo.baseName() + ".info");
            infoFile.open(QIODevice::ReadWrite | QIODevice::Text);
            QTextStream out(&infoFile);
            out << "Raw data info (Written by CamiTK VtkImageComponentExtension)" << endl;
            out << "Data Filename:\t" << fileInfo.absoluteFilePath() << endl;
            int * dims = image->GetDimensions();
            out << "Image dimensions:\t" << dims[0] << "\t" << dims[1] << "\t" << dims[2] << endl;
            out << "Voxel storage type:\t" << image->GetScalarTypeAsString() << endl;
            out << "Number of scalar components:\t" << image->GetNumberOfScalarComponents() << endl;
            double * voxelSpace = image->GetSpacing();
            out << "Voxel spacing:\t" << voxelSpace[0] << "\t" << voxelSpace[1] << "\t" << voxelSpace[2] << endl;
            double * imageOrigin = image->GetOrigin();
            out << "Image origin:\t" << imageOrigin[0] << "\t" << imageOrigin[1] << "\t" << imageOrigin[2] << endl;
            infoFile.flush();
            infoFile.close();
            return true;
        }
        else {
            //-- save as vtk image
            // vtk writer
            vtkSmartPointer<vtkImageWriter> writer;

            // filename extension
            QString fileExt = fileInfo.completeSuffix();

            // filename prefix (for 2D image series)
            QString filePattern = fileInfo.absoluteDir().absolutePath() + fileInfo.baseName();
            filePattern.append("_%04u.").append(fileExt);

            // Writer initialization, depending on file extension
            if (QString::compare(fileExt, "jpg", Qt::CaseInsensitive) == 0) {
                writer = vtkSmartPointer<vtkJPEGWriter>::New();
                writer->SetFileDimensionality(2);
            }
            else
                if (QString::compare(fileExt, "png", Qt::CaseInsensitive) == 0) {
                    writer = vtkSmartPointer<vtkPNGWriter>::New();
                    writer->SetFileDimensionality(2);
                }
                else
                    if ((QString::compare(fileExt, "tiff", Qt::CaseInsensitive) == 0) ||
                            (QString::compare(fileExt, "tif",  Qt::CaseInsensitive) == 0)) {
                        writer = vtkSmartPointer<vtkTIFFWriter>::New();
                        writer->SetFileDimensionality(2);
                    }
                    else
                        if (QString::compare(fileExt, "bmp", Qt::CaseInsensitive) == 0) {
                            writer = vtkSmartPointer<vtkBMPWriter>::New();
                            writer->SetFileDimensionality(2);
                        }
                        else
                            if ((QString::compare(fileExt, "pbm", Qt::CaseInsensitive) == 0) ||
                                    (QString::compare(fileExt, "pgm", Qt::CaseInsensitive) == 0) ||
                                    (QString::compare(fileExt, "ppm", Qt::CaseInsensitive) == 0)) {
                                writer = vtkSmartPointer<vtkPNMWriter>::New();
                                writer->SetFileDimensionality(2);
                            }
                            else
                                if ((QString::compare(fileExt, "mhd", Qt::CaseInsensitive) == 0) ||
                                        (QString::compare(fileExt, "mha", Qt::CaseInsensitive) == 0)) {
                                    writer = vtkSmartPointer<vtkMetaImageWriter>::New();
                                    writer->SetFileDimensionality(3);
                                }
                                else {
                                    QMessageBox::warning(NULL, "Saving Error", "Cannot save file: unrecognized extension \"." + fileExt + "\"");
                                    return false;
                                }

            if (writer) {
                if ((image->GetDataDimension() == 2 ) ||
                        (QString::compare(fileExt, "mhd", Qt::CaseInsensitive) == 0) ||
                        (QString::compare(fileExt, "mha", Qt::CaseInsensitive) == 0))
                    writer->SetFileName(fileInfo.absoluteFilePath().toStdString().c_str());
                else
                    writer->SetFilePattern(filePattern.toStdString().c_str());

                writer->SetInput(image);
                writer->Write();

            }

            return true;
        }
    }
}
