#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "databuf.h"
#include "socket.h"
#include "camcmd.h"
#include "log.h"

#define MODNAME "camcmd"

/*
 * camcmd_signon:  Attempt handshaking and negotiation with the
 *                 camera control port.
 *
 * Arguments:      cmdsock = Socket connect to command port
 *                 pword   = Password to connect to the command port
 *
 * Return values:  Returns -1 on negotiation failure, else 0
 *
 * Bugs:           Insecure over network sniffing traffic 
 */

static
int camcmd_signon( Socket *cmdsock, const char *pword ){
  char pword_buf[ CAMCMD_PWORD_LEN ];
  DataBuf *writebuf, *readbuf;
  int writeres;

  if( (writebuf = databuf_new()) == NULL ){
    camserv_log( MODNAME, "Error allocating databuffer!" );
    return -1;
  }

  if( (readbuf = databuf_new()) == NULL ){
    camserv_log( MODNAME, "Error allocating databuffer!" );
    databuf_dest( writebuf );
    return -1;
  }

  /* Write the password */
  strncpy( pword_buf, pword, sizeof( pword_buf ) -1 );
  pword_buf[ sizeof( pword_buf ) - 1  ] = '\0';
  databuf_buf_set( writebuf, pword_buf, sizeof( pword_buf ));
  while( (writeres = databuf_write( writebuf, socket_query_fd( cmdsock))) ==1);
  if( writeres == -1 ){
    camserv_log( MODNAME, "Error sending password" );
    databuf_dest( writebuf );
    databuf_dest( readbuf );
    return -1;
  }


    
  databuf_dest( writebuf );
  databuf_dest( readbuf );
  return 0;
}

void camcmd_disconnect( Socket *cmdsock ){
  socket_dest( cmdsock );
}

Socket *camcmd_connect( const char *hostname, int port, const char *pword ){
  Socket *res;

  if( (res = socket_connect( hostname, port )) == NULL ){
    camserv_log( MODNAME, "Connection failure" );
    return NULL;
  }

  /* Attempt negotiation */
  if( camcmd_signon( res, pword ) == -1 ){
    camserv_log( MODNAME, "Negotiation failure" );
    socket_dest( res );
    return NULL;
  }

  return res;
}
