/*
 *  Name: big2gb
 *  Documentation:
 *    Convert BIG5 file to GB file
 *  Version: $$Id: big2gb.c 1.0 1998/03/31 01:41:00 blin Exp blin $$
 *
 *  NOTE:
 *   This program is based on the following programs:
 *      dos2unix/unix2dos  Benjamin Lin(blin@socs.uts.edu.au)
 *      hc 3.0             Ricky Yeung(Ricky.Yeung@eng.sun.com) and
 *                         Fung F. Leeconvert.c(lee@umunhum.stanford.edu)
 *          (The GBToBig5Tbl & Big5ToGBTbl is generated from hc 3.0's hc.tab)
 */

#define RCS_AUTHOR   "$$Author: blin $$"
#define RCS_DATE     "$$Date: 1998/03/31 01:41:00 $$"
#define RCS_REVISION "$$Revision: 1.0 $$"
#define VER_AUTHOR   "Rui He"
#define VER_DATE     "1998.03.31"
#define VER_REVISION "1.0"

/* #define DEBUG */

#ifdef __MSBig5__
#  include <dir.h>
#endif __MSBig5
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <utime.h>
#include <sys/stat.h>
#include "big2gb.h"


#ifdef __MSBig5__
#  define R_CNTRL   "rb"
#  define W_CNTRL   "wb"
#else  __MSBig5__
#  define R_CNTRL   "r"
#  define W_CNTRL   "w"
#endif __MSBig5__


typedef struct
{
  int NewFile;                          /* is in new file mode? */
  int Quiet;                            /* is in quiet mode? */
  int KeepDate;                         /* should keep date stamp? */
  int KeepPerm;                         /* should keep permision stamp? */
} CFlag;


void PrintUsage(void)
{
  fprintf(stderr, "big2gb %s Copyright (c) 1994-1998 (%s)\n", VER_REVISION, VER_DATE);
  fprintf(stderr, "Usage: big2gb [-hkqV] [-o file ...] [-n infile outfile ...]\n");
  fprintf(stderr, " -h --help        give this help\n");
  fprintf(stderr, " -k --keepdate    keep output file date\n");
  fprintf(stderr, " -p --keepperm    keep output file permision\n");
  fprintf(stderr, " -q --quiet       quiet mode, suppress all warnings\n");
  fprintf(stderr, "                  always on in stdin->stdout mode\n");
  fprintf(stderr, " -V --version     display version number\n");
  fprintf(stderr, " -o --oldfile     write to old file\n");
  fprintf(stderr, " file ...         files to convert in old file mode\n");
  fprintf(stderr, " -n --newfile     write to new file\n");
  fprintf(stderr, " infile           original file in new file mode\n");
  fprintf(stderr, " outfile          output file in new file mode\n");
}


void PrintVersion(void)
{
  fprintf(stderr, "big2gb %s (%s)\n", VER_REVISION, VER_DATE);
#ifdef DEBUG
  fprintf(stderr, "RCS_AUTHOR: %s\n", RCS_AUTHOR);
  fprintf(stderr, "RCS_DATE: %s\n", RCS_DATE);
  fprintf(stderr, "RCS_REVISION: %s\n", RCS_REVISION);
  fprintf(stderr, "VER_AUTHOR: %s\n", VER_AUTHOR);
  fprintf(stderr, "VER_DATE: %s\n", VER_DATE);
  fprintf(stderr, "VER_REVISION: %s\n", VER_REVISION);
#endif DEBUG
}


/* opens file of name ipFN in read only mode
 * RetVal: NULL if failure
 *         file stream otherwise
 */
FILE* OpenInFile(char *ipFN)
{
  return (fopen(ipFN, R_CNTRL));
}


/* opens file of name ipFN in write only mode
 * RetVal: NULL if failure
 *         file stream otherwise
 */
FILE* OpenOutFile(char *ipFN)
{
  return (fopen(ipFN, W_CNTRL));
}


/* converts stream ipInF to GB format text and write to stream ipOutF
 * RetVal: 0  if success
 *         -1  otherwise
 *
 * TempChar: second byte, LastChar: first byte
 *            Big5 second byte: 0x40-7E,A1-FE
 *               (40-7E) - 40 = (0-3E)
 *               (A1-FE) - 62 = (3F-9C)
 *            Big5 first byte: 0xA1-0xF9
 *               F9 - A1 + 1= 59
 *               0x9D * 0x59 Total
 */

#define Big5ToGBTblSize (sizeof(Big5ToGBTbl)/sizeof(int))
#define GBbox   0xA1F5  /* GB code for the empty box symbol */
#define BIGbox  0xA1BC  /* BIG code for the empty box symbol */

int ConvertBig5ToGB(FILE* ipInF, FILE* ipOutF, CFlag *ipFlag)
{
   int RetVal = 0, index;
   int TempChar,LastChar = 0, OutChar;

   while ((TempChar = getc(ipInF)) != EOF)
   {
       if (TempChar == 0x0A)   /* new line */
       {
           if (LastChar != 0) { putc(LastChar, ipOutF); LastChar = 0; }
           /* if (ferror(ipOutF)) break; */
              /* each line check output file error? */
           putc(TempChar, ipOutF);
       }
       else if (LastChar != 0)  /* a chinese char second byte? */
       {
           OutChar = 0;   
           index = (LastChar - 0xA1)*0x9D + TempChar;
           if (TempChar < 0xA1) index -= 0x40; else index -= 0x62;
           if (index >= 0 && index < Big5ToGBTblSize)
               OutChar = Big5ToGBTbl[index];
           
           if (OutChar != 0)
              { putc(HIBYTE(OutChar), ipOutF); putc(LOBYTE(OutChar), ipOutF); }
           else 
              { putc(HIBYTE(GBbox), ipOutF); putc(LOBYTE(GBbox), ipOutF); }
           LastChar = 0;
       }
       else if (TempChar > 0xA0) LastChar=TempChar; else putc(TempChar,ipOutF);
        /* a new single char, is a chinese first byte ? */
    }
     
    if (ferror(ipInF) || ferror(ipOutF))
    {
        RetVal = -1;
        if (!ipFlag->Quiet)
           fprintf(stderr, "big2gb: Error while converting file!\n");
    }
    else  if (LastChar != 0)   putc(LastChar, ipOutF);
    return RetVal;
}

/* convert file ipInFN to GB format text and write to file ipOutFN
 * RetVal: 0 if success
 *         -1 otherwise
 */
int ConvertBig5ToGBNewFile(char *ipInFN, char *ipOutFN, CFlag *ipFlag)
{
  int RetVal = 0;
  FILE *InF = NULL;
  FILE *TempF = NULL;
  char TempPath[16];
  struct stat StatBuf;
  struct utimbuf UTimeBuf;

  /* retrieve ipInFN file date stamp */
  if ((ipFlag->KeepDate || ipFlag->KeepPerm) && stat(ipInFN, &StatBuf))
    RetVal = -1;

  strcpy (TempPath, "./u2dtmp");
  strcat (TempPath, "XXXXXX");
  mktemp (TempPath);

#ifdef DEBUG
  fprintf(stderr, "big2gb: using %s as temp file\n", TempPath);
#endif DEBUG

  /* can open in file? */
  if ((!RetVal) && ((InF=OpenInFile(ipInFN)) == NULL))
    RetVal = -1;

  /* can open out file? */
  if ((!RetVal) && (InF) && ((TempF=OpenOutFile(TempPath)) == NULL))
  {
    fclose (InF);
    RetVal = -1;
  }

  /* conversion sucessful? */
  if ((!RetVal) && (ConvertBig5ToGB(InF, TempF, ipFlag)))
    RetVal = -1;

   /* can close in file? */
  if ((InF) && (fclose(InF) == EOF))
    RetVal = -1;

  /* can close out file? */
  if ((TempF) && (fclose(TempF) == EOF))
    RetVal = -1;

  if ((!RetVal) && (ipFlag->KeepDate))
  {
    UTimeBuf.actime = StatBuf.st_atime;
    UTimeBuf.modtime = StatBuf.st_mtime;
    /* can change out file time to in file time? */
    if (utime(TempPath, &UTimeBuf) == -1)
      RetVal = -1;
  }

 if ((!RetVal) && (ipFlag->KeepPerm))
 {
   if (chmod(TempPath,StatBuf.st_mode) == -1)
      RetVal = -1;
 }

  /* any error? */
  if ((RetVal) && (unlink(TempPath)))
    RetVal = -1;

  /* can rename temp file to out file? */
  if (!RetVal)
  {
    if (stat(ipOutFN, &StatBuf) == 0)
      unlink(ipOutFN);

    if ((rename(TempPath, ipOutFN) == -1) && (!ipFlag->Quiet))
    {
      fprintf(stderr, "big2gb: problems renaming '%s' to '%s'\n", TempPath, ipOutFN);
      fprintf(stderr, "          output file remains in '%s'\n", TempPath);
      RetVal = -1;
    }
  }

  return RetVal;
}


/* convert file ipInFN to GB format text
 * RetVal: 0 if success
 *         -1 otherwise
 */
int ConvertBig5ToGBOldFile(char* ipInFN, CFlag *ipFlag)
{
  int RetVal = 0;
  FILE *InF = NULL;
  FILE *TempF = NULL;
  char TempPath[16];
  struct stat StatBuf;
  struct utimbuf UTimeBuf;

  /* retrieve ipInFN file date stamp */
  if ((ipFlag->KeepDate || ipFlag->KeepPerm) && stat(ipInFN, &StatBuf))
    RetVal = -1;

  strcpy (TempPath, "./u2dtmp");
  strcat (TempPath, "XXXXXX");
  mktemp (TempPath);

#ifdef DEBUG
  fprintf(stderr, "big2gb: using %s as temp file\n", TempPath);
#endif DEBUG    

  /* can open in file? */
  if ((!RetVal) && ((InF=OpenInFile(ipInFN)) == NULL))
    RetVal = -1;

  /* can open out file? */
  if ((!RetVal) && (InF) && ((TempF=OpenOutFile(TempPath)) == NULL))
  {
    fclose (InF);
    RetVal = -1;
  }

  /* conversion sucessful? */
  if ((!RetVal) && (ConvertBig5ToGB(InF, TempF, ipFlag)))
    RetVal = -1;

   /* can close in file? */
  if ((InF) && (fclose(InF) == EOF))
    RetVal = -1;

  /* can close out file? */
  if ((TempF) && (fclose(TempF) == EOF))
    RetVal = -1;

  if ((!RetVal) && (ipFlag->KeepDate))
  {
    UTimeBuf.actime = StatBuf.st_atime;
    UTimeBuf.modtime = StatBuf.st_mtime;
    /* can change out file time to in file time? */
    if (utime(TempPath, &UTimeBuf) == -1)
      RetVal = -1;
  }

 if ((!RetVal) && (ipFlag->KeepPerm))
 {
   if (chmod(TempPath,StatBuf.st_mode) == -1)
      RetVal = -1;
 }

  /* can delete in file? */
  if ((!RetVal) && (unlink(ipInFN) == -1))
    RetVal = -1;

  /* any error? */
  if ((RetVal) && (unlink(TempPath)))
    RetVal = -1;

  /* can rename out file to in file? */
  if ((!RetVal) && (rename(TempPath, ipInFN) == -1))
  {
    if (!ipFlag->Quiet)
    {
      fprintf(stderr, "big2gb: problems renaming '%s' to '%s'\n", TempPath, ipInFN);
      fprintf(stderr, "          output file remains in '%s'\n", TempPath);
    }
    RetVal = -1;
  }
  return RetVal;
}


/* convert stdin to GB format text and write to stdout
 * RetVal: 0 if success
 *         -1 otherwise
 */
int ConvertBig5ToGBStdio(CFlag *ipFlag)
{
    ipFlag->NewFile = 1;
    ipFlag->Quiet = 1;
    ipFlag->KeepDate = 0;
    ipFlag->KeepPerm = 0;
    return (ConvertBig5ToGB(stdin, stdout, ipFlag));
}


int main (int argc, char *argv[])
{
  /* variable declarations */
  int ArgIdx;
  int CanSwitchFileMode;
  int ShouldExit;
  CFlag *pFlag;

  /* variable initialisations */
  ArgIdx = 0;
  CanSwitchFileMode = 1;
  ShouldExit = 0;
  pFlag = (CFlag*)malloc(sizeof(CFlag));  
  pFlag->NewFile = 0;
  pFlag->Quiet = 0;
  pFlag->KeepDate = 0;
  pFlag->KeepPerm = 0;
  
  /* no option, use stdin and stdout */
  if (argc == 1)
  {
    exit(ConvertBig5ToGBStdio(pFlag));
  }

  while ((++ArgIdx < argc) && (!ShouldExit))
  {
    /* is it an option? */
    if (argv[ArgIdx][0] == '-')
    {
      /* an option */
      if ((strcmp(argv[ArgIdx],"-h") == 0) || (strcmp(argv[ArgIdx],"--help") == 0))
        PrintUsage();
      if ((strcmp(argv[ArgIdx],"-k") == 0) || (strcmp(argv[ArgIdx],"--keepdate") == 0))
        pFlag->KeepDate = 1;
      if ((strcmp(argv[ArgIdx],"-p") == 0) || (strcmp(argv[ArgIdx],"--keepperm") == 0))
        pFlag->KeepPerm = 1;
      if ((strcmp(argv[ArgIdx],"-q") == 0) || (strcmp(argv[ArgIdx],"--quiet") == 0))
        pFlag->Quiet = 1;
      if ((strcmp(argv[ArgIdx],"-V") == 0) || (strcmp(argv[ArgIdx],"--version") == 0))
        PrintVersion();

      if ((strcmp(argv[ArgIdx],"-o") == 0) || (strcmp(argv[ArgIdx],"--oldfile") == 0))
      {
        /* last convert not paired */
        if (!CanSwitchFileMode)
        {
          if (!pFlag->Quiet)
            fprintf(stderr, "big2gb: target of file %s not specified in new file mode\n", argv[ArgIdx-1]);
          ShouldExit = 1;
        }
        pFlag->NewFile = 0;
      }
      if ((strcmp(argv[ArgIdx],"-n") == 0) || (strcmp(argv[ArgIdx],"--newfile") == 0))
      {
        /* last convert not paired */
        if (!CanSwitchFileMode)
        {
          if (!pFlag->Quiet)
            fprintf(stderr, "big2gb: target of file %s not specified in new file mode\n", argv[ArgIdx-1]);
          ShouldExit = 1;
        }
        pFlag->NewFile = 1;
      }
    }
    else
    {
      /* not an option */
      if (pFlag->NewFile)
      {
        if (CanSwitchFileMode)
          CanSwitchFileMode = 0;
        else
        {
          if (!pFlag->Quiet)
            fprintf(stderr, "big2gb: converting file %s to file %s in GB format ...\n", argv[ArgIdx-1], argv[ArgIdx]);
          if (ConvertBig5ToGBNewFile(argv[ArgIdx-1], argv[ArgIdx], pFlag))
          {
            if (!pFlag->Quiet)
              fprintf(stderr, "big2gb: problems converting file %s to file %s\n", argv[ArgIdx-1], argv[ArgIdx]);
            ShouldExit = 1;
          }
          CanSwitchFileMode = 1;
        }
      }
      else
      {
        if (!pFlag->Quiet)
          fprintf(stderr, "big2gb: converting file %s to GB format ...\n", argv[ArgIdx]);
        if (ConvertBig5ToGBOldFile(argv[ArgIdx], pFlag))
        {
          if (!pFlag->Quiet)
            fprintf(stderr, "big2gb: problems converting file %s\n", argv[ArgIdx]);
          ShouldExit = 1;
        }
      }
    }
  }

  if ((!pFlag->Quiet) && (!CanSwitchFileMode))
  {
    fprintf(stderr, "big2gb: target of file %s not specified in new file mode\n", argv[ArgIdx-1]);
    ShouldExit = 1;
  }
  free(pFlag);
  return (ShouldExit);
}

