package Language::INTERCAL::Runtime::Library;

# Library functions for CLC-INTERCAL

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# text for numbers in languages other than English and Scottish Gaelic
# shamelessly stolen from C-INTERCAL (src/numerals.c),
# copyright (C) 1996 Eric S. Raymond. The actual code is mine (lunatic),
# it's only the text of the numbers I've stolen.

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use vars qw($VERSION @EXPORT @ISA);
$VERSION = '0.05';

require Exporter;
use Charset::Baudot;
use Charset::EBCDIC;

@EXPORT = qw(atoi roman i_interleave i_select i_and i_or i_xor i_uninterleave
	     i_unselect i_unand i_unor i_unxor i_initialise convert_charset
	     i_band i_bor i_bxor i_unband i_unbor i_unbxor
	     ato16 ato32 output i_read_num i_read_arr_16 i_read_arr_32
	     input i_write_num i_write_arr_16 i_write_arr_32
	     i_read_arr_c i_write_arr_c
	     _arrlst _arrset _assign _subscript _read _write _value
	     _clone_value _clone_hash _clone_array _stash _retrieve
	     _owner _free _enrol _learns _finish _events _run_db);
@ISA = qw(Exporter);

sub atoi ($) {
    my ($s) = @_;
    unpack(length($s) > 2 ? 'N' : 'n', $s) || 0;
}

sub ato32 ($) {
    my ($s) = @_;
    length($s) > 2 ? $s : pack('N', unpack('n', $s) || 0);
}

sub ato16 ($) {
    my ($s) = @_;
    return $s if length($s) < 3;
    my $v = unpack('N', $s);
    die "275 INCOMPATIBLE NUMBER OF SPOTS\n" if $v > 0xffff;
    pack('n', $v);
}

sub i_initialise (&) {
    my ($code) = @_;

    use Getopt::Long;

    my $input_charset = "EBCDIC";
    my $output_charset = "ASCII";
    my $obsolete = 0;

    Getopt::Long::config qw(no_ignore_case auto_abbrev permute bundling);

    GetOptions("obsolete"  => \$obsolete,
	       "a"         => sub { $input_charset = "ASCII" },
	       "ascii"     => sub { $input_charset = "ASCII" },
	       "b"         => sub { $input_charset = "Baudot" },
	       "baudot"    => sub { $input_charset = "Baudot" },
	       "e"         => sub { $input_charset = "EBCDIC" },
	       "ebcdic"    => sub { $input_charset = "EBCDIC" },
	       "c:s"       => sub { $input_charset = $_[1] },
	       "charset:s" => sub { $input_charset = $_[1] },
	       "A"         => sub { $output_charset = "ASCII" },
	       "ASCII"     => sub { $output_charset = "ASCII" },
	       "B"         => sub { $output_charset = "Baudot" },
	       "BAUDOT"    => sub { $output_charset = "Baudot" },
	       "E"         => sub { $output_charset = "EBCDIC" },
	       "EBCDIC"    => sub { $output_charset = "EBCDIC" },
	       "C:s"       => sub { $output_charset = $_[1] },
	       "CHARSET:s" => sub { $output_charset = $_[1] },
	       "o:s"       => sub { close STDOUT; open(STDOUT, "> " . $_[1]) or die "012 $_[1]\: $!\n" },
	       "output:s"  => sub { close STDOUT; open(STDOUT, "> " . $_[1]) or die "012 $_[1]\: $!\n" })
    or die "003 Usage: $0 [-aAbBcCeEo] [input files]\n";

    my $cin = convert_charset($input_charset, "EBCDIC");
    my $cout = convert_charset("ASCII", $output_charset);
    &$code(
	sub {
	    my $t;
	    if (@_) {
		read STDIN, $t, $_[0];
	    } else {
		$t = <STDIN>;
		$t = &$cin($t);
	    }
	    $t;
	},
	sub {
	    $| = 1;
	    print STDOUT &$cout(@_);
	},
	$obsolete ? ("next" => 1) : ()
    );
}

sub convert_charset {
    my ($package, $filename, $line) = caller;
    @_ == 2 or
	die "013 SYNTAX IS convert_charset(FROM, TO) AT $filename LINE $line\n";
    my ($from, $to) = @_;
    return sub { '' } if ! $to;
    return sub { join('', @_) } if $from eq $to;
    my $from2ascii = '';
    if ($from ne 'ASCII') {
	my $module = "Charset::$from";
	$@ = '';
	eval "require $module";
	die "111 Character set '$from' not supported\n"
	    if $@ || ! defined \&{"${module}::\L${from}2ascii"};
	$from2ascii = "\$t = ${module}::\L${from}2ascii(\$t); ";
    }
    my $ascii2to = '';
    if ($to ne 'ASCII') {
	my $module = "Charset::$to";
	$@ = '';
	eval "require $module";
	die "111 Character set '$to' not supported\n"
	    if $@ || ! defined \&{"${module}::ascii2\L$to"};
	$ascii2to = "\$t = ${module}::ascii2\L$to(\$t); ";
    }
    eval "sub { my \$t = join('', \@_); $from2ascii$ascii2to\$t }";
}

sub roman ($$) {
    my ($number, $mode) = @_;
    if ($number == 0) {
	return "NIHIL";
    }
    my $result = '';
    my @digits = (
    	[ 1000000000, '\i', '\v', '\x', '\l', '\c', '\d', '\m' ],
    	[    1000000, '\I', '\V', '\X', '\L', '\C', '\D', '\M' ],
    	[       1000,  'i',  'v',  'x',  'l',  'c',  'd',  'm' ],
    	[          1,  'I',  'V',  'X',  'L',  'C',  'D',  'M' ],
    );
    while ($number && @digits) {
    	my $digits = shift @digits;
	my ($multiply, $i, $v, $x, $l, $c, $d, $m) = @$digits;
	next if $number < 4 * $multiply && @digits;
	my $r = '';
	if ($number >= 1000 * $multiply) {
	    $r .= $m x int($number / 1000 / $multiply);
	    $number = $number % (1000 * $multiply);
	}
	if ($number >= 999 * $multiply) {
	    $r .= $i . $m;
	    $number -= 999 * $multiply;
	}
	if (int($number / $multiply) == 499) {
	    $r .= $i . $d;
	    $number -= 499 * $multiply;
	}
	if ($number >= 100 * $multiply) {
	    $r .= _numeral($c, $d, $m, int($number / 100 / $multiply));
	    $number = $number % (100 * $multiply);
	}
	if ($number >= 99 * $multiply) {
	    $r .= $i . $c;
	    $number -= 99 * $multiply;
	}
	if (int($number / $multiply) == 49) {
	    $r .= $i . $l;
	    $number -= 49 * $multiply;
	}
	if ($number >= 10 * $multiply) {
	    $r .= _numeral($x, $l, $c, int($number / 10 / $multiply));
	    $number = $number % (10 * $multiply);
	}
	if ($number >= 4 * $multiply || $r ne '') {
	    $r .= _numeral($i, $v, $x, int($number / $multiply));
	} else {
	    $r .= $i x int($number / $multiply);
	}
	$number = $number % $multiply;
	$result .= $r;
    }
    $result =~ s.\\._\010.g if $mode;
    $result;
}

sub _numeral ($$$$) {
    my ($i, $v, $x, $value) = @_;
    if ($value >= 9) { return $i . $x };
    if ($value >= 5) { return $v . ($i x ($value - 5)) }
    if ($value >= 4) { return $i . $v };
    return $i x $value;
}

sub i_interleave ($$) {
    my $val1 = atoi($_[0]);
    my $val2 = atoi($_[1]);
    die "533 INTERLEAVE REQUIRES TWO 16 BITS VALUES\n"
	if $val1 >= 0x10000 || $val2 >= 0x10000;
    $val1 = (($val1 & 0x0000ff00) << 8) | ($val1 & 0x000000ff);
    $val1 = (($val1 & 0x00f000f0) << 4) | ($val1 & 0x000f000f);
    $val1 = (($val1 & 0x0c0c0c0c) << 2) | ($val1 & 0x03030303);
    $val1 = (($val1 & 0x22222222) << 2) | (($val1 & 0x11111111) << 1);
    $val2 = (($val2 & 0x0000ff00) << 8) | ($val2 & 0x000000ff);
    $val2 = (($val2 & 0x00f000f0) << 4) | ($val2 & 0x000f000f);
    $val2 = (($val2 & 0x0c0c0c0c) << 2) | ($val2 & 0x03030303);
    pack('N', $val1 | (($val2 & 0x22222222) << 1) | ($val2 & 0x11111111));
}

sub i_select ($$) {
    my $bit = unpack("%32b*", $_[1]);
    my $val1 = atoi($_[0]);
    my $val2 = atoi($_[1]);
    my $res = 0;
    my $mask = 1;
    while ($val2) {
	if ($val2 & $mask) {
	    $res |= $val1 & $mask;
	    $val2 ^= $mask;
	    $mask <<= 1;
	} else {
	    $val1 >>= 1;
	    $val2 >>= 1;
	}
    }
    pack($bit > 16 ? 'N' : 'n', $res);
}

sub i_and ($) {
    my ($val) = @_;
    my $pack = length($val) > 2 ? 'N' : 'n';
    my $num = unpack($pack, $val);
    my $shift = $num >> 1;
    $shift |= $pack eq 'N' ? 0x80000000 : 0x8000 if $num & 1;
    "$val" & pack($pack, $shift);
}

sub i_or ($) {
    my ($val) = @_;
    my $pack = length($val) > 2 ? 'N' : 'n';
    my $num = unpack($pack, $val);
    my $shift = $num >> 1;
    $shift |= $pack eq 'N' ? 0x80000000 : 0x8000 if $num & 1;
    "$val" | pack($pack, $shift);
}

sub i_xor ($) {
    my ($val) = @_;
    my $pack = length($val) > 2 ? 'N' : 'n';
    my $num = unpack($pack, $val);
    my $shift = $num >> 1;
    $shift |= $pack eq 'N' ? 0x80000000 : 0x8000 if $num & 1;
    "$val" ^ pack($pack, $shift);
}

sub i_band ($$) {
    my ($val1, $val2) = @_;
    if (length($val1) != length($val2)) {
	$val1 = pack('N', unpack('n', $val1)) if length($val1) < 3;
	$val2 = pack('N', unpack('n', $val2)) if length($val2) < 3;
    }
    $val1 & $val2;
}

sub i_bor ($$) {
    my ($val1, $val2) = @_;
    if (length($val1) != length($val2)) {
	$val1 = pack('N', unpack('n', $val1)) if length($val1) < 3;
	$val2 = pack('N', unpack('n', $val2)) if length($val2) < 3;
    }
    $val1 | $val2;
}

sub i_bxor ($$) {
    my ($val1, $val2) = @_;
    if (length($val1) != length($val2)) {
	$val1 = pack('N', unpack('n', $val1)) if length($val1) < 3;
	$val2 = pack('N', unpack('n', $val2)) if length($val2) < 3;
    }
    $val1 ^ $val2;
}

sub i_unband ($) {
    my ($val1) = @_;
    ($val1, chr(255) x length($val1));
}

sub i_unbor ($) {
    my ($val1) = @_;
    ($val1, chr(0) x length($val1));
}

sub i_unbxor ($) {
    my ($val1) = @_;
    ($val1, chr(0) x length($val1));
}

sub output ($@) {
    my ($handle, @data) = @_;
    return if ! defined $handle || ! ref $handle;
    if (UNIVERSAL::isa($handle, 'CODE')) {
	&$handle(@data);
    } elsif (UNIVERSAL::isa($handle, 'SCALAR')) {
	$$handle .= join('', @data);
    } elsif (UNIVERSAL::isa($handle, 'ARRAY')) {
	push @$handle, join('', @data);
    } elsif (UNIVERSAL::isa($handle, 'GLOB')) {
	print $handle @data;
    }
}

sub input ($;$) {
    my ($handle, $size) = (@_, 0);
    return '' if ! defined $handle || ! ref $handle;
    my $data = '';
    if (UNIVERSAL::isa($handle, 'CODE')) {
	$data = &$handle($size ? $size : ());
    } elsif (UNIVERSAL::isa($handle, 'ARRAY')) {
	$data = shift @$handle;
	if ($size) { $data = substr($data, 0, $size) }
    } elsif (UNIVERSAL::isa($handle, 'GLOB')) {
	if ($size) {
	    read $handle, $data, $size;
	} else {
	    $data = <$handle>;
	}
    }
    $data;
}

sub i_read_num ($$$) {
    my ($handle, $value, $roman) = @_;
    output($handle, roman($value, $roman), "\n");
}

sub i_read_arr_16 ($$) {
    my ($handle, $values) = @_;
    my $line = join('', map { sprintf("%c", $_) } grep {$_} @$values);
    output($handle, baudot2ascii($line), "\n");
}

sub i_read_arr_32 ($$$) {
    my ($handle, $values, $io) = @_;
    my $line = '';
    my $value;
    for $value (@$values) {
	next if ! $value;
	my $val0 = $value;
	my $bits0 = 0;
	my $bits1 = 0;
	my $i;
	for ($i = 0; $i < 8; $i++) {
	    $bits0 >>= 1;
	    $bits1 >>= 1;
	    $bits0 |= 0x80 if $val0 & 2;
	    $bits1 |= 0x80 if $val0 & 1;
	    $val0 >>= 2;
	}
	$val0 = 0;
	for ($i = 0; $i < 8; $i++) {
	    $val0 >>= 1;
	    if ($io & 1) {
		$val0 |= 0x80 if $bits0 & 1;
		$bits0 >>= 1;
	    } else {
		$val0 |= 0x80 if ! ($bits1 & 1);
		$bits1 >>= 1;
	    }
	    $io >>= 1;
	}
	$line .= chr($val0);
	$io = $val0;
    }
    output($handle, $line);
}

sub i_read_arr_c ($$$) {
    my ($handle, $values, $io) = @_;
    my $line = '';
    my $value;
    for $value (@$values) {
	$$io -= $value;
	my $val = $$io;
	$val = (($val & 0x0f) << 4) | (($val & 0xf0) >> 4);
	$val = (($val & 0x33) << 2) | (($val & 0xcc) >> 2);
	$val = (($val & 0x55) << 1) | (($val & 0xaa) >> 1);
	$line .= chr($val);
    }
    output($handle, $line);
}

sub i_write_num ($;$) {
    my ($handle, $code) = @_;
    my $line = input($handle, 0);
    $line = &$code($line) if defined $code;
    _convert_number($line);
}

# text for numbers in languages other than English and Scottish Gaelic
# shamelessly stolen from C-INTERCAL (src/numerals.c),
# copyright (C) 1996 Eric S. Raymond. The actual code is mine (lunatic),
# it's only the text of the numbers I've stolen.

my %numbers = (
# English
    'OH'          => 0,
    'ZERO'        => 0,
    'ONE'         => 1,
    'TWO'         => 2,
    'THREE'       => 3,
    'FOUR'        => 4,
    'FIVE'        => 5,
    'SIX'         => 6,
    'SEVEN'       => 7,
    'EIGHT'       => 8,
    'NINE'        => 9,
    'NINER'       => 9,
# Scottish Gaelic
    # Write to the Lunatic if you wonder how these are pronounced, or why
    # there are so many different forms.
    'NONI'        => 0,
    'AON'         => 1,
    'A H-AON'     => 1,
    'AONAR'       => 1,
    'DA'          => 2,
    'DHA'         => 2,
    'A DHA'       => 2,
    'DITHIS'      => 2,
    'TRI'         => 3,
    'A TRI'       => 3,
    'TRIUIR'      => 3,
    'CEITHIR'     => 4,
    'A CEITHIR'   => 4,
    'CEATHRAR'    => 4,
    'COIG'        => 5,
    'A COIG'      => 5,
    'C\`OIG'      => 5,
    'A C\`OIG'    => 5,
    'CIG'        => 5,
    'A CIG'      => 5,
    'COIGNEAR'    => 5,
    'C\`OIGNEAR'  => 5,
    'CIGNEAR'    => 5,
    'SIA'         => 6,
    'SE'          => 6,
    'A SIA'       => 6,
    'A SE'        => 6,
    'SEANAR'      => 6,
    'SEACHD'      => 7,
    'A SEACHD'    => 7,
    'SEACHDNAR'   => 7,
    'OCHD'        => 8,
    'A H-OCHD'    => 8,
    'OCHDNAR'     => 8,
    'NAOI'        => 9,
    'A NAOI'      => 9,
    'NAONAR'      => 9,
# Sanskrit
    'SUTYA'       => 0,
    'SHUTYA'      => 0,
    'EKA'         => 1,
    'DVI'         => 2,
    'TRI'         => 3,
    'CHATUR'      => 4,
    'PANCHAN'     => 5,
    'SHASH'       => 6,
    'SAPTAM'      => 7,
    'ASHTAN'      => 8,
    'NAVAN'       => 9,
# Basque
    'ZEROA'       => 0,
    'BAT'         => 1,
    'BI'          => 2,
    'HIRO'        => 3,
    'LAU'         => 4,
    'BORTZ'       => 5,
    'SEI'         => 6,
    'ZAZPI'       => 7,
    'ZORTZI'      => 8,
    'BEDERATZI'   => 9,
# Tagalog
    'WALA'        => 0,
    'ISA'         => 1,
    'DALAWA'      => 2,
    'TATLO'       => 3,
    'APAT'        => 4,
    'LIMA'        => 5,
    'ANIM'        => 6,
    'PITO'        => 7,
    'WALO'        => 8,
    'SIYAM'       => 9,
# Classical Nahuatl
    'AHTLE'       => 0,
    'CE'          => 1,
    'OME'         => 2,
    'IEI'         => 3,
    'NAUI'        => 4,
    'NACUILI'     => 5,
    'CHIQUACE'    => 6,
    'CHICOME'     => 7,
    'CHICUE'      => 8,
    'CHICUNAUI'   => 9,
# Georgian
    'NULI'        => 0,
    'ERTI'        => 1,
    'ORI'         => 2,
    'SAMI'        => 3,
    'OTXI'        => 4,
    'XUTI'        => 5,
    'EKSVI'       => 6,
    'SHVIDI'      => 7,
    'RVA'         => 8,
    'CXRA'        => 9,
# Kwakiutl (technically, Kwak'wala)
    "KE'YOS"      => 0,
    "'NEM"        => 1,
    "MAL'H"       => 2,
    "YUDEXW"      => 3,
    "MU"          => 4,
    "SEK'A"       => 5,
    "Q'ETL'A"     => 6,
    "ETLEBU"      => 7,
    "MALHGWENALH" => 8,
    "'NA'NE'MA"   => 9,
# Volap\"uk
    'NOS'         => 0,
    'BAL'         => 1,
    'TEL'         => 2,
    'KIL'         => 3,
    'FOL'         => 4,
    'LUL'         => 5,
    'M\\"AL'      => 6,
    'ML'         => 6,
    'VEL'         => 7,
    'J\\"OL'      => 8,
    'JL'         => 8,
    'Z\\"UL'      => 9,
    'ZL'         => 9,
);

my $regex = '\\s*('
	  . join('|', map {quotemeta($_)}
			  sort {length($b) <=> length($a)}
			       keys %numbers)
	  . ')';
$regex =~ s/ /\\s*/g;

sub _convert_number ($) {
    my ($line) = @_;
    my $val = 0;
    while ($line =~ s/^$regex//io) {
	$val *= 10;
	$val += $numbers{uc($1)};
    }
    $val;
}

sub i_write_arr_16 ($$;$) {
    my ($handle, $values, $code) = @_;
    my $line = input($handle, 0);
    chomp $line;
    $line = &$code($line) if defined $code;
    $line = ascii2baudot($line);
    die "997 INPUT RECORD TOO LONG FOR ARRAY\n" if length($line) > @$values;
    my $i = 0;
    while ($i < length($line)) {
	$values->[$i] = ord(substr($line, $i, 1));
	$i++;
    }
    while ($i < @$values) {
	$values->[$i] = 0;
	$i++;
    }
}

sub i_write_arr_32 ($$$) {
    my ($handle, $values, $io) = @_;
    my $line = input($handle, scalar(@$values));
    my $ptr = 0;
    my @val = ();
    while ($line ne '') {
	my $chr = ord($line);
	my $chr0 = $chr;
	$line = substr($line, 1);
	my $bits0 = 0;
	my $bits1 = 0;
	my $i;
	for ($i = 0; $i < 8; $i++) {
	    if ($io & 0x80) {
		$bits0 <<= 1;
		$bits0 |= 1 if $chr & 0x80;
	    } else {
		$bits1 <<= 1;
		$bits1 |= 1 if ! ($chr & 0x80);
	    }
	    $chr <<= 1;
	    $io <<= 1;
	}
	$chr = int(rand 0xffff) + 1;
	for ($i = 0; $i < 8; $i++) {
	    $chr <<= 2;
	    $chr |= 2 if $bits0 & 0x80;
	    $chr |= 1 if $bits1 & 0x80;
	    $bits0 <<= 1;
	    $bits1 <<= 1;
	}
	$values->[$ptr] = $chr;
	++$ptr;
	$io = $chr0;
    }
}

sub i_write_arr_c ($$$) {
    my ($handle, $values, $io) = @_;
    my $line = substr(input($handle, scalar(@$values)), 0, scalar(@$values));
    for (my $ptr = 0; $ptr < length($line); $ptr++) {
	my $c = ord(substr($line, $ptr, 1));
	$values->[$ptr] = ($c - $$io) & 0xff;
	$$io = $c;
    }
    if (length($line) < @$values) {
	$values->[length($line)] = 256;
    }
}

sub i_uninterleave ($) {
    my ($n) = atoi(shift @_);
    my $n1 = 0;
    my $n2 = 0;
    my $b;
    for ($b = 0; $b < 16; $b++) {
	$n1 >>= 1;
	$n2 >>= 1;
	$n1 |= 0x8000 if $n & 2;
	$n2 |= 0x8000 if $n & 1;
	$n >>= 2;
    }
    (pack('n', $n1), pack('n', $n2));
}

sub i_unselect ($) {
    my ($n) = shift @_;
    ($n, chr(255) x length($n));
}

sub i_unand ($) {
    my ($val) = shift @_;
    my $pack = length($val) > 2 ? 'N' : 'n';
    my $num = unpack($pack, $val);
    my $shift = $num << 1;
    $shift |= 1 if $num & ($pack eq 'N' ? 0x80000000 : 0x8000);
    my $res = "$val" | pack($pack, $shift);
    die "275 INCOMPATIBLE ASSIGNMENT (CANNOT FIND X SUCH THAT &X IS $num)\n"
	if i_and($res) ne $val;
    $res;
}

sub i_unor ($) {
    my ($val) = shift @_;
    my $pack = length($val) > 2 ? 'N' : 'n';
    my $num = unpack($pack, $val);
    my $shift = $num << 1;
    $shift |= 1 if $num & ($pack eq 'N' ? 0x80000000 : 0x8000);
    my $res = "$val" & pack($pack, $shift);
    die "275 INCOMPATIBLE ASSIGNMENT (CANNOT FIND X SUCH THAT VX IS $num)\n"
	if i_or($res) ne $val;
    $res;
}

sub i_unxor ($) {
    my ($val) = shift @_;
    my $bits = 8 * length($val);
    my $num = atoi($val);
    my $res = 0;
    my $carry = 0;
    my $mask = 1;
    while ($bits-- > 0) {
	if ($carry) {
	    $res |= $mask;
	}
	$carry = $carry != ($num & 1);
	$mask <<= 1;
	$num >>= 1;
    }
    $res = pack(length($val) > 2 ? 'N' : 'n', $res);
    die "275 INCOMPATIBLE ASSIGNMENT (CANNOT FIND X SUCH THAT V\010-X IS $num)\n"
	if i_xor($res) ne $val;
    $res;
}

sub _assign ($$@) {
    my ($regs, $regname, @data) = @_;
    if (exists $regs->{$regname} && $regs->{$regname}[4]) {
	# overloaded
	my $ooverload = $regs->{$regname}[4];
	$regs->{$regname}[4] = 0;
	if (exists $regs->{'@0'}) {
	    unshift @{$regs->{'@0'}[2]}, $regname;
	} else {
	    $regs->{'@0'} = [0, 1, [$regname], 0, 0, [], []];
	}
	eval { &{$ooverload->[1]}(@data); };
	$regs->{$regname}[4] = $ooverload;
	shift @{$regs->{'@0'}[2]};
	die $@ if $@;
	return;
    }
    die "241 CANNOT ASSIGN TO CLASS $regname\n" if $regname !~ /^[\.,:;]/;
    if ($regname =~ /^[,;]/) {
	return if exists $regs->{$regname} && ! $regs->{$regname}[1];
	my @subs = map { atoi($_) } @data;
	map { $_ || die "241 ILLEGAL SUBSCRIPT\n" } @subs;
	my $s = $regname;
	$s =~ s/^,/./;
	$s =~ s/^;/:/;
	$s .= ' ';
	for my $rp (keys %$regs) {
	    delete $regs->{$rp} if substr($rp, 0, length($s)) eq $s;
	}
	if (exists $regs->{$regname}) {
	    $regs->{$regname}[0] = [@subs];
	} else {
	    $regs->{$regname} = [[@subs], 1, [], 0, 0, [], []];
	}
    } else {
	die "241 $regname IS NOT AN ARRAY REGISTER\n" if @data > 1;
	my $datum = shift @data;
	$datum = $regname =~ /^:/ ? ato32($datum) : ato16($datum);
	my $rp = $regname;
	if ($rp =~ /\s/) {
	    $rp = $`;
	    $rp =~ s/^\./,/;
	    $rp =~ s/^:/;/;
	}
	return if exists $regs->{$rp} && ! $regs->{$rp}[1];
	if (exists $regs->{$regname}) {
	    $regs->{$regname}[0] = $datum;
	} else {
	    $regs->{$regname} = [$datum, 1, [], 0, 0, [], []];
	}
    }
}

sub _subscript ($$@) {
    my ($regs, $regname, @subs) = @_;
    die "241 $regname IS NOT AN ARRAY REGISTER\n" if $regname !~ /^[,;]/;
    die "241 ARRAY $regname NOT DIMENSIONED\n"
	if ! exists $regs->{$regname} || ! $regs->{$regname}[0];
    my @dim = @{$regs->{$regname}[0]};
    my $s = $regname;
    $s =~ s/^,/./;
    $s =~ s/^;/:/;
    die "241 WRONG NUMBER OF SUBSCRIPTS TO $regname\n" if @dim != @subs;
    while (@subs) {
	my $d = shift @subs;
	my $m = shift @dim;
	$s .= ' ' . $d;
	die "241 ILLEGAL SUBSCRIPT\n" if $d < 1;
	die "241 SUBSCRIPT TOO LARGE\n" if $d > $m;
    }
    $s;
}

sub _read ($$$$$@) {
    my ($regs, $output, $roman, $io, $arrayio, @regname) = @_;
    while (@regname) {
	my $regname = shift @regname;
	my $value = 0;
	if (exists $regs->{$regname} && $regs->{$regname}[4]) {
	    # overloaded
	    $value = _value($regs, $regname, 1);
	    i_read_num($output, $value, $roman);
	    return;
	}
	die "241 CANNOT READ CLASS $regname\n" if $regname !~ /^[\.,:;]/;
	if ($regname =~ /^[\.:]/) {
	    i_read_num($output,
		       exists $regs->{$regname} ? atoi($regs->{$regname}[0]) : 0,
		       $roman);
	} else {
	    die "241 ARRAY $regname NOT DIMENSIONED\n" if ! exists $regs->{$regname};
	    my @s = @{$regs->{$regname}[0]};
	    my $r = $regname;
	    $r =~ s/^,/./;
	    $r =~ s/^;/:/;
	    my @rv = _arrlst($regs, $r, @s);
	    if ($arrayio eq 'C') {
		i_read_arr_c($output, \@rv, $io);
	    } else {
		i_read_arr_16($output, \@rv) if $regname =~ /^,/;
		i_read_arr_32($output, \@rv, $$io) if $regname =~ /^;/;
	    }
	}
    }
}

sub _write ($$$$@) {
    my ($regs, $input, $io, $arrayio, @regname) = @_;
    while (@regname) {
	my $regname = shift @regname;
	if (exists $regs->{$regname} && $regs->{$regname}[4]) {
	    # overloaded
	    _write($regs, '.0', $input, $io);
	    _assign($regs, $regname, $regs->{'.0'}[0]);
	    return;
	}
	die "241 CANNOT WRITE CLASS $regname\n" if $regname !~ /^[\.,:;]/;
	if ($regname =~ /^[\.:]/) {
	    my $v = pack($regname =~ /^:/ ? 'N' : 'n',
			 i_write_num($input, \&ebcdic2ascii));
	    my $rp = $regname;
	    if ($rp =~ /^(.\d+)\s/) {
		$rp = $1;
		$rp =~ s/^\./,/;
		$rp =~ s/^:/;/;
	    }
	    if (exists $regs->{$regname}) {
		$regs->{$regname}[0] = $v if $regs->{$rp}[1];
	    } elsif (exists $regs->{$rp}) {
		$regs->{$regname} = [$v, 1, [], 0, 0, [], []]
		    if $regs->{$rp}[1];
	    } else {
		$regs->{$regname} = [$v, 1, [], 0, 0, [], []];
	    }
	} else {
	    die "241 ARRAY $regname NOT DIMENSIONED\n"
		if ! exists $regs->{$regname};
	    my @s = @{$regs->{$regname}[0]};
	    my $r = $regname;
	    $r =~ s/^,/./;
	    $r =~ s/^;/:/;
	    my @rv = _arrlst($regs, $r, @s);
	    if ($arrayio eq 'C') {
		i_write_arr_c($input, \@rv, $io);
	    } else {
		i_write_arr_16($input, \@rv, \&ebcdic2ascii) if $regname =~ /^,/;
		i_write_arr_32($input, \@rv, $$io) if $regname =~ /^;/;
	    }
	    my $ptr = 0;
	    _arrset($regs, $regname =~ /^;/ ? 'N' : 'n', $r, \@rv, \$ptr, @s)
		if $regs->{$regname}[1];
	}
    }
}

sub _arrlst ($$@) {
    my ($k, $v, @s) = @_;
    if (@s) {
	my $mx = shift @s;
	map {_arrlst($k, "$v $_", @s)} (1..$mx);
    } else {
	exists $k->{$v} ? atoi($k->{$v}[0]) : 0;
    }
}

sub _arrset ($$$$$@) {
    my ($k, $p, $v, $rv, $rp, @s) = @_;
    if (@s) {
	my $mx = shift @s;
	my $n = 1;
	while ($mx-- > 0) { _arrset($k, $p, "$v $n", $rv, $rp, @s); $n++; }
    } else {
	$k->{$v}[0] = pack($p, $rv->[$$rp++]);
    }
}

sub _value ($$$) {
    my ($regs, $regname, $int) = @_;
    if (exists $regs->{$regname} && $regs->{$regname}[4]) {
	# overloaded, look it up avoiding loops
	my $ooverload = $regs->{$regname}[4];
	$regs->{$regname}[4] = 0;
	if (exists $regs->{'@0'}) {
	    unshift @{$regs->{'@0'}[2]}, $regname;
	} else {
	    $regs->{'@0'} = [0, 1, [$regname], 0, 0, [], []];
	}
	my $value;
	eval { $value = &{$ooverload->[0]}(); };
	$regs->{$regname}[4] = $ooverload;
	shift @{$regs->{'@0'}[2]};
	die $@ if $@;
	return $int ? atoi($value) : $value;
    }
    die "241 $regname is not a value register\n" if $regname !~ /^[\.:]/;
    if (! exists $regs->{$regname}) {
	return 0 if $int;
	return pack($regname =~ /^:/ ? 'N' : 'n', 0);
    }
    return atoi($regs->{$regname}[0]) if $int;
    return $regs->{$regname}[0];
}

sub _clone_value {
    my ($v) = @_;
    return $v if ! ref $v;
    return _clone_array($v) if ref $v eq 'ARRAY';
    return _clone_hash($v) if ref $v eq 'HASH';
    return $v if ref $v eq 'CODE';
    die "774 GENETIC ERROR\n";
}

sub _clone_array {
    my ($v) = @_;
    scalar [map {_clone_value($_)} @$v];
}

sub _clone_hash {
    my ($v) = @_;
    scalar {map {($_ => _clone_value($v->{$_}))} keys %$v};
}

sub _stash ($$) {
    my ($regs, $regname) = @_;
    if (exists $regs->{$regname}) {
	my $r = $regs->{$regname};
	my %values = ();
	my $arrname = '';
	if ($regname =~ /^[,;]/) {
	    my $s = $regname;
	    $s =~ s/^,/./;
	    $s =~ s/^;/:/;
	    $s .= ' ';
	    for my $k (keys %$regs) {
		next if substr($k, 0, length($s)) ne $s;
		my $v = $regs->{$k};
		$values{$k} = [_clone_value($v->[0]),
			       _clone_value($v->[4])];
	    }
	    $arrname = $s;
	}
	unshift @{$r->[5]}, [_clone_value($r->[0]),
			     $r->[1],
			     _clone_array($r->[2]),
			     _clone_value($r->[4]),
			     _clone_array($r->[6]),
			     \%values,
			     $arrname];
    } else {
	$regs->{$regname} = [0, 1, [], 0, 0, [0], []];
    }
}

sub _retrieve ($$) {
    my ($regs, $regname) = @_;
    if (exists $regs->{$regname}) {
	my $r = $regs->{$regname};
	die "436 $regname IS HIDDEN TOO WELL\n" if ! @{$r->[5]};
	my $p = shift @{$r->[5]};
	return if ! $regs->{$regname}[1];
	if ($p) {
	    my $v;
	    my $a;
	    ($r->[0], $r->[1], $r->[2], $r->[4], $r->[6], $v, $a) = @$p;
	    for my $k (keys %$v) {
		($regs->{$k}[0], $regs->{$k}[4]) = @{$v->{$k}};
	    }
	    if ($a ne '') {
		for my $k (keys %$regs) {
		    delete $regs->{$k} if substr($k, 0, length($a)) eq $a
				       && ! exists $v->{$k};
		}
	    }
	} else {
	    delete $regs->{$regname};
	}
    } else {
	die "436 $regname IS HIDDEN TOO WELL\n";
    }
}

sub _owner ($$@) {
    my ($regs, $regname, @path) = @_;
    while (@path) {
	my $p = shift @path;
	die "512 $regname IS A FREE REGISTER\n"
	    if ! exists $regs->{$regname} || ! @{$regs->{$regname}[2]};
	die "512 $regname DOES NOT HAVE THAT MANY OWNERS\n"
	    if $p > @{$regs->{$regname}[2]};
	$regname = $regs->{$regname}[2][$p - 1];
    }
    $regname;
}

sub _free ($$$) {
    my ($regs, $slave, $master) = @_;
    die "512 $slave IS A FREE REGISTER\n"
	if ! exists $regs->{$slave} || ! @{$regs->{$slave}[2]};
    my $i = 0;
    my $s = $regs->{$slave}[2];
    while ($i < @$s) {
	if ($s->[$i] eq $master) {
	    splice @$s, $i, 1;
	    return;
	}
	$i++;
    }
    die "512 $slave DOES NOT BELONG TO $master\n";
}

sub _enrol ($@) {
    my ($lectures, @subj) = @_;
    my @classes = ();
    FOO: for my $class (keys %$lectures) {
	my $subj;
	for $subj (@subj) {
	    next FOO if ! exists $lectures->{$class}{$subj};
	}
	push @classes, $class;
    }
    die "799 SORRY, THIS MUST BE A HOLIDAY\n" if ! @classes;
    die "603 CLASS WAR BETWEEN $classes[0] and $classes[1]\n" if @classes > 1;
    $classes[0];
}

sub _learns ($$$$$$) {
    my ($regs, $regname, $lectures, $subject, $stack, $return) = @_;
    die "822 $regname IS NOT A STUDENT\n"
	if ! exists $regs->{$regname} || ! @{$regs->{$regname}[6]};
    for my $class (@{$regs->{$regname}[6]}) {
	next if ! exists $lectures->{$class}{$subject};
	my $belong = _clone_array($regs->{$class}[2]);
	unshift @{$regs->{$class}[2]}, $regname;
	push @$stack, [$return, $class, $belong];
	goto $lectures->{$class}{$subject};
    }
    die "823 #$subject NOT IN $regname\'S CURRICULUM\n";
}

sub _finish ($$) {
    my ($regs, $stack) = @_;
    die "801 NOT IN A LECTURE\n" if ! @$stack;
    my $v = pop @$stack;
    my ($return, $class, $belong) = @$v;
    if (exists $regs->{$class}) {
	$regs->{$class}[2] = $belong;
    } else {
	$regs->{$class} = [0, 1, $belong, 0, 0, [], []];
    }
    goto $return;
}

sub _events ($$$) {
    my ($return, $label, $events) = @_;
    my $i = 0;
    while ($i < @$events) {
	my ($cond, $body) = @{$events->[$i]};
	$i ++;
	$@ = '';
	eval { &$cond() };
	next if $@;
	splice(@$events, $i - 1, 1);
	push @$return, $label;
	goto $body;
    }
    $@ = '';
}

sub _run_db ($) {
    my $obj = shift;
    1 while eval { ! &{$obj->{'step'}} };
    if ($@ eq "\n") {
	$@ = '';
    }
}

1;

__END__

=head1 NAME

Language::INTERCAL::Runtime::Library - Runtime library for CLC-INTERCAL

=head1 SYNOPSIS

    use Language::INTERCAL::Runtime::Library;

    sub program {
        ...
    }

    program();

=head1 DESCRIPTION

I<Language::INTERCAL::Runtime::Library> provides the runtime library for
CLC-INTERCAL's default back end (I<Perl>), as well as the other perl back end
(I<PerlText>). You should never need to access this package directly, as the
compiler does that automatically.

=head1 COPYRIGHT

This module is part of CLC-INTERCAL.

Copyright (c) 1999 by Claudio Calvelli E<lt>C<lunatic@assurdo.com>E<gt>,
all (f)rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

=head1 SEE ALSO

A qualified psychiatrist.

