/*
 * Time-stamp: <99/02/19 08:49:23 panic>
 * Author:	The C-Mix Project <cmix@diku.dk>
 *         	Peter Holst Andersen <txix@diku.dk>
 *              Arne John Glenstrup  <panic@diku.dk>
 * Contents:	Functions for creating the scene: create_*
 */

#include "ray.h"
#include <math.h>
#include <stdio.h>

#ifndef __CMIX
#  define __CMIX(X)
#endif

#define NEXT(OBJ) (OBJ.last + 1 >= OBJ.max ? \
  (fprintf(stderr, \
	   "error: too many %s in the scene, please increase %s in 'ray.h'.\n", \
	   OBJ.name, OBJ.maxname), \
   exit(-1), 0) : ++OBJ.last)

static void
compute_surface_normal(vectorType* n, vectorType* v1, vectorType* v2) {
  /* n is the normal to the surface, v1 is given and is the orientation
   * of the surface (for checked discs: the pattern orientation).
   * v2 is computed as a vector normal to both n and v1
   */
  myfloat a, b, c;
  a = n->x * v1->y - n->y * v1->x; if (isnan(a)) a = 0.0;
  if (fzero(a))
    v2->z = 0.0;
  else {
    b = (n->z * v1->x - n->x * v1->z) / a; if (isnan(b)) b = 0.0;
    c = (n->y * v1->z - n->z * v1->y) / a; if (isnan(c)) c = 0.0;
    v2->z = 1 / sqrt(b * b + c * c + 1.0);
  };
  a = n->y * v1->z - n->z * v1->y; if (isnan(a)) a = 0.0;
  if (fzero(a))
    v2->x = 0.0;
  else {
    b = (n->x * v1->y - n->y * v1->x) / a; if (isnan(b)) b = 0.0;
    c = (n->z * v1->x - n->x * v1->z) / a; if (isnan(c)) c = 0.0;
    v2->x = 1 / sqrt(b * b + c * c + 1.0);
    if (!fzero(v2->z) && !fzero(b))
      v2->x = copysign(v2->x, copysign(1.0, v2->z) * copysign(1.0, b));
  };
  a = n->z * v1->x - n->x * v1->z; if (isnan(a)) a = 0.0;
  if (fzero(a))
    v2->y = 0.0;
  else {
    b = (n->y * v1->z - n->z * v1->y) / a; if (isnan(b)) b = 0.0;
    c = (n->x * v1->y - n->y * v1->x) / a; if (isnan(c)) c = 0.0;
    v2->y = 1 / sqrt(b * b + c * c + 1.0);
    if (!fzero(v2->x) && !fzero(b))
      v2->y = copysign(v2->y, copysign(1.0, v2->x) * copysign(1.0, b));
    else if (!fzero(v2->z) && !fzero(c))
      v2->y = copysign(v2->y, copysign(1.0, v2->z) * copysign(1.0, c));
  };
#ifdef DEBUG
  __CMIX(pure)fprintf
    (stderr, "/* Created surface: normal = (%g, %g, %g) */\n",
     n->x, n->y, n->z);
  __CMIX(pure)fprintf
    (stderr, "/*                  x-unit = (%g, %g, %g) */\n",
     v1->x, v1->y, v1->z);
  __CMIX(pure)fprintf
    (stderr, "/*                  y-unit = (%g, %g, %g) */\n",
     v2->x, v2->y, v2->z);
#endif
  __CMIX(pure)fprintf
    (stderr, "/* Check: n*x = %g, n*y = %g, x*y = %g */\n",
     n->x *v1->x + n->y *v1->y + n->z *v1->z,
     n->x *v2->x + n->y *v2->y + n->z *v2->z,
     v1->x*v2->x + v1->y*v2->y + v1->z*v2->z);
}

surfaceT create_simple_surface(myfloat am, myfloat di,
			       myfloat re, myfloat tr, myfloat ref, int specpow)
{
  int i;
  i = NEXT(surfaceIdx);
  surface[i].tag = SIMPLE;
  surface[i].u.sinfo.ambient.red = am;
  surface[i].u.sinfo.ambient.green = am;
  surface[i].u.sinfo.ambient.blue = am;
  surface[i].u.sinfo.diffuse.red = di;
  surface[i].u.sinfo.diffuse.green = di;
  surface[i].u.sinfo.diffuse.blue = di;
  surface[i].u.sinfo.reflectivity = re;
  surface[i].u.sinfo.transparency = tr;
  surface[i].u.sinfo.refrindex = ref;
  surface[i].u.sinfo.specpow = specpow;
  return i;
}

surfaceT create_color_surface(myfloat am, myfloat di,
			      myfloat re, myfloat tr, myfloat ref, int specpow,
			      myfloat red, myfloat green, myfloat blue)
{
  int i;
  i = NEXT(surfaceIdx);
  surface[i].tag = SIMPLE;
  surface[i].u.sinfo.ambient.red = am * red;
  surface[i].u.sinfo.ambient.green = am * green;
  surface[i].u.sinfo.ambient.blue = am * blue;
  surface[i].u.sinfo.diffuse.red = di * red;
  surface[i].u.sinfo.diffuse.green = di * green;
  surface[i].u.sinfo.diffuse.blue = di * blue;
  surface[i].u.sinfo.reflectivity = re;
  surface[i].u.sinfo.transparency = tr;
  surface[i].u.sinfo.refrindex = ref;
  surface[i].u.sinfo.specpow = specpow;
  return i;
}

surfaceT create_special_color_surface(myfloat amred, myfloat amgreen, myfloat amblue,
				      myfloat dired, myfloat digreen, myfloat diblue, 
				      myfloat re, myfloat tr,
				      myfloat ref, int specpow)
{
  int i;
  i = NEXT(surfaceIdx);
  surface[i].tag = SIMPLE;
  surface[i].u.sinfo.ambient.red = amred;
  surface[i].u.sinfo.ambient.green = amgreen;
  surface[i].u.sinfo.ambient.blue = amblue;
  surface[i].u.sinfo.diffuse.red = dired;
  surface[i].u.sinfo.diffuse.green = digreen;
  surface[i].u.sinfo.diffuse.blue = diblue;
  surface[i].u.sinfo.reflectivity = re;
  surface[i].u.sinfo.transparency = tr;
  surface[i].u.sinfo.refrindex = ref;
  surface[i].u.sinfo.specpow = specpow;
  return i;
}

surfaceT create_checked_surface(surfaceT s1, surfaceT s2, myfloat checksize)
{
  int i;
  i = NEXT(surfaceIdx);
  surface[i].tag = CHECKED;
  surface[i].u.checked.s1 = s1;
  surface[i].u.checked.s2 = s2;
  surface[i].u.checked.checksize = checksize;
  surface[i].u.checked.dbl_checksize = checksize * 2.0;
  return i;
}

objectT create_disc(myfloat cx, myfloat cy, myfloat cz,
			myfloat nx, myfloat ny, myfloat nz,
			myfloat vx, myfloat vy, myfloat vz,
			myfloat r, int s)
{
  objectType *obj;
  vectorType n, v, v2;
  obj = scene + NEXT(objectIdx); (obj + 1)->tag = NONE;
  obj->tag = DISC;
  obj->prop.surface = s;	
  obj->obj.disc.c.x = cx;
  obj->obj.disc.c.y = cy;
  obj->obj.disc.c.z = cz;
  obj->obj.disc.n.x = nx;
  obj->obj.disc.n.y = ny;
  obj->obj.disc.n.z = nz;
  obj->obj.disc.v1.x = vx;
  obj->obj.disc.v1.y = vy;
  obj->obj.disc.v1.z = vz;
  obj->obj.disc.r = r;
  obj->obj.disc.r2 = r*r;
  vector_norm_S(&obj->obj.disc.n);
  vector_norm_S(&obj->obj.disc.v1);
  compute_surface_normal(&obj->obj.disc.n, &obj->obj.disc.v1,
			 &obj->obj.disc.v2);
  obj->obj.disc.d = obj->obj.disc.n.x*cx + obj->obj.disc.n.y*cy + 
    obj->obj.disc.n.z*cz;
  return obj;
}

objectT create_square(myfloat cx, myfloat cy, myfloat cz,
		   myfloat nx, myfloat ny, myfloat nz,
		   myfloat vx, myfloat vy, myfloat vz,
		   myfloat r, int s)
{
  objectType *obj;
  vectorType n, v, v2;
  obj = scene + NEXT(objectIdx); (obj + 1)->tag = NONE;
  obj->tag = SQUARE;
  obj->prop.surface = s;
  obj->obj.square.c.x = cx;
  obj->obj.square.c.y = cy;
  obj->obj.square.c.z = cz;
  obj->obj.square.n.x = nx;
  obj->obj.square.n.y = ny;
  obj->obj.square.n.z = nz;
  obj->obj.square.v1.x = vx;
  obj->obj.square.v1.y = vy;
  obj->obj.square.v1.z = vz;
  vector_norm_S(&obj->obj.square.n);
  vector_norm_S(&obj->obj.square.v1);
  compute_surface_normal(&obj->obj.square.n, &obj->obj.square.v1,
			 &obj->obj.square.v2);
  obj->obj.square.r2 = r*r;
  obj->obj.square.d = obj->obj.square.n.x*cx + obj->obj.square.n.y*cy +
    obj->obj.square.n.z*cz;
  return obj;
}

objectT create_sphere(myfloat x, myfloat y, myfloat z,
			  myfloat nx, myfloat ny, myfloat nz,
			  myfloat r, int s)
{
  objectType *obj;
  vectorType n, v, v2;
  myfloat l;
  obj = scene + NEXT(objectIdx); (obj + 1)->tag = NONE;
  obj->tag = SPHERE;
  obj->prop.surface = s;
  obj->obj.sphere.r = r;
  obj->obj.sphere.r2 = r*r;
  obj->obj.sphere.c.x = x;
  obj->obj.sphere.c.y = y;
  obj->obj.sphere.c.z = z;
  n.x = nx; n.y = ny; n.z = nz;
  vector_norm_S(&n);
  if (fzero(n.z)) {
    v.x = 0.0; v.y = 0.0; v.z = 1.0;
    v2.x = -n.y; v2.y = n.x; v2.z = 0.0;
  } else {
    if (fzero(n.x) && fzero(n.y)) {
      v.x = 1.0; v.y = 0.0;
    } else {
      v.x = -n.y; v.y = n.x;
    }
    v.z = 0.0;
    vector_norm_S(&v);
    l = sqrt(n.x * n.x + n.y * n.y);
    if (fzero(l)) {
      v2.x = -v.y; v2.y = v.x; v2.z = 0.0;
    } else {
      v2.x = -n.x * n.z / l; v2.y = -n.y * n.z / l;
      v2.z = sqrt(n.x * n.x + n.y * n.y);
    }
  }
  obj->obj.sphere.n = n;
  obj->obj.sphere.v1 = v;
  obj->obj.sphere.v2 = v2;
  fprintf(stderr, "sphere s%d %lf %lf %lf %lf\n", s, r, x, y, z);
#ifdef DEBUG
  __CMIX(pure)fprintf
    (stderr, "/* Sphere: pole (%g, %g, %g) */\n", n.x, n.y, n.z);
  __CMIX(pure)fprintf
    (stderr, "/* Sphere: equator (%g, %g, %g) */\n", v.x, v.y, v.z);
  __CMIX(pure)fprintf
    (stderr, "/* Sphere: normal (%g, %g, %g) */\n", v2.x, v2.y, v2.z);
#endif
  __CMIX(pure)fprintf
    (stderr, "/* Check: p*e = %g, p*n = %g, e*n = %g */\n",
     n.x*v.x  + n.y*v.y  + n.z*v.z,
     n.x*v2.x + n.y*v2.y + n.z*v2.z,
     v.x*v2.x + v.y*v2.y + v.z*v2.z);
  return obj;
}    

lightT create_light(myfloat r, myfloat g, myfloat b,
		  myfloat x, myfloat y, myfloat z)
{
  lightType *li;
  li = light + NEXT(lightIdx);
  li->p.x = x;
  li->p.y = y;
  li->p.z = z;
  li->c.red = r;
  li->c.green = g;
  li->c.blue = b;
  return li;
}
