#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <GL/glut.h>

extern "C" char prg_name[];

typedef unsigned char uint8;

double* line, * lineoffs;
static pixelpipe[2], keypipe[2], cmdpipe[2];
static double* window;
int rayWin;
int rayWindowHeight;
static int rayWindowWidth;
static int rayWindowy;
static int lineleft;
static int use_color;

#define CLOSEWINDOW 42

static pid_t makePipesAndFork() {
  pid_t pid;
  if (pipe(pixelpipe)) { perror(prg_name); exit(-1); }
  if (pipe(cmdpipe)) { perror(prg_name); exit(-1); }
  if (pipe(keypipe)) { perror(prg_name); exit(-1); }
  if ((pid = fork()) == -1) { perror(prg_name); exit(-1); }
  fcntl(pixelpipe[0], F_SETFL, O_NONBLOCK);
  fcntl(cmdpipe[0], F_SETFL, O_NONBLOCK);
  return pid;
}  

static void SetPixel(int X, int Y, double Red, double Green, double Blue)
{
    glColor3d(Red, Green, Blue);
    glBegin(GL_POINTS);
    glVertex2i(X, Y);
    glEnd();
}

static void DisplayCallBack()
{
  double* nextRGB, *red, *green, *blue;
  int x, y;

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glFinish();

  // This should really be done by using DrawPixels!!!
  nextRGB = window;
  for (y = 0; y < rayWindowy; y++) {
    for (x = 0; x < rayWindowWidth; x++) {
      red = nextRGB++; green = nextRGB++; blue = nextRGB++;
      SetPixel(x, y, *red, *green, *blue);
    }
  }
  
  glFinish();    // Force the graphics to the window
  
  // check for errors
  GLenum ErrorCode;
  while ((ErrorCode = glGetError()) != GL_NO_ERROR) {
    fprintf(stderr, "%s\n", gluErrorString(ErrorCode));
  }
}

static void KeyboardCallBack(unsigned char Key, int Xmouse, int Ymouse)
{
  int ch, notused;
  notused = Xmouse = Ymouse;
  ch = Key;
  write(keypipe[1], &ch, 1 * sizeof(int));
}

static void MouseCallBack(int Button, int State, int Xmouse, int Ymouse)
{
  int ch, notused;
  notused = Xmouse = Ymouse;
  if (Button == GLUT_LEFT_BUTTON && State == GLUT_UP) {
    ch = 256;
    write(keypipe[1], &ch, 1 * sizeof(int));
  }
}

static void IdleCallBack()
{
  int st, x;
  int cmd;
  st = read(cmdpipe[0], &cmd, 1 * sizeof(int));
  if (st != -1 && cmd == CLOSEWINDOW) {
    exit(0);
  }
  st = read(pixelpipe[0], lineoffs, lineleft);
  if (st == -1) {
    if (errno == EAGAIN)
      return;
    else
      perror("Ray tracer");
  } else {
    lineleft -= st;
    lineoffs += st / sizeof(double);
  }
  if (lineleft == 0) {
    lineleft = rayWindowWidth * 3 * sizeof(double);
    lineoffs = line;
    for (x = 0; x < rayWindowWidth; x++) {
      window[rayWindowWidth * rayWindowy * 3 + x * 3    ] = line[x * 3];
      window[rayWindowWidth * rayWindowy * 3 + x * 3 + 1] = line[x * 3 + 1];
      window[rayWindowWidth * rayWindowy * 3 + x * 3 + 2] = line[x * 3 + 2];
      SetPixel(x, rayWindowy, line[x * 3], line[x * 3 + 1], line[x * 3 + 2]);
    }
    rayWindowy++;
    glFinish();    // Force the graphics to the window
  }
}

extern "C" int initWindow(char *title, int height, int width, int use_c)
{
  pid_t pid;
  use_color = use_c;
  line = (double*)malloc(width * 3 * sizeof(double));
  rayWindowHeight = height;
  rayWindowWidth = width;
  
  // Initialize OpenGL to draw individual pixels in an X-window
  //
  // Create a window with Width x Height pixels which are addressed
  // by x = 0,..., Width-1 and y = 0,..., Height-1
  //
  int argc = 1;
  char* argv[1];
  argv[0] = prg_name;
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_RGBA | GLUT_SINGLE | GLUT_DEPTH); 
  glutInitWindowSize(width, height);
  rayWin = glutCreateWindow(title); 
  glClearColor(0.5, 0.6, 0.4, 0.0);      

  glEnable(GL_DEPTH_TEST);
  glClearDepth(1.0);
  glDepthFunc(GL_LEQUAL);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  // Near = 0 and Far = 1 are distances from the viewpoint along the
  // negative Z-axis. So Near = 0 and Far = 1 means that the Z-coordinates
  // must be in the range [-1, 0]
  glOrtho(0.0, GLdouble(width), 0.0, GLdouble(height), 0.0, 1.0);
  glTranslated(0.25, 0.25, 0.0);    // hack to make both HP and ALPHA work
  glViewport(0, 0, GLsizei(width), GLsizei(height));

  glutKeyboardFunc(KeyboardCallBack);
  glutMouseFunc(MouseCallBack);
  glutIdleFunc(IdleCallBack);
  glutDisplayFunc(DisplayCallBack);

  rayWindowy = 0;
  pid = makePipesAndFork();
  if (pid == 0) {
    line = (double*)malloc(width * 3 * sizeof(double));
    lineoffs = line;
    lineleft = rayWindowWidth * 3 * sizeof(double);
    window = (double*)calloc(width * height * 3, sizeof(double));
    glutMainLoop(); // Child process runs glutMainLoop();
    // which takes over and relies on the call-back functions
  };
  return 0; // Parent process simply returns
}

extern "C" void closeWindow() {
  int cmd;
  cmd = CLOSEWINDOW;
  write(cmdpipe[1], &cmd, 1 * sizeof(int));
}

extern "C" void plot(int x, int y, double red, double green, double blue)
{
  int notused;
  notused = y;
  if (!use_color) {
    red = green = blue = (red + green + blue) / 3.0;
  }
  line[x * 3]     = red   ;
  line[x * 3 + 1] = green ;
  line[x * 3 + 2] = blue  ;
}

extern "C" void refreshLine (int y) {
  int st;
  if (y != rayWindowy++) {
    fprintf(stderr, "Error: refreshLine(%d) called while y = %d\n",
	    y, rayWindowy);
    exit(-1);
  }
  st = write(pixelpipe[1], line, rayWindowWidth * 3 * sizeof(double));
}

extern "C" int waitKeyOrClick() {
  int ch[1];
  int st;
  if ((st = read(keypipe[0], ch, 1 * sizeof(int))) == -1) {
    perror("Ray tracer");
    exit(-1);
  }
  return ch[0];
}
