#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/mman.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <string.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <errno.h>
#include <sys/un.h>
#include <db.h>
#include <glib.h>

#include "var.h"
#include "uaddr.h"
#include "unode.h"
#include "display.h"
#include "action.h"
#include "user_manage.h"
#include "misc.h"
#include "network.h"
#include "dc_com.h"
#include "bdb.h"
#include "tos_key.h"
#include "timed_out_string.h"

#define MAX_REGISTRATION_TIME (12*3600)		/* up to 12 hours without refresh */

static void	init_uaddr_node(void);
static void manage_uaddr_node(void);

/**************************/
/* Global Transfer System */
/*********************************************************************/
/* the following functions provide the capability to move a transfer */
/* from one client to another. Only queued transfers can be moved.   */
/* thus, a transfer can go from one hub to another if the user from  */
/* where we download has gone to another hub. If the .dctc dir is on */
/* a shared disc (NFS or something like that), a transfer can even   */
/* migrate from one computer to another.                             */
/* only /DL are taken into account.                                  */
/*********************************************************************/

#define MAX_UNRESOLVED_IP 1000

/**************************************************************/
/* scan the list of given address to find associated nickname */
/******************************************************************************************/
/* this function must be called regularly to send $Ping to the IP in the uaddr_partial_db */
/******************************************************************************************/
static void scan_addresses_and_purge_partial_uaddr(void)
{
	DBC *cursor;
	int ret;
	int i;

	typedef struct
	{
		char dl_addr[UADDR_DL];
		UADDR_PARTIAL uap;
		int status;			/* action to do: 0= update, 1=delete */
	} TMP_STORE;

	TMP_STORE *ts;
	int nb_ts=0;

	ts=calloc(sizeof(TMP_STORE),MAX_UNRESOLVED_IP);
	if(ts==NULL)
		return;
	
	ret=uaddr_partial_db->cursor(uaddr_partial_db,NULL,&cursor,0);
	if(ret==0)
	{
		DBT key;
		DBT data;

		memset(&key,0,sizeof(key));
		memset(&data,0,sizeof(data));

		key.data=ts[nb_ts].dl_addr;
		key.ulen=UADDR_DL-1;
		key.flags=DB_DBT_USERMEM;

		data.data=&(ts[nb_ts].uap);
		data.ulen=sizeof(UADDR_PARTIAL);
		data.flags=DB_DBT_USERMEM;

		ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
		while(ret==0)
		{
			char *p;

			ts[nb_ts].dl_addr[key.size]='\0';			/* close the C-string */
			ts[nb_ts].status=0;								/* default: we try to update the entry */

			p=strchr(ts[nb_ts].dl_addr,':');
			if(p==NULL)
				ts[nb_ts].status=1;		/* delete this entry */
			else
			{
				char host[UADDR_DL];
				unsigned short port;

				strcpy(host,ts[nb_ts].dl_addr);

				host[p-ts[nb_ts].dl_addr]='\0';
				port=strtoul(p+1,NULL,10);

				{	/* note: this methode is fast but has one severe problem */
					/* if the remote client has "registered" a FQDN inside UADDR database, when this client */
					/* receives the pong string, it contains the remote host IP, not FQDN */
					struct sockaddr_in dest;
					dest.sin_family=AF_INET;
					dest.sin_port=htons(port);
					if(str_to_inaddr(host,&(dest.sin_addr))==0)
					{
						ts[nb_ts].uap.ping_counter++;
						if(ts[nb_ts].uap.ping_counter>MAX_UADDR_PING)
						{
							ts[nb_ts].status=1;		/* delete this entry */
						}
						else
						{
							int out;
							const char *ping_str="$Ping |";
							out=sendto(srch_sck,ping_str,strlen(ping_str),MSG_NOSIGNAL,(void*)&dest,sizeof(struct sockaddr_in));
							if(out==-1)
							{
								ts[nb_ts].status=1;		/* delete this entry */
							}
						}
					}
					else
					{
						ts[nb_ts].status=1;		/* delete this entry */
					}
				}
			}
			
			nb_ts++;
			if(nb_ts>=MAX_UNRESOLVED_IP)
				break;

			/* load the next key */
			memset(&key,0,sizeof(key));
			memset(&data,0,sizeof(data));

			key.data=ts[nb_ts].dl_addr;
			key.ulen=UADDR_DL-1;
			key.flags=DB_DBT_USERMEM;

			data.data=&(ts[nb_ts].uap);
			data.ulen=sizeof(UADDR_PARTIAL);
			data.flags=DB_DBT_USERMEM;

			ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
		}
		cursor->c_close(cursor);
	}

	for(i=0;i<nb_ts;i++)
	{
		if(ts[i].status==0)		/* update the entry ? */
		{
			set_key_data(uaddr_partial_db,ts[i].dl_addr,strlen(ts[i].dl_addr),&(ts[i].uap),sizeof(UADDR_PARTIAL));
		}
		else
		{	/* delete the key */
			del_key_data(uaddr_partial_db,ts[i].dl_addr,strlen(ts[i].dl_addr));
		}
	}

	if(ts)
		free(ts);
}

/****************************************************************/
/* scan the ready to use UADDR table and delete too old entries */
/****************************************************************/
static void expire_ready_uaddr(void)
{
	DBC *cursor;
	int ret;
	GStringChunk *gsc;
	GPtrArray *gpa;

	gsc=g_string_chunk_new(UADDR_DL);
	gpa=g_ptr_array_new();

	/* build a list of too old entries */
	ret=uaddr_ready_db->cursor(uaddr_ready_db,NULL,&cursor,0);
	if(ret==0)
	{
		DBT key;
		DBT data;
		char nick[UADDR_NICK_SIZE];
		UADDR_READY uar;
		time_t min_time;

		min_time=time(NULL)-MAX_REGISTRATION_TIME;

		memset(&key,0,sizeof(key));
		memset(&data,0,sizeof(data));

		key.data=nick;
		key.ulen=UADDR_NICK_SIZE-1;
		key.flags=DB_DBT_USERMEM;

		data.data=&uar;
		data.ulen=sizeof(uar);
		data.flags=DB_DBT_USERMEM;

		ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
		while(ret==0)
		{
			nick[key.size]='\0';			/* close the C-string */

			if(uar.register_time<min_time)
			{	/* entry is too old, keep its content */
				g_ptr_array_add(gpa,g_string_chunk_insert(gsc,nick));
			}

			ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
		}
		cursor->c_close(cursor);
	}

	/* and purge entries marked as too old */
	if((gpa)&&(gpa->len))
	{
		int i;
		for(i=0;i<gpa->len;i++)
		{
			/* delete the key */
			char *key;

			key=g_ptr_array_index(gpa,i);

			del_key_data(uaddr_ready_db,key,strlen(key));
		}
	}

	if(gsc)
		g_string_chunk_free(gsc);
	if(gpa)
		g_ptr_array_free(gpa,TRUE);
}


/************************************************************************************/
/* this thread purges oldest UADDR entries and tries to fill entry without nickname */
/************************************************************************************/
static void *uaddr_thread(void *dummy)
{
	time_t last_check_step=0;

	init_uaddr_node();

	while(1)
	{
		if((time(NULL)-last_check_step)>60)		/* at least 60 seconds between 2 checks */
		{
			/* there is maybe some addresses without nickname */
			scan_addresses_and_purge_partial_uaddr();

			/* delete uaddr here for a too long time without usage */
			expire_ready_uaddr();
	
			last_check_step=time(NULL);
		}

		manage_uaddr_node();
	}
	pthread_exit(NULL);
}

/******************************************************************************/
/* to reduce duplicated code, the DCTC being the clock master is also the one */
/* performing UADDR action                                                    */
/******************************************************************************/
void create_uaddr_thread(void)
{
	pthread_attr_t thread_attr;
	pthread_t thread_id;

	pthread_attr_init (&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
	if(pthread_create(&thread_id,&thread_attr, (void*)uaddr_thread,NULL))
	{
		disp_msg(ERR_MSG,"create_uaddr_thread","Fatal Error","Fail to start UADDR thread",NULL);
	}
}

/********************************/
/* perform UADDR initialisation */
/**********************************************************************/
/* this function is the only one which doesn't need to lock the UADDR */
/**********************************************************************/
void init_uaddr(void)
{
}

/*************************************************************/
/* the following function removes an entry to the UADDR file */
/*************************************************************/
void delete_uaddr_entry_by_name(char *nickname)
{
	/* a deletion by nickname can only occurs on UADDR ready DB */
	del_key_data(uaddr_ready_db,nickname,strlen(nickname));
}

/*************************************************************/
/* the following function removes an entry to the UADDR file */
/*************************************************************/
void delete_uaddr_entry_by_addr(char *dl_addr)
{
	/* a deletion by nickname can only occurs on UADDR partial DB */
	del_key_data(uaddr_partial_db,dl_addr,strlen(dl_addr));
}

/**********************************************************/
/* check if the given nickname has a known remote address */
/**********************************************************/
/* output: 1=yes, 0=no */
/***********************/
int check_uaddr_entry_by_name(char *nickname)
{
	UADDR_READY *uar;
	int out;
	int ret;

	ret=get_key_data(uaddr_ready_db,nickname,strlen(nickname),(void**)(&uar),&out);
	if(ret!=0)
		return 0;		/* not found */

	free(uar);
	return 1;		/* found */
}

/************************************************/
/* get the remote address of the given nickname */
/*********************************************************************************/
/* output: NULL (not found) or a GString (host:port) to free when no more useful */
/*********************************************************************************/
GString *get_uaddr_dl_addr_by_name(char *nickname)
{
	UADDR_READY *uar;
	int out;

	GString *ret=NULL;

	if(get_key_data(uaddr_ready_db,nickname,strlen(nickname),(void**)(&uar),&out)==0)
	{
		ret=g_string_new(uar->dl_addr);
		free(uar);
	}

	return ret;
}

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* nick is always uniq in the database */
/***************************************/
/* output: 0= ok    */
/*         1= error */
/********************/
int add_uaddr_entry(char *nickname, const char *addr_dl)
{
	UADDR_READY uar;

	strncpy_max(uar.dl_addr,addr_dl,UADDR_DL);
	uar.register_time=time(NULL);
	uar.cyclic_dirty=0;

	if(set_key_data(uaddr_ready_db,nickname,strlen(nickname),&uar,sizeof(uar)))
		return 1;
	return 0;
}

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* addr_dl is always uniq in the database */
/******************************************/
/* output: 0= ok    */
/*         1= error */
/********************/
int add_uaddr_entry_addr_dl_only(char *addr_dl)
{
	UADDR_PARTIAL uap;

	uap.ping_counter=0;

	if(set_key_data(uaddr_partial_db,addr_dl,strlen(addr_dl),&uap,sizeof(uap)))
		return 1;
	return 0;
}

#if 0
/*************************************************************************************************/
/* the following function increases the ping_counter of the UADDR entry having the given address */
/*************************************************************************************************/
/* output: -1= error                 */
/*         new value of ping counter */
/*************************************/
static int increase_ping_counter(char *addr_dl)
{
	UADDR_PARTIAL *uap;
	int out;
	int ret=-1;

	if(get_key_data(uaddr_partial_db,addr_dl,strlen(addr_dl),(void**)&uap,&out)==0)
	{
		uap->ping_counter++;
		ret=uap->ping_counter;

		set_key_data(uaddr_partial_db,addr_dl,strlen(addr_dl),(void*)uap,out);
		free(uap);
	}

	return ret;
}
#endif

/******************************************************************************************************************************/
/* the following function increases the cyclic error counter in the ping_counter of the UADDR entry having the given nickname */
/******************************************************************************************************************************/
/* output: -1= error                 */
/*         new value of ping counter */
/*************************************/
int uaddr_increase_error_flag(char *nickname)
{
	UADDR_READY *uar;
	int out;
	int ret=-1;

	if(get_key_data(uaddr_ready_db,nickname,strlen(nickname),(void**)&uar,&out)==0)
	{
		uar->cyclic_dirty++;
		ret=uar->cyclic_dirty;

		set_key_data(uaddr_ready_db,nickname,strlen(nickname),(void*)uar,out);
		free(uar);
	}

	return ret;
}

/***************************************************************************************************************************/
/* the following function resets the cyclic error counter in the ping_counter of the UADDR entry having the given nickname */
/***************************************************************************************************************************/
/* output: -1= error                 */
/*         new value of ping counter */
/*************************************/
int uaddr_reset_error_flag(char *nickname)
{
	UADDR_READY *uar;
	int out;
	int ret=-1;

	if(get_key_data(uaddr_ready_db,nickname,strlen(nickname),(void**)&uar,&out)==0)
	{
		uar->cyclic_dirty=0;
		ret=uar->cyclic_dirty;

		set_key_data(uaddr_ready_db,nickname,strlen(nickname),(void*)uar,out);
		free(uar);
	}

	return ret;
}

/*************/
/* end UADDR */
/*************/
void exit_uaddr(void)
{
}

/*******************************************/
/* output UADDR content into CMD_KB format */
/*******************************************/
void list_uaddr_content(void)
{
	DBT key;
	DBT data;
	int ret;
	DBC *cursor;

	disp_msg(UADDR_LST_BEGIN,"",NULL);

	/* first, display the UADDR_READY entries */
	ret=uaddr_ready_db->cursor(uaddr_ready_db,NULL,&cursor,0);
	if(ret==0)
	{
		char nick[UADDR_NICK_SIZE];
		UADDR_READY uar;

		memset(&key,0,sizeof(key));
		memset(&data,0,sizeof(data));

		key.data=nick;
		key.ulen=UADDR_NICK_SIZE-1;
		key.flags=DB_DBT_USERMEM;

		data.data=&uar;
		data.ulen=sizeof(uar);
		data.flags=DB_DBT_USERMEM;

		ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
		while(ret==0)
		{
			nick[key.size]='\0';			/* close the C-string */

			disp_msg(UADDR_LST_ENTRY,NULL,nick,uar.dl_addr,NULL);

			ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
		}
		cursor->c_close(cursor);
	}

	/* and the UADDR partial entries */
	ret=uaddr_partial_db->cursor(uaddr_partial_db,NULL,&cursor,0);
	if(ret==0)
	{
		DBT key;
		DBT data;
		char dl_addr[UADDR_DL];
		UADDR_PARTIAL uap;

		memset(&key,0,sizeof(key));
		memset(&data,0,sizeof(data));

		key.data=dl_addr;
		key.ulen=UADDR_DL-1;
		key.flags=DB_DBT_USERMEM;

		data.data=&uap;
		data.ulen=sizeof(uap);
		data.flags=DB_DBT_USERMEM;

		ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
		while(ret==0)
		{
			dl_addr[key.size]='\0';			/* close the C-string */

			disp_msg(UADDR_LST_ENTRY,NULL,"",dl_addr,NULL);

			ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
		}
		cursor->c_close(cursor);
	}


	disp_msg(UADDR_LST_END,"",NULL);
}

/*****************************************************************************/
/* this function scans the UADDR files and looks if there is something to do */
/*****************************************************************************/
/* this function must be called regularly */
/******************************************/
void uaddr_action(void)
{
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------------ UADDR node -------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

static GString *l_unode_name=NULL;

static int l_unode_socket=-1;		/* local unix UDP socket */
static int r_unode_socket=-1;		/* internet UDP socket */

/***************************************************************************************************/
/* create the uaddr node. The node consists in 2 sockets                                           */
/* 1 unix udp socket named: $HOME/.dctc/dctc-unode.udp (to receive local UADDR and search queries) */
/* 1 internet udp socket using the port number stored in unode_port                                */
/***************************************************************************************************/
/* this function can be called more than 1 time to try to create missing socket */
/********************************************************************************/
static void	init_uaddr_node(void)
{
	if(l_unode_name==NULL)
	{
		l_unode_name=g_string_new(dctc_dir->str);
		l_unode_name=g_string_append(l_unode_name,"/dctc-unode.udp");
	}

	if(l_unode_socket==-1)
	{
		l_unode_socket=socket(AF_UNIX,SOCK_DGRAM,0);
		if(l_unode_socket!=-1)
		{
			int zz;
			struct sockaddr_un name;

			name.sun_family=AF_UNIX;
         strcpy(name.sun_path,l_unode_name->str);

			zz=1;
	
			if(setsockopt(l_unode_socket,SOL_SOCKET,SO_REUSEADDR,(char*)&zz,sizeof(zz)))
			{
            perror("l_unode_socket - setsockopt");
				close(l_unode_socket);
				l_unode_socket=-1;
			}
			else 
			{
         	if(bind(l_unode_socket,(void *)&name,sizeof(struct sockaddr_un)))
         	{
					if((errno!=EINVAL)&&(errno!=EADDRINUSE))
					{
            		perror("l_unode_socket - bind");
						close(l_unode_socket);
						l_unode_socket=-1;
					}
					else
					{
						/* it looks like we cannot bind an AF_UNIX socket on a FS object even if it is */
						/* also an AF_UNIX socket and even if it is not used. Hopefully, thanks to semaphore */
						/* we know that we are the only program attached to this socket */
						unlink(l_unode_name->str);
						name.sun_family=AF_UNIX;
         			strcpy(name.sun_path,l_unode_name->str);
         			if(bind(l_unode_socket,(void *)&name,sizeof(struct sockaddr_un)))
         			{
            			perror("l_unode_socket - bind2");
							close(l_unode_socket);
							l_unode_socket=-1;
						}
					}
         	}
			}
		}
		else
			perror("init_uaddr_node: l_unode_socket");
	}

	if(r_unode_socket==-1)
	{
		unode_port=wanted_unode_port;
		r_unode_socket=_x_udp(unode_port);
	}
}

/************************************************/
/* resolve hostname/hostip and update the entry */
/************************************************/
static void refresh_host_ip_assoc(DBT *key, DBT *data,char *hostname, UNODE_DATA *udt)
{
	char *t;

	udt->resolv_creation_date=time(NULL);
	udt->dest_addr.sin_family=AF_INET;

	t=strchr(hostname,':');
	if(t!=NULL)
		udt->dest_addr.sin_port=htons(atoi(t+1));
	else
		udt->dest_addr.sin_port=htons(unode_port);

	if(str_to_inaddr(hostname,&(udt->dest_addr.sin_addr))!=0)
	{	/* if the address is not found, force it to 0 */
		udt->dest_addr.sin_addr.s_addr=0;
	}
}

static void update_unode_ips(void)
{
	DBC *cursor;
	int ret;

	ret=unode_db->cursor(unode_db,NULL,&cursor,0);
	if(ret==0)
	{
		DBT key;
		DBT data;
		char buf1[8192];
		UNODE_DATA udt;

		time_t cur_time;

		GStringChunk *gsc;
		GArray *ga;
		typedef struct
		{
			gchar *ptr_name;
			UNODE_DATA udt;
		} TMP_UPD;

		TMP_UPD tmp_upd;

		gsc=g_string_chunk_new(64);
		ga=g_array_new(FALSE,FALSE,sizeof(TMP_UPD));

		memset(&key,0,sizeof(key));
		memset(&data,0,sizeof(data));

		key.data=buf1;
		key.ulen=sizeof(buf1)-1;
		key.flags=DB_DBT_USERMEM;

		data.data=&udt;
		data.ulen=sizeof(udt);
		data.flags=DB_DBT_USERMEM;

		cur_time=time(NULL);

		ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
		while(ret==0)
		{
			if(udt.entry_enabled)
			{
				if(udt.dynamic_ip)
				{
					if((cur_time-udt.resolv_creation_date)>MAX_IP_VALIDITY)
					{
						/* the main problem here is that we cannot run db->put before the cursor->c_close */
						buf1[key.size]='\0';
						refresh_host_ip_assoc(&key,&data,buf1,&udt);

						tmp_upd.ptr_name=g_string_chunk_insert(gsc,buf1);
						memcpy(&(tmp_upd.udt),&udt,sizeof(udt));
						ga=g_array_append_val(ga,tmp_upd);
					}
				}
			}
			ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
		}
		cursor->c_close(cursor);

		/* insert all updates entries */
		if(ga->len!=0)
		{
			int i;

			for(i=0;i<ga->len;i++)
			{
				TMP_UPD *tud;

				tud=&(g_array_index(ga,TMP_UPD,i));

				set_key_data(unode_db,tud->ptr_name,strlen(tud->ptr_name),&(tud->udt),sizeof(UNODE_DATA));
			}
		}

		g_array_free(ga,TRUE);
		g_string_chunk_free(gsc);
	}
}

/******************************************************************************************/
/* send the given string to all other known unode                                         */
/* the function takes care to avoid transmission of multiple identical string in a minute */
/******************************************************************************************/
static void send_to_other_unode(char *string, int string_len)
{
	DBC *cursor;
	int ret;
	time_t cur_time;

	if(r_unode_socket==-1)		/* not internet socket exists, abort here */
		return;

	if(tos_entry_exists(UNDSND_TOSKEY,string,string_len,0))	/* don't send the same string too fast */
		return;

	update_unode_ips();

	ret=unode_db->cursor(unode_db,NULL,&cursor,0);
	if(ret==0)
	{
		DBT key;
		DBT data;
		char buf1[8192];
		UNODE_DATA udt;

		memset(&key,0,sizeof(key));
		memset(&data,0,sizeof(data));

		key.data=buf1;
		key.ulen=sizeof(buf1)-1;
		key.flags=DB_DBT_USERMEM;

		data.data=&udt;
		data.ulen=sizeof(udt);
		data.flags=DB_DBT_USERMEM;

		cur_time=time(NULL);

		ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
		while(ret==0)
		{
			if(udt.entry_enabled)
			{
				if(udt.dest_addr.sin_addr.s_addr!=0)
				{
					int out;

					out=sendto(r_unode_socket,string,string_len,MSG_NOSIGNAL,(void*)&(udt.dest_addr),sizeof(struct sockaddr_in));
					if(out!=string_len)
					{
						perror("send_to_other_unode - sendto fails");
					}
				}
			}
			ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
		}
		
		cursor->c_close(cursor);
	}
	add_tos_entry(UNDSND_TOSKEY,UNODE_MIN_RESEND_DELAY,string,string_len,NULL,0);	/* don't send the same string too fast */
}

/****************************************************/
/* read local unix socket and handle its processing */
/****************************************************/
static void handle_local_unode_message(void)
{
	char buf[8192];
	int ret;
	char *t;

	ret=recv(l_unode_socket,buf,sizeof(buf)-1,MSG_NOSIGNAL);
	if(ret==-1)
	{
		disp_msg(ERR_MSG,"handle_local_unode_message","recv error",strerror(errno),NULL);
		return;
	}

	if(ret<2)
		return;

	buf[ret]='\0';
	t=strrchr(buf,'|');
	if(t==NULL)
		return;

	if(!strncmp(buf,"$Search ",strlen("$Search ")))
	{
		/* it is a search query */
		if(!strncmp(buf,"$Search Hub:",strlen("$Search Hub:")))
		{
			/* it is a search query in passive mode, discard it */
			return;
		}

		send_to_other_unode(buf,ret);
	}
	else if(!strncmp(buf,"$UADDR? ",strlen("$UADDR? ")))
	{
		/* it is a UADDR query */
		GString *tmp;
		
		*t='\0';		/* remote the trailing | */

		tmp=get_uaddr_dl_addr_by_name(buf+strlen("$UADDR? "));
		if(tmp!=NULL)
		{	/* a address is already in cache, ignore the query */
			g_string_free(tmp,TRUE);
			return;
		}

		*t='|';		/* restore the trailing | */
	
		send_to_other_unode(buf,ret);
	}
}

/**************************************************/
/* check if the given "from" is an allowed client */
/**************************************************/
/* output: 0=no, 1=yes */
/***********************/
static int is_a_valid_from(struct sockaddr *from, int from_len)
{
	DBC *cursor;
	int ret;
	int end_code=0;
	time_t cur_time;

	update_unode_ips();

	ret=unode_db->cursor(unode_db,NULL,&cursor,0);
	if(ret==0)
	{
		DBT key;
		DBT data;
		char buf1[8192];
		UNODE_DATA udt;

		memset(&key,0,sizeof(key));
		memset(&data,0,sizeof(data));

		key.data=buf1;
		key.ulen=sizeof(buf1)-1;
		key.flags=DB_DBT_USERMEM;

		data.data=&udt;
		data.ulen=sizeof(udt);
		data.flags=DB_DBT_USERMEM;

		cur_time=time(NULL);

		ret=cursor->c_get(cursor,&key,&data,DB_FIRST);
		while(ret==0)
		{
			if(udt.entry_enabled)
			{
				if(udt.dest_addr.sin_addr.s_addr!=0)
				{

					if(from_len==sizeof(struct sockaddr_in))
					{
						if(!memcmp(from,&(udt.dest_addr),from_len))
						{
							end_code=1;
							break;
						}
					}
				}
			}
			ret=cursor->c_get(cursor,&key,&data,DB_NEXT);
		}
		
		cursor->c_close(cursor);
	}

	return end_code;
}

/**************************************************/
/* read internet socket and handle its processing */
/**************************************************/
static void handle_remote_unode_message(void)
{
	char buf[8192];
	int ret;
	char *t;
	struct sockaddr from;
	int from_len=sizeof(from);

	ret=recvfrom(r_unode_socket,buf,sizeof(buf)-1,MSG_NOSIGNAL,&from,&from_len);
	if(ret==-1)
	{
		disp_msg(ERR_MSG,"handle_remote_unode_message","recv error",strerror(errno),NULL);
		return;
	}

	if(ret<2)
		return;

	buf[ret]='\0';
	t=strrchr(buf,'|');
	if(t==NULL)
		return;

	if(!is_a_valid_from(&from,from_len))	/* someone tries to jam the system */
		return;

	if(!strncmp(buf,"$Search ",strlen("$Search ")))
	{
		/* it is a search query */
		if(!strncmp(buf,"$Search Hub:",strlen("$Search Hub:")))
		{
			/* it is a search query in passive mode, discard it. This case should never happens */
			return;
		}

		/* send the search query to all other DCTC on this computer */
		send_dc_line_to_dctc_link_direct(NULL,buf);

		/* and also to the my own dctc */
		add_new_sim_input(0,buf);
	}
	else if(!strncmp(buf,"$UADDR? ",strlen("$UADDR? ")))
	{
		/* it is a UADDR query */
		GString *tmp;
		
		*t='\0';		/* remote the trailing | */

		tmp=get_uaddr_dl_addr_by_name(buf+strlen("$UADDR? "));
		if(tmp==NULL)
		{	/* there is no known user having the given name */
			return;
		}

		/* tip top, this personne is known */
		tmp=g_string_prepend(tmp,"$UADDR |");
		g_string_sprintfa(tmp,"|%s|",buf+strlen("$UADDR? "));

		/* and return the reply to the sender */
		if(sendto(r_unode_socket,tmp->str,tmp->len,MSG_NOSIGNAL|MSG_DONTWAIT,&from,from_len)!=tmp->len)
		{
			disp_msg(ERR_MSG,"handle_remote_unode_message","sendto fail",strerror(errno),NULL);
		}
		g_string_free(tmp,TRUE);
	}
	else if(!strncmp(buf,"$UADDR ",strlen("$UADDR ")))
	{
		/* it is an UADDR result */
		gchar **arr;

		arr=g_strsplit(buf,"|",0);
		if(arr!=NULL)
		{
			add_uaddr_entry(arr[2],arr[1]);
			g_strfreev(arr);
		}
	}
}

static void manage_uaddr_node(void)
{
	fd_set rd;
	int n;

	n=-1;
	FD_ZERO(&rd);

	if(l_unode_socket!=-1)
	{
		FD_SET(l_unode_socket,&rd);
		n=MAX(n,l_unode_socket);
	}

	if(r_unode_socket!=-1)
	{
		FD_SET(r_unode_socket,&rd);
		n=MAX(n,r_unode_socket);
	}

	if(n==-1)
	{
		/* nothing to monitor */
		sleep(1);
	}
	else
	{
		struct timeval tv;
		int ln;

		/* wait at most 1 second */
		tv.tv_sec=1;
		tv.tv_usec=0;

		ln=select(n+1,&rd,NULL,NULL,&tv);
		if(ln>0)
		{
			if( (l_unode_socket!=-1) && FD_ISSET(l_unode_socket,&rd) )
			{
				handle_local_unode_message();
			}

			if( (r_unode_socket!=-1) && FD_ISSET(r_unode_socket,&rd) )
			{
				handle_remote_unode_message();
			}
		}
	}

	/* check if the user wants to modify the UNODE port */
	if(wanted_unode_port!=unode_port)
	{
		if(r_unode_socket!=-1)
		{
			shutdown(r_unode_socket,2);
			close(r_unode_socket);
			r_unode_socket=-1;
		}
	}

	if((l_unode_socket==-1)||(r_unode_socket==-1))
	{
		init_uaddr_node();
	}

}


