#!/usr/bin/perl -w
#
# $Id: autopartkit-check.pl 180 2004-02-15 10:59:39Z pere $
#
# Script that verifies a autopartkit table and prints some useful
# information (minimum disksize, ...)
#
# Copyright (C) 2003 Morten Werner Olsen <werner@skolelinux.no>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# On Debian GNU/Linux systems, the complete text of the GNU General
# Public License can be found in `/usr/share/common-licenses/GPL'.
#

use strict;

# check input
if (scalar (@ARGV) ne 2) {
    print "Usage: $0 <filename> <ramsize in MB>\n";
    exit 0;
}

my $filename = $ARGV[0];
my $ramsize = $ARGV[1] + 0; # casting to int by adding 0
#FIXME - handle error when $ARGV[1] is not an int.

# check that filename exists and are readable
unless (-r $filename) { die "Error: $filename is not a readable file.\n"; };
if (-d $filename) { die "Error: $filename is a directory.\n"; };
if (-z $filename) { die "Error: $filename is empty.\n"; };

# check that ramsize is reasonable
if (($ramsize < 64) or ($ramsize > 16384)) {
    die "Error: $ramsize is not a reasonable size of RAM\n";
}

# open file and put the lines in @lines
open FILE, $filename;
my @lines = <FILE>;
close FILE;

# put all relevant into the hash entries
my %entries;
my $ent = \%entries;
my @lvms;
my @lvmparts;
my @normals;
for my $line (@lines) {
    # drop comments, lines starting with whitespace and newlines
    if ($line =~ /^[\#\s\n]/) {
	next;
    }
    
    # add entries in to %entries
    my ($mount, $type, $min, $max) = split /[ ]+/, $line, 5;
    $ent->{mount}->{$mount}->{type} = $type;

    # remove \n from strings
    $ent =~ s/\n//;
    $min =~ s/\n//;
    $max =~ s/\n//;

    # replace $RAMSIZE with $ramsize
    $min =~ s/\$RAMSIZE/$ramsize/;
    $max =~ s/\$RAMSIZE/$ramsize/;

    # handle +, -, * and / in $min and $max
    if ($min =~ /[\+\-\*\/].*/) {
	$min = solve ($min);
    }
    if ($max =~ /[\+\-\*\/]/) {
	$max = solve ($max);
    }

    $ent->{mount}->{$mount}->{min} = $min; 
    $ent->{mount}->{$mount}->{max} = $max;

    # check partition type, and put it in the right box
    if ($type eq "lvm") {
	push @lvms, $mount;
    } elsif ($type =~ /lvm:/) {
	push @lvmparts, $mount;
    } else {
	push @normals, $mount;
    }
}

# check that lvm-volumes isn't overfull
for my $lvm (@lvms) {
    my @parts;
    my $min = 0;
    my $max = 0;
    my $lvmmin = $ent->{mount}->{$lvm}->{min};
    my $lvmmax = $ent->{mount}->{$lvm}->{max};

    # find parts in this lvm and add their min and max to $min and $max
    for my $part (@lvmparts) {
	if ($ent->{mount}->{$part}->{type} =~ /:$lvm:/) {
	    push @parts, $part;
	    $min += $ent->{mount}->{$part}->{min};
	    $max += $ent->{mount}->{$part}->{max};
	}
    }

    # check min and max-sizes
    if ($min > $lvmmin) {
	print "  Error: $lvm\'s minimum is more than filled up:\n" .
	    "    Defined min-size  : $lvmmin\n" .
            "    Utilized min-size : $min\n\n";
	exit 1;
    }
    if (($lvmmax ne -1) and ($max > $lvmmax)) {
	print "  Error: $lvm\'s maximum is more than filled up:\n" .
	    "    Defined max-size  : $lvmmax\n" .
            "    Utilized max-size : $max\n\n";
	exit 1;
    }
}

# print minimum harddisk-usage:
my $diskusage = 0;
for my $key (@normals) {
    $diskusage += $ent->{mount}->{$key}->{min};
}
for my $key (@lvms) {
    $diskusage += $ent->{mount}->{$key}->{min};
}

print "$filename\n  Minimum disk space is: $diskusage MB\n";


sub solve {
    my ($eqn) = @_;

    while ($eqn =~ /(\d+)\*(\d+)/) {
	my ($d1, $d2) = ($1, $2);
	my $ans = $d1 * $d2;
	$eqn =~ s/\d+\*\d+/$ans/;
    }

    while ($eqn =~ /(\d+)\/(\d+)/) {
	my ($d1, $d2) = ($1, $2);
	my $ans = $d1 / $d2;
	$eqn =~ s/\d+\/\d+/$ans/;
    }

    while ($eqn =~ /(\d+)\+(\d+)/) {
	my ($d1, $d2) = ($1, $2);
	my $ans = $d1 + $d2;
	$eqn =~ s/\d+\+\d+/$ans/;
    }

    while ($eqn =~ /(\d+)\-(\d+)/) {
	my ($d1, $d2) = ($1, $2);
	my $ans = $d1 - $d2;
	$eqn =~ s/\d+\-\d+/$ans/;
    }
    return $eqn;
}
