/**
 * Non-discrete Finite Automaton to Discrete Finite Automaton
 * compiler.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */

#include <stdio.h>
#include "dfa.h"
#include "nfa.h"
#include "regex.h"
#include "util.h"


/** Prints the specified NFA. */
void
print_nfa( const NFA* nfa)
{
	int i;
	State* state = nfa->states;
	print_set( &nfa->firstpos);
	putchar( '\n');
	print_set( &nfa->lastpos);
	putchar( '\n');
	for( i = 0; i < nfa->count; i++, state++) {
		print_range( state->symbols);
		print_set( &state->followpos);
		if( state->token)
			printf( " %d", state->token);
		putchar( '\n');
	}
}


/** Reads a set of symbols from the specified file until the
 * delimiter. If the first character of the string is the delimeter,
 * it is part of the string and does not terminate it. */
static void
read_symbols( char* symbols, FILE* file)
{
	char* p = symbols;
	char c = read_char( file);
	do {
		*p++ = c;
		c = read_char( file);
	} while( c != ']');
	*p++ = '\0';
}


/** Reads a DFA from the specified file. */
void
read_nfa( NFA* nfa, FILE* file)
{
	char c;
	State* state = nfa->states;
	clear_nfa( nfa);
	read_set( &nfa->firstpos, file);
	fscanf( file, " ");
	read_set( &nfa->lastpos, file);
	fscanf( file, " ");
	while( (c = read_char( file)) != EOF) {
		if( c == '[')
			read_symbols( state->symbols, file);
		else
			sprintf( state->symbols, "%c", c);
		read_set( &state->followpos, file);
		if( getc( file) != '\n') {
			fscanf( file, "%d", &state->token);
			assume_near( getc( file) == '\n', file,
					"expected end-of-line");
		} else
			state->token = 0;
		state++;
		nfa->count++;
	}
}


/** Compiles the specified NFA to a DFA.
 * @see Aho, Fig. 3.44
 */
void
compile_nfa( DFA* dfa, const NFA* nfa)
{
	Set dstates[STATES] = { nfa->firstpos };
	int states = 1;
	int t;
	clear_dfa( dfa);
	for( t = 0; t < states; t++) {
		int final;
		char a;
		for( a = '\t'; a < CHAR_MAX; a++) {
			Set set = dstates[t];
			Set u;
			int i;
			clear_set( &u);
			while( (i = remove_first_element( &set)) != EMPTY_SET) {
				const State* p = &nfa->states[i];
				if( strchr( p->symbols, a) != NULL)
					add_set( &u, &p->followpos);
			}
			if( is_empty( &u)) {
				dfa->transition[t][(int)a] = NO_TRANSITION;
				continue;
			}
			for( i = 0; i < states; i++)
				if( equals( &dstates[i], &u))
					break;
			if( i == states) {
				assumex( states < STATES,
						"limit of %d DFA states exceeded", STATES);
				dstates[states++] = u;
			}
			dfa->transition[t][(int)a] = i;
		}
		final = first_intersection( &dstates[t], &nfa->lastpos);
		if( final != EMPTY_SET) {
			add_element( &dfa->final, t);
			dfa->tokens[t] = nfa->states[final].token;
		}
	}
}
