/** Discrete push-down automaton.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include "pda.h"
#include "symbolstring.h"
#include "util.h"
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>


/** Clears this automaton. */
void
clear_pda( PDA* pda)
{
	memset( pda->move, SYNTAX_ERROR, sizeof pda->move);
}


/** Returns the string for the specified symbol. */
const char*
get_symbol( const PDA* pda, int a)
{
	return a ? get_key( &pda->g.symbols, a) : "$";
}


/** Prints the action table. */
static void
print_action( const PDA* pda)
{
	int s;
	for( s = 0; s < STATES; s++) {
		int a;
		for( a = 0; a <= pda->g.symbols.count ; a++) {
			const Action* action = &pda->move[s][a];
			if( a && !contains_element( &pda->g.terminal, a))
				continue;
			printf( action->reduce ?
					(action->move ? "%3d r%-3d %s\n" : "%1$3d acc  %3$s\n") :
					(action->move ? "%3d s%-3d %s\n" : ""),
					s, action->move, get_symbol( pda, a));
		}
	}
}


/** Prints the goto table. */
static void
print_goto( const PDA* pda)
{
	int s;
	for( s = 0; s < STATES; s++) {
		int a;
		for( a = 1; a <= pda->g.symbols.count; a++) {
			const Action* action = &pda->move[s][a];
			if( !contains_element( &pda->g.nonterminal, a))
				continue;
			if( !action->reduce && action->move)
				printf( "%3d  %-3d %s\n",
						s, action->move, get_symbol( pda, a));
		}
	}
}


/** Prints this automaton. */
void
print_pda( const PDA* pda)
{
	print_action( pda);
	print_goto( pda);
}


/** Reads an automaton from the specified file. */
void
read_pda( PDA* pda, FILE* file)
{
	int from, to, a, production;
	char* symbol;
	clear_pda( pda);
	while( fscanf( file, "%d %d %*s %as", &from, &to, &symbol) == 3) {
		int a = find( &pda->g.symbols, symbol);
		Action* action;
		action = &pda->move[from][a];
		action->reduce = false;
		action->move = to;
		free( symbol);
	}
	while( fscanf( file, "%% %d %*s %d", &from, &production) == 2) {
		for( a = 0; a <= pda->g.symbols.count; a++) {
			Action* action = &pda->move[from][a];
			if( action->reduce)
				break;
			if( !action->move) {
				action->reduce = true;
				action->move = production;
			}
			if( production == 0)
				break;
		}
	}
}


static int move( const PDA* pda, String* w, int s, int precedence);


/** Returns the precedence of this tree. */
static int
get_precedence( const PDA* pda, const Tree* tree)
{
	int precedence = pda->g.precedence[tree->data];
	return precedence >= 0 ? precedence :
		get_precedence( pda, tree->children[0]);
}


/** Shifts the next input symbol. */
static int
shift( const PDA* pda, String* w, int s, int precedence)
{
	int a = first( w);
	int s_prime = pda->move[s][a].move;
	if( s_prime != SYNTAX_ERROR) {
		Tree* node = next( w);
		int i = move( pda, w, s_prime, precedence);
		w->next->children[i] = node;
		return i;
	} else
		die( "syntax error in state %d on token %s "
				"just before line %d column %d",
				s, get_symbol( pda, a), w->line, w->column);
}


/** Returns the production used to reduce. */
static int
reduce( const PDA* pda, String* w, int s)
{
	Tree* node;
	int r = pda->move[s][0].move;
	if( r == ACCEPT) {
		// Return s' -> s $
		node = create_node( 0);
		node->count = 2;
	} else {
		Production* p = &pda->g.productions[r-1];
		node = create_node( p->lhs);
		node->count = p->count;
	}
	push_symbol( w, node);
	return node->count;
}


/** Returns true if the immediate action is to reduce. */
static bool
reduce_now( const PDA* pda, int s, int left, int right)
{
	int r;
	if( !pda->move[s][0].reduce)
		// Can't reduce. Shift!
		return false;

	// Resolve this shift/reduce conflict.
	if( left <= 0 || right <= 0)
		// Precedence rules do not apply. Shift!
		return false;

	r = pda->move[s][0].move;
	assumex( r != 0, "unexpected accept action");
	switch( pda->g.productions[r-1].count) {
		case 2:
			// Unary expression. Reduce!
			return true;
		case 3:
			// Resolve by precedence. Reduce if equal precedence.
			return left >= right;
		default:
			// Non-operator production. Shift!
			return false;
	}
}


/** Runs this automaton on the specified string. */
static int
move( const PDA* pda, String* w, int s, int left)
{
	int a = first(w);
	int right = get_precedence( pda, w->next);
	int i = 
		pda->move[s][a].reduce ||
		reduce_now( pda, s, left, right) ?
		reduce( pda, w, s) :
		shift( pda, w, s, right ? right : left);
	return i > 0 ? i-1 : move( pda, w, s, left);
}


/** Parses the specified input stream.
 * @see Aho Fig. 4.30. LR parsing program. */
Tree*
parse( const PDA* pda, FILE* input)
{
	String w;
	set_string( &w, input, &pda->g.symbols);
	move( pda, &w, INITIAL_STATE, 0);
	{
		Tree* top = next( &w);
		Tree* tree = top->children[1];
		top->count = 0;
		destroy_tree( top);
		return tree;
	}
}
