/* Emcast - Endhost multicast library
 * Copyright (C) 2001  The Regents of the University of Michigan
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

#include "gethostbyname.h"

#include <config.h>

#include <stdlib.h>
#include <sys/types.h>

#include <errno.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <netdb.h>




/* Thread safe gethostbyname.  The only valid fields are sin_len,
   sin_family, and sin_addr.  Nice name */

int
emgethostbyname (const char* hostname, struct sockaddr_in* sa)
{
  int rv = -1;

#ifdef HAVE_GETHOSTBYNAME_R_GLIBC
  {
    struct hostent result_buf, *result;
    size_t len;
    char* buf;
    int herr;
    int res;

    len = 1024;
    buf = (char*) malloc(len);

    while ((res = gethostbyname_r (hostname, &result_buf, buf, len, &result, &herr)) 
	   == ERANGE)
      {
	len *= 2;
	buf = (char*) realloc (buf, len);
      }

    if (res || result == NULL || result->h_addr_list[0] == NULL)
      goto done;

    if (sa)
      {
	sa->sin_family = result->h_addrtype;
	memcpy(&sa->sin_addr, result->h_addr_list[0], result->h_length);
      }

    rv = 0;

  done:
    free (buf);
  }

#else
#ifdef HAVE_GET_HOSTBYNAME_R_SOLARIS
  {
    struct hostent result;
    size_t len;
    char* buf;
    int herr;
    int res;

    len = 1024;
    buf = (char*) malloc (len);

    while ((res = gethostbyname_r (hostname, &result, buf, len, &herr)) == ERANGE)
      {
	len *= 2;
	buf = (char*) realloc (buf, len);
      }

    if (res || hp == NULL || hp->h_addr_list[0] == NULL)
      goto done;

    if (sa)
      {
	sa->sin_family = result->h_addrtype;
	memcpy(&sa->sin_addr, result->h_addr_list[0], result->h_length);
      }

    rv = 0;

  done:
    free (buf);
  }

#else
#ifdef HAVE_GETHOSTBYNAME_R_HPUX
  {
    struct hostent result;
    struct hostent_data buf;
    int res;

    res = gethostbyname_r (hostname, &result, &buf);

    if (res == 0)
      {
	if (sa)
	  {
	    sa->sin_family = result.h_addrtype;
	    memcpy(&sa->sin_addr, result.h_addr_list[0], result.h_length);
	  }
	
	rv = 0;
      }
  }

#else
  {
    struct hostent* he;

    he = gethostbyname(hostname);
    if (he != NULL && he->h_addr_list[0] != NULL)
      {
	if (sa)
	  {
	    sa->sin_family = he->h_addrtype;
	    memcpy(&sa->sin_addr, he->h_addr_list[0], he->h_length);
	  }

	rv = 0;
      }
  }
#endif
#endif
#endif

  return rv;
}

