/* Emcast - Endhost multicast library
 * Copyright (C) 2001  The Regents of the University of Michigan
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <assert.h>
#include <limits.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "libemcast.h"
#include "emcast-protocol.h"
#include "util.h"
#include "gethostbyname.h"
#include "tempfifo.h"


const unsigned int emcast_major_version = EMCAST_MAJOR_VERSION;
const unsigned int emcast_minor_version = EMCAST_MINOR_VERSION;
const unsigned int emcast_micro_version = EMCAST_MICRO_VERSION;
const unsigned int emcast_interface_age = EMCAST_INTERFACE_AGE;
const unsigned int emcast_binary_age    = EMCAST_BINARY_AGE;

#ifndef MIN
#define MIN(A,B) ((A<B)?(A):(B))
#endif

#ifndef INET_ADDRSTRLEN
#define INET_ADDRSTRLEN 16
#endif

#ifndef socklen_t
#define socklen_t size_t
#endif

#define MAXBUF USHRT_MAX
#define SHORTCPY(BUF, OFF, SHRT) do {    \
   unsigned short _shrt = htons(SHRT);  \
   ((char*) BUF)[OFF]   = ((char*) (&_shrt))[0];  \
   ((char*) BUF)[OFF+1] = ((char*) (&_shrt))[1];  \
   } while (0)

#define FIFOTMPL "/tmp/emcastXXXXXX"

/* Emcast struct is defined elsewhere */
struct _Emcast
{
  enum {IPM, PROCESS} type; 
  int fd;
  int ioerror;	/* Is emcast invalid due to internal error? */
  union
  {
    struct sockaddr sa;
    struct
    {
      int   fd_out;
      int   fd_in;
      pid_t pid;
    } p;
  } u;
};

#define msa 	 	u.sa
#define pfd_out  	u.p.fd_out
#define pfd_in	 	u.p.fd_in
#define ppid	 	u.p.pid

#define IOERROR(EM, ER) do{(EM)->ioerror = 1; /* fprintf (stderr, "libemcast: IO error %d at line %d\n", ER, __LINE__); */ return (ER); } while (0)
#define return_EMCAST_EINTERNAL do{ /* fprintf (stderr, "libemcast: Internal error at line %d\n", __LINE__); */ return EMCAST_EINTERNAL; } while (0)


static int process_init (const char* proto, pid_t* pid, int* infd, int* outfd,
			 int* fifofdp);
static ssize_t drainn (int fd, size_t n);


/**

   emcast_new:
   @emcastp: Emcast object (this function sets)
   @protocol: Multicast protocol (or URL)

   Create a new Emcast session for the specified protocol (or URL).
   If a URL is passed, the protocol is determined from that.  If the
   protocol is NULL, IP Multicast is used.  Otherwise, the appropriate
   handler is launched.

   Returns: 0 on success, error number if failure.  emcastp is set.
   
*/
int
emcast_new (Emcast** emcastp, const char* protocol)
{
  Emcast* emcast;
  char* semi;

  if (!emcastp)
    return EMCAST_EBADARG;

  *emcastp = NULL;

  /* If the second part is only integer, then we're using IPM (it's
     the port number). */
  semi = strchr (protocol, ':');
  if (semi)
    {
      semi++;
      while (*semi && isdigit((int)*semi)) ++semi;
    }

  if (protocol == NULL || (semi && !*semi))
    {
      int sock;

      sock = socket (AF_INET, SOCK_DGRAM, 0);
      if (sock < 0) 
	return_EMCAST_EINTERNAL;

      emcast = (Emcast*) calloc (1, sizeof (Emcast));
      if (!emcast) 
	return_EMCAST_EINTERNAL;
      emcast->type = IPM;
      emcast->fd = sock;
    }
  else
    {
      pid_t pid;
      int infd, outfd, fifofd;
      int rv;
      int protocol_len;
      char* proto;
      char* semi;

      /* Extract protocol from protocol/URL. */
      protocol_len = strlen(protocol);
      proto = malloc (protocol_len + 1);
      if (!proto) 
	return_EMCAST_EINTERNAL;
      memcpy (proto, protocol, protocol_len);
      proto[protocol_len] = '\0';
      semi = strchr (proto, ':');
      if (semi)	*semi = '\0';

      rv = process_init (proto, &pid, &infd, &outfd, &fifofd);
      free (proto);
      if (rv < 0)
	return rv;

      emcast = (Emcast*) calloc (1, sizeof (Emcast));
      if (!emcast)
	{
	  close (fifofd);
	  close (infd);
	  close (outfd);
	  kill (pid, -9);
	  waitpid (pid, NULL, 0);
	  return_EMCAST_EINTERNAL;
	}
      emcast->type    = PROCESS;
      emcast->fd      = fifofd;
      emcast->pfd_out = outfd;
      emcast->pfd_in  = infd;
      emcast->ppid    = pid;
    }

  *emcastp = emcast;
  return 0;
}


static int
process_init (const char* proto, pid_t* pidp, int* infdp, int* outfdp, 
	      int* fifofdp)
{
  const char* p;
  int inp[2] = {0, 0};
  int outp[2];
  pid_t pid = 0;
  char* buf;
  int rv;
  char RV;
  short version;
  char fifonam[] = FIFOTMPL;
  int fifonam_len = 0;
  int fifofd;

  assert (proto && pidp && infdp && outfdp && fifofdp);

  /* Make sure the protocol is all alphanumeric */
  for (p = proto; *p != '\0'; ++p)
    if (!isalnum((int)*p) && *p != '-' && *p != '_')
      return EMCAST_EBADURL;

  /* Create pipes */
  if (pipe(inp) || pipe(outp))
    return_EMCAST_EINTERNAL;

  /* Create FIFO */
  if (!tempfifo (fifonam))
    return_EMCAST_EINTERNAL;
  fifonam_len = strlen (fifonam);

  /* Fork */
 fork_again:
  if ((pid = fork()) < 0)		/* Fork failure */
    {
      /* Try again? */
      if (errno == EAGAIN)
	{
	  sleep (0);
	  goto fork_again;
	}

      /* Clean up */
      close (inp[0]);  close (inp[1]);
      close (outp[0]); close (outp[1]);
      unlink (fifonam);

      return_EMCAST_EINTERNAL;
    }

  else if (pid == 0)			/* Child */
    {
      char* prog;

      /* Close our end of the pipes */
      assert (close (inp[1] ) == 0);
      assert (close (outp[0]) == 0);

      /* Set STDIN/STDOUT */
      assert (dup2 (inp[0],  STDIN_FILENO)  != -1);
      assert (dup2 (outp[1], STDOUT_FILENO) != -1);

      /* Append -emcast to protocol to form program name */
      prog = (char*) malloc (strlen(proto) + sizeof ("-emcast") + 1);
      assert (prog);
      strcpy (prog, proto);
      strcat (prog, "-emcast");

      /* Exec the handler */
      execl ("/bin/sh", "/bin/sh", "-c", prog, NULL);

      /* NOT REACHED */
      _exit(EXIT_FAILURE);
    }

  /* Close our end of the pipes */
  assert (close (inp[0]) == 0);
  assert (close (outp[1]) == 0);

  /* Write INIT (EMCAST_INIT, EMCAST_VERSION) */
  buf = (char*) malloc (2 + 2 + 2 + fifonam_len);
  SHORTCPY(buf, 0, EMCAST_INIT);
  SHORTCPY(buf, 2, EMCAST_VERSION);
  SHORTCPY(buf, 4, fifonam_len);
  memcpy (&buf[6], fifonam, fifonam_len);
  rv = writen (inp[1], buf, (2 + 2 + 2 + fifonam_len));
  free (buf);
  if (rv != (2 + 2 + 2 + fifonam_len)) goto int_error;

  /* Read return - if it fails, then fork failed */
  rv = readn (outp[0], &RV, 1);
  if (rv == 0)
    {
      unlink (fifonam);
      close (inp[1]);
      close (outp[0]);
      kill (pid, -9);
      waitpid (pid, NULL, 0);

      return EMCAST_ENOPROTO;
    }
  else if (rv != 1 || RV != 0)
    goto int_error;

  /* Read version */
  rv = readn (outp[0], (char*) &version, 2);
  if (rv != 2) goto int_error;
  version = ntohs(version);
  if (version != 1) goto int_error;

  /* Open fifo */
  fifofd = open (fifonam, O_RDONLY);
  if (fifofd < 0) 
    goto int_error;

  /* Delete the fifo - it has now been opened by both ends */
  unlink (fifonam);

  *pidp    = pid;
  *infdp   = outp[0];
  *outfdp  = inp[1];
  *fifofdp = fifofd;

  return 0;

 int_error:
  /* fifo not open */
  unlink (fifonam);
  close (inp[1]);
  close (outp[0]);
  kill (pid, -9);
  waitpid (pid, NULL, 0);

  return_EMCAST_EINTERNAL;
}



/**

   emcast_join:
   @url: URL of the multicast group

   Join the multicast group specified by the URL.  The URL is protocol
   specific.  If the URL is in the form "address:port" where address
   is a dotted decimal IPv4 address or a domain name and port is a 16
   bit port, then IP multicast is used.  Otherwise, an end-host
   multicast handler process will be launched to handle the
   connection.

   Returns: File descriptor on success (>= 0), negative error number
   if failure.

*/
int
emcast_join (Emcast* emcast, const char* url)
{
  /* Emcast expects emcast pointer and URL */
  if (emcast == NULL || url == NULL)
    return EMCAST_EBADARG;

  /* IP Multicast */
  if (emcast->type == IPM)
    {
      char* semi;
      long int port;
      int   hostname_len;
      char* hostname;
      struct in_addr inaddr;
      struct sockaddr sa;
      struct sockaddr_in* sa_in = (struct sockaddr_in*) &sa;
      const int on = 1;
      struct ip_mreq mreq;
      int rv;

      /* URL is in form A:B. */
      semi = strchr(url, ':');
      if (semi == NULL)
	return EMCAST_EBADURL;

      /* Get port */
      ++semi;
      port = strtol (semi, NULL, 10);
      /* (we verified that the port was all digits above) */
      if (port == LONG_MIN || port == LONG_MAX)
	return EMCAST_EBADURL;

      /* Copy address */
      hostname_len = semi - url - 1;
      hostname = (char*) malloc (hostname_len + 1);
      if (!hostname) return_EMCAST_EINTERNAL;
      memcpy (hostname, url, hostname_len);
      hostname[hostname_len] = '\0';

      /* Try to read the name as if were dotted decimal */
      if (inet_aton(hostname, &inaddr) != 0)
	{
	  memcpy(&sa_in->sin_addr, (char*) &inaddr, sizeof(struct in_addr));
	}

      /* Otherwise, do a DNS lookup */
      else
	{
	  rv = emgethostbyname (hostname, sa_in);
	  if (rv != 0)
	    {
	      free (hostname);
	      return EMCAST_EBADURL;
	    }
	}
      free (hostname);

      /* Set family and port */
      sa_in->sin_family = AF_INET;
      sa_in->sin_port = htons(port);

      /* Save address */
      memcpy (&emcast->msa, &sa, sizeof(emcast->msa));

      /* Set socket option to share the UDP port */
      rv = setsockopt (emcast->fd, SOL_SOCKET, SO_REUSEADDR, 
		       (void*) &on, sizeof(on));
      if (rv != 0)
	return_EMCAST_EINTERNAL;

      /* Bind to the socket to some local address and port */
      rv = bind (emcast->fd, &sa, sizeof(sa));
      if (rv != 0)
	return_EMCAST_EINTERNAL;

      /* Create the multicast request structure */
      memcpy (&mreq.imr_multiaddr, &(sa_in)->sin_addr, sizeof(struct in_addr));
      mreq.imr_interface.s_addr = htonl(INADDR_ANY);

      /* Join the group */
      rv = setsockopt (emcast->fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
		       (void*) &mreq, sizeof(mreq));
      if (rv != 0)
	return_EMCAST_EINTERNAL;
    }

  /* EM */
  else if (emcast->type == PROCESS)
    {
      int url_len;
      char* buf;
      int rv;
      char RV;

      /* URL length must be less than USHRT_MAX */
      url_len = strlen(url);
      if (url_len > USHRT_MAX)
	return EMCAST_EBADARG;

      /* Write JOIN */
      buf = malloc (2 + 2 + url_len);
      if (!buf) return_EMCAST_EINTERNAL;
      SHORTCPY(buf, 0, EMCAST_JOIN);
      SHORTCPY(buf, 2, url_len);
      memcpy (&buf[4], url, url_len);

      rv = writen (emcast->pfd_out, buf, 2 + 2 + url_len);
      free (buf);
      if (rv != (2 + 2 + url_len)) IOERROR(emcast, EMCAST_EINTERNAL);

      /* Read response */
      rv = readn (emcast->pfd_in, &RV, 1);
      if (rv != 1 || RV != 0) IOERROR(emcast, EMCAST_EINTERNAL);
    }
  else
    return_EMCAST_EINTERNAL;

  return emcast->fd;
}



/**

   emcast_leave:
   @emcast: Emcast group

   Leave the group.  @emcast cannot be used after this call.

   Returns: 0 on success, error number if failure.

 */
int
emcast_leave (Emcast* emcast)
{
  int status = 0;

  if (emcast == NULL)
    return EMCAST_EBADARG;

  if (emcast->type == IPM)
    {
      int rv;
      struct ip_mreq mreq;

      /* Create the multicast request structure */
      memcpy(&mreq.imr_multiaddr,
	     &((struct sockaddr_in*) &emcast->msa)->sin_addr,
	     sizeof(struct in_addr));
      mreq.imr_interface.s_addr = htonl(INADDR_ANY);

      /* Leave the group */
      rv = setsockopt (emcast->fd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
		       (void*) &mreq, sizeof(mreq));
      if (rv != 0) status = EMCAST_EINTERNAL;
    }

  else if (emcast->type == PROCESS)
    {

      /* Write LEAVE if there wasn't an IO error */
      if (!emcast->ioerror)
	{
	  int rv;
	  unsigned short id;
	  char res;

	  /* Write data */
	  id = htons (EMCAST_LEAVE);
	  rv = writen (emcast->pfd_out, &id, 2);
	  if (rv != 2) { status = EMCAST_EINTERNAL; goto skip; }

	  /* Read response - we ignore any error, but do return it */
	  rv = readn (emcast->pfd_in, &res, 1);
	  if (rv != 1 || res != 0) status = EMCAST_EINTERNAL;
	}

      /* Close fd's */
    skip:
      close (emcast->fd);
      close (emcast->pfd_in);
      close (emcast->pfd_out);
      kill (emcast->ppid, -9);
      waitpid (emcast->ppid, NULL, 0);
    }
  else
    return_EMCAST_EINTERNAL;


  free (emcast);

  return status;
}




/* **************************************** */


/**
   emcast_send:
   @emcast: Emcast group
   @buf: Data to send
   @len: length of @buf

   Send data to the group.  If @len is 0, an empty message is sent to
   the group.

   Returns: number of bytes sent on success (>= 0), negative error
   number if failure.

*/
int
emcast_send (Emcast* emcast, const void* buf, size_t len)
{
  if (emcast == NULL || (!buf && len))
    return EMCAST_EBADARG;

  if (emcast->ioerror)
    return_EMCAST_EINTERNAL;

  if (emcast->type == IPM)
    {
      int bytes_sent;

      bytes_sent = sendto (emcast->fd, (void*) buf, len,
			   0, &emcast->msa, sizeof(emcast->msa));

      if (bytes_sent != len)
	return_EMCAST_EINTERNAL;
    }

  else if (emcast->type == PROCESS)
    {
      int rv; 
      char* wbuf;
      char res;

      if (len > USHRT_MAX) return EMCAST_EBADARG;

      /* Write data */
      wbuf = (char*) malloc (4 + len);
      if (!wbuf) return_EMCAST_EINTERNAL;
      SHORTCPY(wbuf, 0, EMCAST_SEND);
      SHORTCPY(wbuf, 2, len);
      memcpy (&wbuf[4], buf, len);
      rv = writen (emcast->pfd_out, wbuf, 4 + len);
      free (wbuf);
      if (rv != (4 + len)) IOERROR(emcast, EMCAST_EINTERNAL);
      
      /* Read response */
      rv = readn (emcast->pfd_in, &res, 1);
      if (rv != 1) IOERROR(emcast, EMCAST_EINTERNAL);
      if (res != 0) return_EMCAST_EINTERNAL;
    }
  else
    return_EMCAST_EINTERNAL;

  return len;
}




/**
   emcast_recv:
   @emcast: Emcast group
   @buf: Buffer to write to
   @len: Length of @buf

   Receive data from the group.  @buf is @len bytes long.

   Returns: Bytes received on succcess (>= 0), negative error number
   if failure.

*/
int
emcast_recv (Emcast* emcast, void* buf, size_t len)
{
  return emcast_recvfrom (emcast, buf, len, NULL, NULL);
}



/**
   emcast_recvfrom:
   @emcast: Emcast group
   @buf: Buffer to write to
   @len: Length of @buf
   @from: From buffer
   @fromlen: From identifier length

   Receive data from the group.  @buf is @len bytes long.  @from, if
   non-NULL and long enough, is filled in with the ID of the sender.
   @fromlen is the length of @from and will be filled in with the
   bytes copied to @from.  The required length of @from and its final
   contents are protocol dependent.

   IP multicast treats @from as a struct sockaddr.

   Returns: Bytes received on succcess (>= 0), negative error number
   if failure.

*/
int
emcast_recvfrom (Emcast* emcast, void* buf, int len, void* from, size_t* fromlen)
{
  if (!emcast || !buf || !len)
    return EMCAST_EBADARG;

  if (emcast->type == IPM)
    {
      int bytes_received;
      struct sockaddr from_sa;
      int from_sa_len = sizeof(struct sockaddr);

      bytes_received = recvfrom (emcast->fd, (void*) buf, len, 
				 0, &from_sa, &from_sa_len);

      /* Set the address from where this is from */

      if (bytes_received == -1)
	return_EMCAST_EINTERNAL;

      if (from && fromlen)
	{
	  if (from_sa_len > *fromlen)
	    {
	      memcpy (from, &from_sa, from_sa_len);
	      *fromlen = from_sa_len;
	    }
	  else
	    *fromlen = 0;
	}

      return bytes_received;
    }

  else if (emcast->type == PROCESS)
    {
      char hdr[6];
      int rv;
      unsigned short id;
      unsigned short mlen;
      unsigned short mfromlen;

      /* Read header */
      rv = readn (emcast->fd, hdr, sizeof(hdr));
      if (rv != 6) IOERROR(emcast, EMCAST_EINTERNAL);
      memcpy (&id, &hdr[0], 2); 	id = ntohs(id);
      memcpy (&mlen, &hdr[2], 2); 	mlen = ntohs(mlen);
      memcpy (&mfromlen, &hdr[4], 2); 	mfromlen = ntohs(mfromlen);

      if (id != EMCAST_RECV) return_EMCAST_EINTERNAL;

      /* If a buffer is too small, drain and give an error */
      if (len < mlen || (from && mfromlen < *fromlen))
	{
	  rv = drainn (emcast->fd, mlen);
	  if (rv) IOERROR(emcast, EMCAST_EINTERNAL);
	  rv = drainn (emcast->fd, mfromlen);
	  if (rv) IOERROR(emcast, EMCAST_EINTERNAL);
	  return EMCAST_EBADARG;
	}

      /* Read data */
      rv = readn (emcast->fd, buf, mlen);
      if (rv != mlen) IOERROR(emcast, EMCAST_EINTERNAL);

      /* Read from */
      if (from)
	{
	  rv = readn (emcast->fd, from, mfromlen);
	  if (rv != mfromlen) IOERROR(emcast, EMCAST_EINTERNAL);
	  *fromlen = mfromlen;
	}
      else if (mfromlen)	/* Read into dummy buffer */
	{
	  rv = drainn (emcast->fd, mfromlen);
	  if (rv) IOERROR(emcast, EMCAST_EINTERNAL);
	}
	  
      return mlen;
    }
  else
    return_EMCAST_EINTERNAL;

  /* NOTREACHED */
  return 0;
}



/* **************************************** */


/**

   emcast_setopt
   @emcast: Emcast group
   @optname: Name of the option
   @optval: (Pointer to the) Value of the option 
   @optlen: Size of the value

   Set a group option.  The option name is case sensitive.  

   All handlers implement the "loopback" option, which controls
   whether the host receives any packets it sends.  The value of
   "loopback" is an integer in network byte order.  If 0, loopback is
   turned off.  If 1, loopback is turned on.  All other options are
   handler specific.  Where appropriate, values should be in network
   byte order.

   IP Multicast also has a "ttl" option.  Time-to-live (TTL)
   determines the maximum number of hops a sent multicast packet can
   travel.  TTL is an integer in network byte order.

   Returns: 0 on success, error number if failure.

 */
int
emcast_setopt (Emcast* emcast, const char* optname, const void* optval, size_t optlen)
{
  int rv;

  if (!emcast || !optname)
    return EMCAST_EBADARG;

  if (emcast->type == IPM)
    {
      if (!strcmp (optname, "loopback"))
	{
	  char flag = 0;

	  if (optlen < sizeof(int))
	    return EMCAST_EBADVAL;

	  if (*(int*) optval)
	    flag = 1;

	  rv = setsockopt(emcast->fd, IPPROTO_IP, IP_MULTICAST_LOOP,
			  (void*) &flag, sizeof(flag));
	  if (rv != 0)
	    return_EMCAST_EINTERNAL;
	}

      else if (!strcmp (optname, "ttl"))
	{
	  char ttl;

	  if (optlen < sizeof(int))
	    return EMCAST_EBADVAL;

	  ttl = (unsigned char) ntohl(*(unsigned int*) optval);

	  rv = setsockopt(emcast->fd, IPPROTO_IP, IP_MULTICAST_TTL,
			  (void*) &ttl, sizeof(ttl));
	  if (rv != 0)
	    return_EMCAST_EINTERNAL;
	}

      else
	return EMCAST_EBADOPT;

    }

  else if (emcast->type == PROCESS)
    {
      int len;
      char* buf;
      unsigned short optname_len = strlen(optname);
      int rv;
      char RV;

      optname_len = strlen(optname);
      if (optname_len > USHRT_MAX || optlen > USHRT_MAX)
	return EMCAST_EBADARG;

      /* Create message */
      len = 2 + 2 + 2 + optname_len + optlen;
      buf = (char*) malloc(len);
      if (!buf) return_EMCAST_EINTERNAL;
      SHORTCPY(buf, 0, EMCAST_SETOPT);
      SHORTCPY(buf, 2, optname_len);
      SHORTCPY(buf, 4, optlen);
      memcpy (&buf[6], optname, optname_len);
      memcpy (&buf[6+optname_len], optval, optlen);

      /* Write message */
      rv = writen (emcast->pfd_out, buf, len);
      free (buf);
      if (rv != len) IOERROR(emcast, EMCAST_EINTERNAL);

      /* Read response */
      rv = readn (emcast->pfd_in, &RV, 1);
      if (rv != 1) IOERROR(emcast, EMCAST_EINTERNAL);

      if (RV == 1)
	return EMCAST_EBADOPT;
      else if (RV == 2)
	return EMCAST_EBADVAL;
      else if (RV != 0)
	return EMCAST_EFAIL;
    }
  else
    return_EMCAST_EINTERNAL;

  return 0;
}


/**

   emcast_getopt
   @emcast: Emcast group
   @optname: Name of the option
   @optval: Buffer for the value of the option 
   @optlen: Size of the buffer

   Get the value of an a group option.  The option name is case
   sensitive.  See emcast_setopt for more information.

   All handlers implement the "loopback" option.  All other options
   are handler specific.

   Returns: 0 on success, error number if failure.

 */
int
emcast_getopt (Emcast* emcast, const char* optname, void* optval, size_t* optlen)
{
  int rv;

  if (!emcast || !optname || !optval || !optlen)
    return EMCAST_EBADARG;

  if (emcast->type == IPM)
    {
      if (!strcmp ("loopback", optname))
	{
	  char flag;
	  socklen_t flag_size;

	  if (*optlen < sizeof(int))
	    return EMCAST_EBADVAL;

	  flag_size = sizeof(flag);

	  rv = getsockopt(emcast->fd, IPPROTO_IP, IP_MULTICAST_LOOP, 
			  &flag, &flag_size);
	  if (rv < 0)
	    return_EMCAST_EINTERNAL;

	  (*(int*) optval) = htonl(flag);
	  *optlen = sizeof(int);
	}

      else if (!strcmp ("ttl", optname))
	{
	  int ttl;
	  socklen_t ttl_size;

	  if (*optlen < sizeof(int))
	    return EMCAST_EBADVAL;

	  ttl_size = sizeof(ttl);

	  rv = getsockopt(emcast->fd, IPPROTO_IP, IP_TTL, 
			  (void*) &ttl, &ttl_size);
	  if (rv < 0)
	    return_EMCAST_EINTERNAL;

	  (*(int*) optval) = htonl(ttl);
	  *optlen = sizeof(int);
	}

      else
	return EMCAST_EBADOPT;
    }

  else if (emcast->type == PROCESS)
    {
      int rv;
      char* wbuf;
      unsigned optname_len;
      char res;
      unsigned short mlen;

      optname_len = strlen (optname);
      if (optname_len > USHRT_MAX) return EMCAST_EBADARG;

      /* Write GETOPT, length, optname */
      wbuf = (char*) malloc (4 + optname_len);
      if (!wbuf) return_EMCAST_EINTERNAL;
      SHORTCPY(wbuf, 0, EMCAST_GETOPT);
      SHORTCPY(wbuf, 2, optname_len);
      memcpy (&wbuf[4], optname, optname_len);
      rv = writen (emcast->pfd_out, wbuf, 4 + optname_len);
      free (wbuf);
      if (rv != (4 + optname_len)) IOERROR(emcast, EMCAST_EINTERNAL);

      /* Read result, value */
      rv = readn (emcast->pfd_in, &res, 1);
      if (rv != 1) IOERROR(emcast, EMCAST_EINTERNAL);
      if (res == 1) return EMCAST_EBADOPT;
      else if (res != 0) return_EMCAST_EINTERNAL;
      rv = readn (emcast->pfd_in, &mlen, 2);
      if (rv != 2) IOERROR(emcast, EMCAST_EINTERNAL);
      mlen = ntohs (mlen);
      if (mlen > *optlen) /* If buffer too small, drain and return. */
	{
	  rv = drainn (emcast->fd, mlen);
	  if (rv) IOERROR(emcast, EMCAST_EINTERNAL);
	  return EMCAST_EBADARG;
	}
      rv = readn (emcast->pfd_in, optval, mlen);
      if (rv != mlen) IOERROR(emcast, EMCAST_EINTERNAL);
      *optlen = mlen;
    }
  else
    return_EMCAST_EINTERNAL;

  return 0;
}



/* **************************************** */

char* errors[] = {"Success",
		  "General failure", 
		  "Internal failure",
		  "Bad argument",
		  "Malformed/bad URL",
		  "No such protocol",
		  "Bad option",
		  "Bad option value" };


char*  
emcast_strerror (int errnum)
{
  if (errnum < 0)
    errnum = -errnum;

  if (errnum > -EMCAST_EBADVAL)
    return NULL;

  return errors[errnum];
}


/* **************************************** */

static int
drainn (int fd, size_t n)
{
  char dummy[1024];
  unsigned short minlen;
  int rv;

  while (n > 0)
    {
      minlen = MIN(sizeof(dummy), n);
      rv = readn (fd, dummy, minlen);
      if (rv != minlen) return -1;
      n -= rv;
    }

  return n;
}
