--                            GCH COMPONENTS                                --
--                                                                          --
--                          G C H . R U L E S                               --
--                                                                          --
--                              QS_5_2_2                                    --
--                                                                          --
--              Copyright (c) 1999, Vitali Sh.Kaufman.                      --
--                                                                          --
--  Gch is distributed as free software; that is with full sources          --
--  and in the hope that it will be useful, but WITHOUT ANY WARRANTY;       --
--  without even the implied warranty of MERCHANTABILITY or FITNESS         --
--  FOR A PARTICULAR PURPOSE. You can freely copy, modify and redistribute  --
--  this software, provided that full sources are available for the version --
--  being distribute (original and modified), and for a modified version,   --
--  any changes that you have made are clearly indicated.                   --
--                                                                          --
--  Gch was developed by Vitali Sh. Kaufman using a prototype               --
--  and consultations by Sergey I. Rybin.                                   --
------------------------------------------------------------------------------
   ------------------
   -- QS_5_2_2 --
   ------------------
with Ada.Unchecked_Deallocation;
-- to clean memory in Add_Diagnostics

with Asis.Extensions.Flat_Kinds;      use Asis.Extensions.Flat_Kinds;

  separate (Gch.Rules)
   function QS_5_2_2 (E : Element) return Boolean is
   --  As a rule to check we use the following:
   --  "Use  named  parameter association in calls of infrequently
   --  used subprograms or entries with many formal parameters".

   --  This is a global rule.
   --  Constants that clarifies the rule meaning and some
   --  variables that collect information from different elements
   --  are placed into a private part of Gch.Rules package

      Arg_Kind : Flat_Element_Kinds := Flat_Element_Kind (E);
      --  Kind of the Element being visited. We are using the flat Element
      --  classification provided as ASIS-for-GNAT extension to minimize the
      --  number of steps needed for defining the exact position of the
      --  argument Element.
      Subpr_Call : Element;
      Subpr_Decl : Element;
      type Subpr_Kinds is (Proc, Func);
      Subpr_Kind : Subpr_Kinds;
      Result : Boolean := True;
      -- It is never put "False" in the rule checking due to "global"
      -- nature of the rule.

     ---------------------
     --    Is_There     --
     ---------------------
      function Is_There (Sub : Element) return Subpr_Access;
      -- Checks if a subprogram Sub is already in the subprogram list
      -- Subprs and return a reference to it or null
      function Is_There (Sub : Element) return Subpr_Access is
      begin
         Current_Subpr := Subprs;
         while Current_Subpr /= null and then
               not Is_Identical (Current_Subpr.Sub, Sub)
         loop
            Current_Subpr := Current_Subpr.Next;
         end loop;
         return Current_Subpr;
      end Is_There;

     ---------------------
     --  Add_Subpr  --
     ---------------------
      procedure Add_Subpr (Sub1 : Element; Call : Element);
      --    Adds a new subprogram to the subprogram list Subprs.
      --    ###VK do we need subprogram names instead to manage polymorphism?

      procedure Add_Subpr (Sub1 : Element; Call : Element) is
         Temp : Subpr_Access;
      begin
         Temp := new Subpr'
                  (Sub => Sub1,
                   Bad_Calls => (Call, others => Nil_Element),
                   Amount_Calls => 1,
                   Bad_Subpr => True,
                   Next => null);
         if Subprs = null then
            Last_Subpr := Temp;
            Subprs := Temp;
         else
            Last_Subpr.Next := Temp;
            Last_Subpr := Temp;
         end if;
      end Add_Subpr;

     ---------------------
     --   Add_Call      --
     ---------------------
      procedure Add_Call (Call : Element);
      -- Add a "bad" call to the current subprogram record.
      -- This record should be found by Is_There function

      procedure Add_Call (Call : Element) is
      begin
         if Current_Subpr.Amount_Calls < Infrequently then
            Current_Subpr.Amount_Calls := Current_Subpr.Amount_Calls + 1;
            Current_Subpr.Bad_Calls (Current_Subpr.Amount_Calls) := Call;
         else
            Current_Subpr.Bad_Subpr := False;
         end if;
      end Add_Call;

     ----------------------------
     --   Add_Diagnostics      --
     ----------------------------
      procedure Add_Diagnostics;
      -- Add diagnostics about bad calls of bad subprograms.
      -- Can be used during finalization of the rule only.
      procedure Add_Diagnostics is
         Sub : Subpr_Access := Subprs;
         To_Free_Sub : Subpr_Access := Sub;

         --  we have to free some memory here
         procedure Free is new Ada.Unchecked_Deallocation
            (Subpr, Subpr_Access);
      begin
         while Sub /= null loop
            if Sub.Bad_Subpr then
               Result := False;
               for I in 1..Sub.Amount_Calls loop
                  Add_Violation (Sub.Bad_Calls (I), Current_Rule);
               end loop;
            end if;

            --  Sub = To_Free_Sub here
            Sub := Sub.Next;
            Free (To_Free_Sub);
            To_Free_Sub := Sub;

         end loop;
         Subprs := null;  --  it is redundant really
      end  Add_Diagnostics;

      procedure Check_Subprogram_Call (Ass_List : Element_List);
      -- This procedure was created just to manage a problem with object
      -- declaration for unconstrained array type Element_List
      -- We need such a declaration to join several branches of analysis

      procedure Check_Subprogram_Call (Ass_List : Element_List) is
         Assoc_List : Element_List := Ass_List;
      begin
         if Assoc_List'Length < Many then
            return;
         end if;
         -- check if an unnamed association exists
         Main_Loop: for I in Assoc_List'Range loop
           if Is_Nil (Formal_Parameter (Assoc_List(I))) then
               --  Now we work with a "bad" parameter association
               --  of a call
              case Subpr_Kind is
                 when Proc
                  => Subpr_Decl := Corresponding_Called_Entity (E);
                 when Func
                  => Subpr_Decl := Corresponding_Called_Function (E);
              end case;

              if Is_There (Subpr_Decl) /= null then -- Image (E1) ??
                  Add_Call (E);
              else
                  Add_Subpr (Subpr_Decl, E);
                  Waiting := True; -- marks that some potential violations
                                   -- are waiting for a final check

              end if;
              exit Main_Loop;
            end if;
         end loop Main_Loop;
      end Check_Subprogram_Call;

   begin
   -- First of all we have to test if the ASIS implementation is
   -- supported things that are essential for the rule
      if not Is_Formal_Parameter_Named_Notation_Supported then
         raise ASIS_Failed; -- ###VK to add more appropriate diagnostics
      end if;

      --  Check if the given rule can be applied
      --  to a given element.
      if (Arg_Kind in Flat_Expression_Kinds or else
         Arg_Kind in Flat_Statement_Kinds)
      then
         case Arg_Kind is
           when A_Procedure_Call_Statement
                |An_Entry_Call_Statement
                |A_Timed_Entry_Call_Statement
                |A_Conditional_Entry_Call_Statement
                  =>
                     Subpr_Kind := Proc;
                     Check_Subprogram_Call (Call_Statement_Parameters (E));
           when A_Function_Call
                  =>
                     Subpr_Kind := Func;
                     Check_Subprogram_Call (Function_Call_Parameters (E));
           when others =>
                     return True;
         end case;

         return True; -- it is correct due the rule is global
                      -- so, we have to postpone any diagnostics
      else if Arg_Kind = Not_An_Element and then Waiting
         --  it is a finalization part that analyzes information
         --  collected for the whole compilation unit

           then
              Add_Diagnostics; --  Result could be updated there
              Waiting := False;
           end if;
      end if;

      return Result;
   exception
      when ASIS_Inappropriate_Context          |
           ASIS_Inappropriate_Container        |
           ASIS_Inappropriate_Compilation_Unit |
           ASIS_Inappropriate_Element          |
           ASIS_Inappropriate_Line             |
           ASIS_Inappropriate_Line_Number      |
           ASIS_Failed
         =>
         Report_ASIS_Failure ("QS_5_2_2");
         return True;
   end QS_5_2_2;