------------------------------------------------------------------------------
--                                                                          --
--                            GCH COMPONENTS                                --
--                                                                          --
--                          G C H . R U L E S                               --
--                                                                          --
--                               B o d y                                    --
--                                                                          --
--                                                                          --
--              Copyright (c) 1999, Vitali Sh.Kaufman.                      --
--                                                                          --
--  Gch is distributed as free software; that is with full sources          --
--  and in the hope that it will be useful, but WITHOUT ANY WARRANTY;       --
--  without even the implied warranty of MERCHANTABILITY or FITNESS         --
--  FOR A PARTICULAR PURPOSE. You can freely copy, modify and redistribute  --
--  this software, provided that full sources are available for the version --
--  being distribute (original and modified), and for a modified version,   --
--  any changes that you have made are clearly indicated.                   --
--                                                                          --
--  Gch was developed by Vitali Sh. Kaufman using a prototype               --
--  and consultations by Sergey I. Rybin.                                   --
------------------------------------------------------------------------------

with Ada.Wide_Text_IO;                use Ada.Wide_Text_IO;
with Ada.Characters.Handling;         use Ada.Characters.Handling;

with Asis;                            use Asis;
with Asis.Exceptions;                 use Asis.Exceptions;
with Asis.Errors;
with Asis.Compilation_Units;          use Asis.Compilation_Units;
with Asis.Elements;                   use Asis.Elements;
with Asis.Declarations;               use Asis.Declarations;
with Asis.Statements;                 use Asis.Statements;
with Asis.Expressions;                use Asis.Expressions;
with Asis.Extensions;                 use Asis.Extensions;
with Asis.Iterator;                   use Asis.Iterator;

with Asis.Text;                       use Asis.Text;
with Asis.Implementation;             use Asis.Implementation;
with Asis.Implementation.Permissions; use Asis.Implementation.Permissions;

with Gch.Options;                     use Gch.Options;

package body Gch.Rules is

   function QS_5_1_1_1 (E : Element) return Boolean is separate;
   --  As a rule to check we use the following:
   --  "Associate names with loops when they are nested").

   function QS_5_1_1_2 (E : Asis.Element) return Boolean is separate;
   --  As a rule to check we use the following:
   --  "Associate names with any loop that contains
   --  an exit statement".

   function QS_5_2_2 (E : Element) return Boolean is separate;
   --  As a rule to check we use the following:
   --  "Use  named  parameter association in calls of infrequently
   --  used subprograms or entries with many formal parameters".

   function QS_4_3_1_10 (E : Asis.Element) return Boolean is separate;
   --  As a rule to check we use the following:
   --  "Do not explicitly raise predefined or
   --  implementation-defined exceptions".

   function QS_4_3_1_11 (E : Asis.Element) return Boolean is separate;
   --  As a rule to check we use the following:
   --  "Never let an exception propagate beyond its scope".

   function Rule_0 (E : Element) return Boolean is separate;
   --  As a rule to check we use the following:
   --  "Do not write the 'in' for parameters, especially in functions").

   function Rule_1 (E : Element) return Boolean is separate;
   --  As a rule to check we use the
   --  following: "multi-identifier declarations are not alloved").

---------------------------------------------------
-- Common procedures for rule checking functions --
---------------------------------------------------
  ---------------------
  --  Add_Violation  --
  ---------------------
   procedure Add_Violation (Elem: Element; Rule: Rule_Index) is
   -- Adds a new violation found into Diagnostics list
      Temp : Rule_Violation_Node_Access;
   begin

      Temp := new Rule_Violation_Node'
                  (Bad_Element => Elem,
                   Violated_Rule => Rule,
                   Next_Node => null);
      if Diagnostics = null then
         Last_Diagnosis := Temp;
         Diagnostics := Temp;
      else
         Last_Diagnosis.Next_Node := Temp;
         Last_Diagnosis := Temp;
      end if;

      Warnings_Per_Unit := Warnings_Per_Unit + 1;
    end  Add_Violation;

   -------------------------
   -- Report_ASIS_Failure --
   -------------------------
   --  Should be called when an ASIS exception is caught in a rule checking
   --  procedure. Repots ASIS Error Status and ASIS Diagnosis. Rule should
   --  be used as some indication of the rule which check fails (if the
   --  (default) null string is used as an actual for the Rule parameter,
   --  the produced output does not mention the rule causing this failure).
   --
   --  This procedure resets the ASIS diagnosis to an empty string and it
   --  resets ASIS Status to Not_An_Error
   procedure Report_ASIS_Failure (Rule : Wide_String := "") is
   begin
       New_Line;
       Put ("Asis Failed");

       if Rule = "" then
          New_Line;
       else
          Put (" when processing ");
          Put (Rule);
          Put_Line (" rule");
       end if;

       Put_Line ("ASIS Diagnosis:");
       Put (Asis.Implementation.Diagnosis);
       Put ("ASIS Error Status: ");
       Put (Asis.Errors.Error_Kinds'Wide_Image (Asis.Implementation.Status));
       New_Line;
       Failures_Per_Unit := Failures_Per_Unit + 1;

       Asis.Implementation.Set_Status;

   end Report_ASIS_Failure;

   -----------------------------
   -- Simple_Traverse_Element --
   -----------------------------
------------------------------------------------------------------------------
--  This is a partial instantiation of generic Traverse_Element procedure from
--  Asis.Iterator. It allows just a single pre- Operation instead of two pre-
--  and post- operations of Traverse_Element. It seems convenient to have
--  such simplified procedure since very offen we need not any post operation
--  at all.

   procedure Simple_Traverse_Element
     (Element : in     Asis.Element;
      Control : in out Traverse_Control;
      State   : in out State_Information)
   is
      procedure No_Op
        (Element : in     Asis.Element;
         Control : in out Traverse_Control;
         State   : in out State_Information);
      --  "Empty" procedure for post-operation

      procedure No_Op
        (Element : in     Asis.Element;
         Control : in out Traverse_Control;
         State   : in out State_Information)
      is
      begin
         null;
      end No_Op;

      procedure Simple is new
         Traverse_Element (State_Information, Operation, No_Op);
   begin
     Simple (Element, Control, State);
   end Simple_Traverse_Element;

end Gch.Rules;