/*
 * $Id: parser.yy,v 1.4 2000/10/25 22:52:05 mborella Exp $
 * Definitions section
 */

%{

#include <iostream>
#include <cstdio>
#include <string.h>
#include "genparse.h"
#include "clparam_list.h"
#include "logfile.h"
#include "unique_int.h"

#define YYDEBUG 1

// These  prototypes are necessary for C++ compilation
int yyerror(char *);
int yylex();
void dump_short_rep(char);
void add_param_to_list();
void dump_param_type(param_t);

// These are all defined in genparse.cc
extern Clparam_list my_params;
extern Unique_Int longopt_value;
extern Logfile mylog;
extern int yylineno;
extern char *yyfilename;

// This is an object we use over and over to read in parameters
Clparam *param;

%}

%union 
{
  char 	  c;     /* single character */
  char    *str;  /* string */
  int     i;
  param_t p;
}

%token SLASH BEGIN_RANGE END_RANGE RANGE_SPEC1 RANGE_SPEC2 FUNC_SPEC NONE
%token OPEN_BRACE CLOSE_BRACE POUND INCLUDE LT GT MANDATORY DOUBLEQUOTE
%token INVOCATION

%token <str> QUOTED C_VAR ALNUM FILENAME
%token <c>   FIRST_CHAR
%token <i>   FLAG INT FLOAT CHAR STRING 

%type <str>  desc long_param callback default
%type <c>    short_param
%type <p>    type 
%%

/*
 * Rules section
 */

all: globals entries
        | entries
        ;

globals: globals global
        | global
        ;

global: include 
	| invocation
        | mandatory 
        | global_callback
        ;

invocation: POUND INVOCATION C_VAR
	{
		my_params.add_invocation($3);
		mylog.write((string) "adding invocation: " + $3);
	}
	;

include: POUND INCLUDE LT FILENAME GT 
	{ 
  		my_params.add_include($4); 
  		mylog.write((string) "adding include file: " + $4); 
	}
	| POUND INCLUDE QUOTED 
	{ 
		my_params.add_include($3); 
		mylog.write((string) "adding include file: " + $3); 
	}
        ;

mandatory: POUND MANDATORY C_VAR 
	{ 
		my_params.add_mandatory($3); 
		mylog.write((string) "adding mandatory: " + $3); 
	}
        ;

global_callback: callback 
	{ 
		my_params.add_global_callback($1); 
		mylog.write((string) "adding global callback: " + $1); 
	}
	;

entries: entries entry
	| entry
	;

entry: 
	{ 
		param = new Clparam; mylog.write("reading parameter..."); 
	} 
	new_entry 
	{ 
		add_param_to_list(); 
	}

new_entry: param
	| param type 
	{ 
		param->type($2); 
		dump_param_type($2); 
	}
	| param type 
	{
		param->type($2); 
		dump_param_type($2); 
	} 
	options
	;

options: options option
	| option
	;

option: default    
	{ 
		param->default_value($1); 
		mylog.write((string) "  default value: " + $1); 
	}
	| range 
	| callback 
	{ 
		param->callback($1); 
		mylog.write((string) "  callback: " + $1); 
	}
	| descs
	;     

descs: descs desc
	{ 
		param->add_description($2); 
		mylog.write((string) "  description: " + $2);
	}	
	| desc
	{ 
		param->add_description($1); 
		mylog.write((string) "  description: " + $1);
	}	
	;

param: short_param 
	{ 
		param->short_rep($1); 
		dump_short_rep($1); 
		param->internal_name($1 + "___");
		param->external_name($1);
	}
	| NONE SLASH long_param
	{
		param->long_rep($3);
		mylog.write((string) "  long representation: " + $3); 
		param->internal_name((string) "___" + $3);
		param->external_name($3);
	       	param->longopt_value(longopt_value.get());
	        param->long_only(true);
	}
	| short_param SLASH long_param 
	{ 
		param->short_rep($1); 
		dump_short_rep($1); 
		param->long_rep($3); 
		mylog.write((string) "  long representation: " + $3); 
		param->internal_name($1 + (string) "___" + $3);
		param->external_name($1);
	}
	;

short_param: FIRST_CHAR 
	;

long_param: C_VAR 
	;

type: INT        { $$ = int_t; }
	| FLOAT  { $$ = float_t; }
        | STRING { $$ = string_t; }
  	| CHAR   { $$ = char_t; }
	| FLAG   { $$ = flag_t; }
	;

default: ALNUM 
	| OPEN_BRACE QUOTED CLOSE_BRACE { $$ = $2; }
	;

range: BEGIN_RANGE contiguous_range END_RANGE 
	/* | BEGIN_RANGE noncontiguous_range END_RANGE */
	;

contiguous_range: ALNUM range_spec more_range { param->low_range($1); }
	| range_spec more_range
        ;

/* both are needed because of lexing ambiguity */
more_range: ALNUM { param->high_range($1); }
        | C_VAR { param->high_range($1); }
	|
	;

range_spec: RANGE_SPEC1 
	| RANGE_SPEC2
	;

/*
noncontiguous_range: noncontiguous_range ALNUM
        | ALNUM 
	;
*/

callback: C_VAR FUNC_SPEC { $$ = $1; } 
	;

desc: QUOTED 
	;

%%

int yyerror(char *s)
{
  cerr << s << " in " << yyfilename << ": " << yylineno << endl;
  return 1;
}

void dump_short_rep(char c)
{
  mylog.write(LOG_OPTION_NO_CR, "  short representation: "); 
  mylog.write(LOG_OPTION_NO_TS, c);
}

void add_param_to_list()
{
  mylog.write("done reading parameter...adding to list");
  my_params.add(*param);
  delete param;
}

void dump_param_type (param_t p)
{
  mylog.write(LOG_OPTION_NO_CR, "  parameter type: ");
  switch(p)
    {
    case int_t:
      mylog.write(LOG_OPTION_NO_TS, "int");
      break;
    case float_t:
      mylog.write(LOG_OPTION_NO_TS, "float");
      break;
    case string_t:
      mylog.write(LOG_OPTION_NO_TS, "string");
      break;
    case char_t:
      mylog.write(LOG_OPTION_NO_TS, "char");
      break;
    case flag_t:
      mylog.write(LOG_OPTION_NO_TS, "flag");
      break;
    default:
      break;
    }
}
