/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gimageview.h"
#include "gimv_image.h"
#include "fileload.h"
#include "fileutil.h"
#include "prefs.h"
#include "thumbnail.h"
#include "thumbnail_support.h"

/* private functions */
static GtkWidget *resize_image_to_thumbnail_size (GimvImage      *image,
						  gint            thumbsize);
static GtkWidget *create_thumbnail               (Thumbnail      *thumb,
						  gint            thumbsize,
						  ThumbCacheType  type,
						  gboolean        save_cache);
static GtkWidget *load_thumbnail_cache           (Thumbnail      *thumb,
						  gint            thumbsize);
static gboolean   thumbcache_save_check          ();


static GtkWidget *
resize_image_to_thumbnail_size (GimvImage *image, gint thumbsize)
{
   GtkWidget *pixmap = NULL;
   GdkPixmap *gdk_pixmap;
   GdkPixmap *mask;

   gfloat scale;
   gint width = 0, height = 0;

   if (!image || !thumbsize)
      return NULL;

   if (gimv_image_width (image) >= gimv_image_height (image)) {
      scale = (gfloat) thumbsize / (gfloat) gimv_image_width (image);
      width = thumbsize;
      height = gimv_image_height (image) * scale;
   }
   else {
      scale = (gfloat) thumbsize / (gfloat) gimv_image_height (image);
      width = gimv_image_width (image) * scale;
      height = thumbsize;
   }

   if (width > 0 && height > 0) {
      gimv_image_scale_get_pixmap (image, width, height, &gdk_pixmap, &mask);
      pixmap = gtk_pixmap_new (gdk_pixmap, mask);
      gimv_image_free_pixmap_and_mask (gdk_pixmap, mask);
   }

   return pixmap;
}


/*
 *  create_thumbnail:
 *     @ Render thumbnail from original image or thumbnail cache,
 *       and create GtkPixmap Widget.
 *     @ If save_cache equal TRUE and specified thumbnail type,
 *       call save_thumbnail_cache () to save thumbnail to disk.
 *
 *  thumb     : Pointer to Thumbnail struct.
 *  thumbsize : Size of thumbnail. Positive value means thumbnail size is
 *              determined by longer axis, and negative value means determined
 *              by longer axis.
 *  type      : Thumbnail cache type.
 *  Return    : Pointer to GtkPixmap widget(thumbnail).
 */
static GtkWidget *
create_thumbnail (Thumbnail *thumb, gint thumbsize,
		  ThumbCacheType type, gboolean save_cache)
{
   GimvImage *image;
   GtkWidget *gtk_pixmap = NULL;
   gboolean saved = TRUE;

   g_return_val_if_fail (thumb && thumb->info, NULL);

   if (type && !save_cache)
      image = thumbsupport_load_thumbnail_cache (thumb->info->filename, type);
   else
      image = gimv_image_load_file (thumb->info->filename);

   if (image) {
      /* write cache to disk */
      if (save_cache && type) {
	 saved = thumbsupport_save_thumbnail_cache (thumb->info->filename,
						    image, type);
      }

      /* resize to display thumbnail size */
      gtk_pixmap = thumb->pixmap
	 = resize_image_to_thumbnail_size (image, thumbsize);
      thumb->icon
	 = resize_image_to_thumbnail_size (image, 20);

      gimv_image_kill  (image);
   }

   if (gtk_pixmap && saved)
      thumb->cache_type = type;

   return gtk_pixmap;
}


/*
 *  load_thumbnail_cache:
 *     @ Create cache from thumbnail disk cache. 
 *
 *  thumb     : Pointer to Thumbnail struct.
 *  thumbsize : Size of thumbnail.
 *  Return    : Pointer to GtkPixmap widget(thumbnail).
 */
static GtkWidget *
load_thumbnail_cache (Thumbnail *thumb, gint thumbsize)
{
   GtkWidget *pixmap = NULL;
   gint i;

   for (i = 1; i < thumb_prefs_num; i++) {
      if (!pixmap && thumb_prefs[i].load_check (NULL)) {
	 pixmap = create_thumbnail (thumb, thumbsize, i, FALSE);
	 if (pixmap) {
	    return pixmap;
	 }
      }
   }

   return NULL;
}


gchar *
find_thumbcache (const gchar *filename, ThumbCacheType *type)
{
   gchar *thumb_file = NULL;
   gint i;

   for (i = 1; i < thumb_prefs_num; i++) {
      if (thumb_prefs[i].load_check) {
	 thumb_file = thumbsupport_get_thumb_cache_path (filename, i);
	 if (file_exists (thumb_file)) {
	    if (type)
	       *type = i;
	    return thumb_file;
	 }
	 g_free (thumb_file);
      }
   }

   if (type)
      *type = NONE;
   return NULL;
}


static gboolean
thumbcache_save_check ()
{
   switch (conf.cache_write_type) {
   case GIMAGEVIEW:
      if (conf.cache_gimv)
	 return TRUE;
      break;
   case NAUTILUS:
      if (conf.cache_nautilus)
	 return TRUE;
      break;
   case KONQUEROR_L:
      if (conf.cache_konqueror_l)
	 return TRUE;
      break;
   case KONQUEROR_M:
      if (conf.cache_konqueror_m)
	 return TRUE;
      break;
   case KONQUEROR_S:
      if (conf.cache_konqueror_s)
	 return TRUE;
      break;
   case GQVIEW:
      if (conf.cache_gqview)
	 return TRUE;
      break;
   case EE_L:
      if (conf.cache_ee_l)
	 return TRUE;
      break;
   case EE_M:
      if (conf.cache_ee_m)
	 return TRUE;
      break;
   case EE_S:
      if (conf.cache_ee_s)
	 return TRUE;
      break;
   case XVPIC:
      break;
   default:
      if (conf.cache_xvpics)
	 return TRUE;
      break;
   }

   return FALSE;
}


/*
 *  create_thumbnail_auto:
 *     @ Create thumbnail. If disk cache is not exist, create it automaticary.
 *
 *  thumb     : Pointer to Thumbnail struct.
 *  type      : Thumbnail load type
 *              (from cache or original image (or other method...?)).
 *  Return    : Pointer to GtkPixmap widget(thumbnail).
 */
GtkWidget *
create_thumbnail_auto (Thumbnail *thumb, gint thumb_size, ThumbLoadType type)
{
   GtkWidget *pixmap = NULL;

   /* load chache  */
   if (type != CREATE_THUMB) {
      pixmap = load_thumbnail_cache (thumb, thumb_size);
   }

   /* create cache if not exist */
   if (!pixmap && conf.cache_write_type && thumbcache_save_check ())
      pixmap = create_thumbnail (thumb, thumb_size,
				 conf.cache_write_type, TRUE);

   /* create thumbnail from original image */
   if (!pixmap) {
      pixmap = create_thumbnail (thumb, thumb_size, NONE, FALSE);
   }

   return pixmap;
}
