/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "gimageview.h"

#include "dnd.h"
#include "fileutil.h"
#include "icon_widget.h"
#include "thumbnail_view.h"
#include "thumbview_list.h"

#define RAW_SPACE 1
#define COL_SPACE 15
#define BORDER_WIDTH 5

typedef struct ThumbViewData_Tag
{
   GtkWidget *table;
   GtkWidget *eventbox;
   GtkWidget *hbox;
   gint       rownum;
   gint       colnum;
   gint       dest_mode;

   Thumbnail *focused;
} ThumbViewData;


typedef struct ThumbnailData_Tag
{
   GtkWidget *widget;
} ThumbnailData;


static ThumbViewData *listview_new                    (ThumbView *tv);
static void           calc_col_row_num                (ThumbView *tv,
						       gint       mode,
						       gint      *colnum,
						       gint      *rownum);


/******************************************************************************
 *
 *   callback functions.
 *
 ******************************************************************************/
static void
cb_icon_button_press (GtkWidget *icon, GdkEventButton *event, gpointer data)
{
   Thumbnail *thumb = data;
   ThumbView *tv;
   gboolean clear = TRUE, found = FALSE;

   g_return_if_fail (icon && thumb);

   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   /* set selection */
   if (event->type == GDK_BUTTON_PRESS && (event->button == 1)) {
      if (event->state & GDK_SHIFT_MASK) {
	 if (event->state & GDK_CONTROL_MASK)
	    clear = FALSE;

	 found = thumbview_set_selection_multiple (thumb, TRUE, clear);
	 if (!found)
	    thumbview_set_selection_multiple (thumb, FALSE, clear);
      } else if (!thumb->selected) {
	 if (~event->state & GDK_CONTROL_MASK)
	    thumbview_set_selection_all (tv, FALSE);
	 thumbview_set_selection (thumb, TRUE);
      } else if (thumb->selected) {
	 if (event->state & GDK_CONTROL_MASK)
	    thumbview_set_selection (thumb, FALSE);
      }
   }

   /* relay management to reference callback function */
   thumbview_thumb_button_press_cb (icon, event, thumb);
}


static void
cb_icon_focus_in (GtkWidget *icon, GdkEventFocus *event, Thumbnail *thumb)
{
   ThumbView *tv;
   ThumbViewData *tv_data;
   ThumbnailData *thumb_data;

   g_return_if_fail (thumb);

   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_if_fail (tv_data && thumb_data);

   if (tv_data->focused == thumb) return;
   tv_data->focused = thumb;

   listview_adjust (tv, thumb);
}


static gint
cb_expose (GtkWidget *widget, GdkEventExpose *event, ThumbView *tv)
{
   thumbview_resize (tv);
   return TRUE;
}



/******************************************************************************
 *
 *   private functions.
 *
 ******************************************************************************/
static ThumbViewData *
listview_new (ThumbView *tv)
{
   ThumbViewData *tv_data = NULL;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data) {
      tv_data = g_new0 (ThumbViewData, 1);
      tv_data->rownum = 20;
      tv_data->colnum = 20;
      g_hash_table_insert (tv->disp_mode_data, LIST_VIEW_LABEL, tv_data);
   }

   return tv_data;
}


static void
calc_col_row_num (ThumbView *tv, gint mode, gint *colnum, gint *rownum)
{
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *vadj;
   GList *node;
   gint frame_height, icon_size;
   gchar *label;

   g_return_if_fail (tv);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   label = thumbview_num_to_label (mode);
   if (!strcmp (label, LIST_THUMB_LABEL))
      icon_size = tv->ThumbnailSize;
   else
      icon_size = ICON_SIZE;

   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);
   frame_height = vadj->page_size;

   if (frame_height > 0.0)
      *rownum = frame_height / (icon_size + RAW_SPACE + 4);
   else
      *rownum = 10;
   if (!strcmp (label, LIST_THUMB_LABEL))
      ++*rownum;

   if (*rownum < 1) *rownum = 1;

   *colnum = g_list_length (tv->thumblist) / *rownum;
}


static gboolean
calc_thumbframe_pos (Thumbnail *thumb, gint *col, gint *row)
{
   ThumbView *tv;
   ThumbViewData *tv_data;
   GList *node;
   gint pos;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data) return FALSE;

   node = g_list_find (tv->thumblist, thumb);

   if (!node)
      return FALSE;

   pos = g_list_position (tv->thumblist, node);

   *row = pos % tv_data->rownum;
   *col = pos / tv_data->rownum;

   return TRUE;
}


static GtkWidget *
listview_append_thumb_frame (ThumbView *tv, Thumbnail *thumb, gint dest_mode)
{
   ThumbViewData *tv_data;
   ThumbnailData *thumb_data;
   GdkPixmap *pixmap = NULL;
   GdkBitmap *mask = NULL;
   GtkWidget *widget = NULL;
   gint col, row, frame_size;
   gchar *label, *icon_label, buf[BUF_SIZE], *time_str;

   g_return_val_if_fail (tv, NULL);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (tv_data, NULL);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   if (!thumb_data) {
      thumb_data = g_new0 (ThumbnailData, 1);
      g_hash_table_insert (thumb->mode_data, LIST_VIEW_LABEL, thumb_data);
   }

   label = thumbview_num_to_label (dest_mode);
   if (!strcmp (label, LIST_THUMB_LABEL)) {
      frame_size = tv->ThumbnailSize;
      time_str = fileutil_time2str (thumb->info->st.st_mtime);
      g_snprintf (buf, BUF_SIZE, "Name: %s\nSize: %dKB\nDate: %s",
		  g_basename(thumb->info->filename),
		  (int) thumb->info->st.st_size / 1024,
		  time_str);
      icon_label = g_strdup (buf);
      g_free (time_str);
   } else {
      frame_size = ICON_SIZE;
      icon_label = g_strdup (g_basename(thumb->info->filename));
   }

   thumb_data->widget = icon_widget_new (NULL, NULL,
					 icon_label,
					 frame_size, frame_size);
   g_free (icon_label);

   if (!strcmp (label, LIST_VIEW_LABEL) && thumb->icon) {
      gtk_pixmap_get (GTK_PIXMAP (thumb->icon), &pixmap, &mask);
      icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
			      pixmap, mask);
   } else if (!strcmp (label, LIST_THUMB_LABEL) && thumb->pixmap) {
      gtk_pixmap_get (GTK_PIXMAP (thumb->pixmap), &pixmap, &mask);
      icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
			      pixmap, mask);
   }

   gtk_widget_show (thumb_data->widget);

   /* set callback */
   gtk_signal_connect(GTK_OBJECT (thumb_data->widget),"button_press_event",
		      GTK_SIGNAL_FUNC (cb_icon_button_press), thumb);
   gtk_signal_connect (GTK_OBJECT(thumb_data->widget),"button_release_event",
		       GTK_SIGNAL_FUNC(thumbview_thumb_button_release_cb), thumb);
   gtk_signal_connect (GTK_OBJECT(thumb_data->widget),"motion_notify_event",
		       GTK_SIGNAL_FUNC(thumbview_motion_notify_cb), thumb);
   gtk_signal_connect(GTK_OBJECT(thumb_data->widget),"focus_in_event",
		      GTK_SIGNAL_FUNC(cb_icon_focus_in), thumb);

   /* for drag file list */
   dnd_src_set (thumb_data->widget, dnd_types, dnd_types_num);
   gtk_signal_connect(GTK_OBJECT (thumb_data->widget), "drag_begin",
		      GTK_SIGNAL_FUNC (thumbview_drag_begin_cb), tv);
   gtk_signal_connect(GTK_OBJECT (thumb_data->widget), "drag_data_get",
		      GTK_SIGNAL_FUNC (thumbview_drag_data_get_cb), tv);
   gtk_signal_connect(GTK_OBJECT (thumb_data->widget), "drag-data-delete",
		      GTK_SIGNAL_FUNC (thumbview_drag_data_delete_cb), tv);
   gtk_signal_connect(GTK_OBJECT (thumb_data->widget), "drag_end",
		      GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);

   /* attach to table */
   calc_thumbframe_pos (thumb, &col, &row);
   if (!strcmp (label, LIST_VIEW_LABEL)) {
      widget = gtk_hbox_new (FALSE, 0);
      gtk_widget_show (widget);
      gtk_box_pack_start (GTK_BOX (widget), thumb_data->widget, FALSE, FALSE, 0);
   } else if (!strcmp (label, LIST_THUMB_LABEL)) {
      widget = thumb_data->widget;
   }
   gtk_table_attach (GTK_TABLE (tv_data->table), widget,
		     col, col + 1, row, row + 1,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   listview_set_selection (thumb, thumb->selected);

   return thumb_data->widget;
}



/******************************************************************************
 *
 *   public functions.
 *
 ******************************************************************************/
/*
 *  listview_remove_thumbnail_data:
 *
 *  tv : Pointer to the ThumbView struct.
 */
void
listview_remove_thumbview_data (ThumbView *tv)
{
   ThumbViewData *tv_data;

   if (!tv) return;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data) return;

   g_hash_table_remove (tv->disp_mode_data, LIST_VIEW_LABEL); 
   g_free (tv_data);
}


/*
 *  listview_remove_thumbnail_data:
 *
 *  thumb : Pointer to the Thumbnail struct.
 */
void
listview_remove_thumbnail_data (Thumbnail *thumb)
{
   ThumbnailData *thumb_data;

   if (!thumb) return;

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   if (!thumb_data) return;

   g_hash_table_remove (thumb->mode_data, LIST_VIEW_LABEL); 
   g_free (thumb_data);
}


gboolean
list_view_append_thumb_frames (ThumbView *tv, GList *start, gint dest_mode)
{
   Thumbnail  *thumb;
   GList *node;

   g_return_val_if_fail (tv, FALSE);

   node = start;
   while (node) {
      thumb = (Thumbnail *) node->data;

      listview_append_thumb_frame (tv, thumb, dest_mode);

      node = g_list_next (node);
   }

   return TRUE;
}


GtkWidget *
listview_add_thumbnail (Thumbnail *thumb, gint dest_mode,
			ThumbLoadType type)
{
   ThumbnailData *thumb_data;
   GdkPixmap *pixmap = NULL;
   GdkBitmap *mask = NULL;
   gchar *label;

   g_return_val_if_fail (thumb, NULL);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (thumb_data, NULL);

   label = thumbview_num_to_label (dest_mode);

   if (!strcmp (label, LIST_VIEW_LABEL) && thumb->icon) {
      gtk_pixmap_get (GTK_PIXMAP (thumb->icon), &pixmap, &mask);
      icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
			      pixmap, mask);
   } else if (!strcmp (label, LIST_THUMB_LABEL) && thumb->pixmap) {
      gtk_pixmap_get (GTK_PIXMAP (thumb->pixmap), &pixmap, &mask);
      icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
			      pixmap, mask);
   }

   return thumb->icon;
}


GtkWidget *
listview_redraw (ThumbView *tv, gint dest_mode, GtkWidget *scroll_win)
{
   ThumbViewData *tv_data;
   GtkScrolledWindow *scrollwin;
   GList *node;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return NULL;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data)
      tv_data = listview_new (tv);

   tv_data->dest_mode = dest_mode;

   if (tv_data->eventbox) {
      gtk_widget_destroy (tv_data->eventbox);
      tv_data->eventbox = NULL;
   }

   if (scroll_win) {
      if (GTK_BIN (tv->container)->child)
	 gtk_widget_destroy (GTK_BIN (tv->container)->child);   
      tv_data->eventbox = listview_create (tv, dest_mode);
      scrollwin = GTK_SCROLLED_WINDOW (scroll_win);
      gtk_scrolled_window_add_with_viewport (scrollwin, tv_data->eventbox);
   }

   return tv_data->eventbox;
}


/*
 *  listview_refresh_thumbnail:
 *     @ Create thumbnail from original image and draw it.
 *
 *  thumb  : Pointer to the Thumbnail struct.
 *  Return : TRUE if success.
 */
gboolean
listview_refresh_thumbnail (Thumbnail *thumb)
{
   GtkWidget *widget;
   ThumbView *tv;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   widget = listview_add_thumbnail (thumb, tv->disp_mode, LOAD_CACHE);

   if (widget)
      return TRUE;
   else
      return FALSE;
}


GtkWidget *
listview_resize (ThumbView *tv)
{
   ThumbViewData *tv_data;
   gint col, row;
   GList *node;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return NULL;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (tv_data, NULL);

   calc_col_row_num (tv, tv->disp_mode, &col, &row);
   if (row == tv_data->rownum) {
      return tv_data->eventbox;
   } else {
      tv_data->colnum = col;
      tv_data->rownum = row;
   }

   gtk_container_foreach (GTK_CONTAINER(tv_data->table),
			  GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
   gtk_table_resize (GTK_TABLE (tv_data->table),
		     tv_data->rownum, tv_data->colnum);
   list_view_append_thumb_frames (tv, tv->thumblist, tv->disp_mode);

   return tv_data->eventbox;
}


void
listview_adjust (ThumbView *tv, Thumbnail *thumb)
{
   ThumbViewData *tv_data;
   ThumbnailData *thumb_data;   
   GList *node;
   GtkWidget *icon;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   gint top, buttom, left, right;
   gint frame_top, frame_buttom, frame_left, frame_right;
   gfloat pos;

   g_return_if_fail (tv);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_if_fail (tv_data);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_if_fail (thumb_data);

   icon = thumb_data->widget;
   g_return_if_fail (icon);

   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   left = icon->allocation.x;
   right = left + icon->allocation.width;
   top = icon->allocation.y;
   buttom = top + icon->allocation.height;

   frame_left = hadj->value;
   frame_right = frame_left + hadj->page_size;
   frame_top = vadj->value;
   frame_buttom = frame_top + vadj->page_size;

   if (right > frame_right) {
      pos = right - (int) hadj->page_size;
      gtk_adjustment_set_value (hadj, pos);
   } else if (left < frame_left) {
      gtk_adjustment_set_value (hadj, left);
   }

   if (buttom > frame_buttom) {
      pos = buttom - (int) vadj->page_size;
      gtk_adjustment_set_value (vadj, pos);
   } else if (top < frame_top) {
      gtk_adjustment_set_value (vadj, top);
   }
}


gboolean
listview_set_selection (Thumbnail *thumb, gboolean select)
{
   ThumbnailData *thumb_data;

   g_return_val_if_fail (thumb, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (thumb_data, FALSE);

   thumb->selected = select;
   if (thumb->selected) {
      gtk_widget_set_state (thumb_data->widget, GTK_STATE_SELECTED);
   } else {
      gtk_widget_set_state (thumb_data->widget, GTK_STATE_NORMAL);
   }

   return TRUE;
}


static void
set_style (GtkWidget *widget, GtkStyle *style)
{
  gtk_widget_set_style (widget, style);

  if (GTK_IS_CONTAINER(widget))
    gtk_container_foreach (GTK_CONTAINER (widget),
			   GTK_SIGNAL_FUNC(set_style), style);
}


GtkWidget *
listview_create (ThumbView *tv, gint dest_mode)
{
   ThumbViewData *tv_data = NULL;
   GtkWidget *hbox;

   g_return_val_if_fail (tv, NULL);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data)
      tv_data = listview_new (tv);

   calc_col_row_num (tv, dest_mode, &tv_data->colnum, &tv_data->rownum);

   tv_data->eventbox = gtk_event_box_new ();
   tv_data->hbox = hbox = gtk_hbox_new (FALSE, 0);
   tv_data->table = gtk_table_new (tv_data->rownum, tv_data->colnum, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (tv_data->table), RAW_SPACE);
   gtk_table_set_col_spacings (GTK_TABLE (tv_data->table), COL_SPACE);
   gtk_container_set_border_width (GTK_CONTAINER (tv_data->table), BORDER_WIDTH);

   gtk_container_add (GTK_CONTAINER (tv_data->eventbox), hbox);
   gtk_box_pack_start (GTK_BOX (hbox), tv_data->table, FALSE, FALSE, 0);

   gtk_widget_show (tv_data->eventbox);
   gtk_widget_show (hbox);
   gtk_widget_show (tv_data->table);

   /* set callback */
   gtk_signal_connect (GTK_OBJECT (tv_data->eventbox), "expose_event",
		       GTK_SIGNAL_FUNC (cb_expose), tv);

   /* for drag and drop */
   dnd_dest_set (tv_data->eventbox, dnd_types, dnd_types_num);
   gtk_signal_connect(GTK_OBJECT (tv_data->eventbox), "drag_data_received",
		      GTK_SIGNAL_FUNC (thumbview_drag_data_received_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tv_data->eventbox), "drag_end",
		      GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);

   /* set icons */
   list_view_append_thumb_frames (tv, tv->thumblist, dest_mode);

   return tv_data->eventbox;
}
