/* glpspx.h */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#ifndef _GLPSPX_H
#define _GLPSPX_H

#include "glpinv.h"

#define spx_get_aj            _glp_spx_get_aj
#define spx_get_ak            _glp_spx_get_ak
#define spx_get_bi            _glp_spx_get_bi
#define spx_ini_basis         _glp_spx_ini_basis
#define spx_eval_xnj          _glp_spx_eval_xnj
#define spx_eval_bbar         _glp_spx_eval_bbar
#define spx_eval_zeta         _glp_spx_eval_zeta
#define spx_eval_row          _glp_spx_eval_row
#define spx_eval_pi           _glp_spx_eval_pi
#define spx_eval_col          _glp_spx_eval_col
#define spx_reset_gvec        _glp_spx_reset_gvec
#define spx_update_gvec       _glp_spx_update_gvec
#define spx_exact_gvec        _glp_spx_exact_gvec
#define spx_check_gvec        _glp_spx_check_gvec
#define spx_scale_mat         _glp_spx_scale_mat

#define spx_create_b          _glp_spx_create_b
#define spx_invert_b          _glp_spx_invert_b
#define spx_repair_b          _glp_spx_repair_b
#define spx_ftran             _glp_spx_ftran
#define spx_btran             _glp_spx_btran
#define spx_update_b          _glp_spx_update_b
#define spx_delete_b          _glp_spx_delete_b

typedef struct SPXMAT SPXMAT;
typedef struct SPXBAS SPXBAS;

struct SPXMAT
{     /* matrix of constraint coefficients; this matrix specifies the
         system of equality constraints xR = A*xS, where xR is a vector
         of auxiliary (logical) variables, xS is a vector of structural
         variables */
      int m;
      /* number of rows = number of auxiliary variables */
      int n;
      /* number of columns = number of structural variables */
      int (*column)(void *info, int j, int rn[], double aj[]);
      /* in order to obtain j-th column (1 <= j <= n) of the matrix A
         the simplex method routines call the routine column with the
         parameter j; in response this routine should store row indices
         and numerical values of non-zero elements of the j-th column of
         the matrix A to locations rn[1], ..., rn[cnt] and aj[1], ...,
         aj[cnt], respectively, where cnt is the number of non-zeros in
         the j-th column, which should be returned on exit; neither zero
         nor duplicate elements are allowed; the routine column may be
         called more than once for the same column number */
      void *info;
      /* a transit pointer passed to the routine column */
};

struct SPXBAS
{     /* representation of the basis matrix */
      /* currently this representation is based on LU-factorization */
      INV *inv;
      /* pointer to an invertable form of the basis matrix */
};

extern int spx_get_aj(SPXMAT *A, int j, int rn[], double aj[]);
/* get column of the constraint matrix */

extern int spx_get_ak(SPXMAT *A, int k, int rn[], double ak[]);
/* get column of the expanded constraint matrix */

extern int spx_get_bi(SPXMAT *A, int indb[], int i, int rn[],
      double bi[]);
/* get column of the basis matrix */

extern int spx_ini_basis(SPXMAT *A, int typx[], int indb[]);
/* build an initial basis */

extern double spx_eval_xnj(double lb[], double ub[], int indn[],
      int tagn[], int j);
/* determine value of non-basic variable */

extern void spx_eval_bbar(SPXMAT *A, SPXBAS *B, double lb[],
      double ub[], int indn[], int tagn[], double bbar[], int rn[],
      double ak[]);
/* compute values of basic variables */

extern void spx_eval_zeta(int m, SPXBAS *B, int i, double zeta[]);
/* compute row of the inverse */

extern void spx_eval_row(SPXMAT *A, int indn[], double zeta[],
      double ai[], int rn[], double ak[]);
/* compute row of the simplex table */

extern void spx_eval_pi(int m, SPXBAS *B, int indb[], double c[],
      double pi[]);
/* compute simplex multipliers */

extern void spx_eval_col(SPXMAT *A, SPXBAS *B, int indn[], int j,
      double aj[], int save, int rn[], double ak[]);
/* compute column of the simplex table */

extern void spx_reset_gvec(int m, int n, int indn[], int ref[],
      double gvec[]);
/* reset the reference space and the vector gamma */

extern void spx_update_gvec(SPXMAT *A, SPXBAS *B, int typx[],
      int ref[], int indb[], int indn[], int p, int q, double ap[],
      double aq[], double gvec[], int _rn[], double ak[], double w[]);
/* update the vector gamma for adjacent basis */

extern double spx_exact_gvec(SPXMAT *A, SPXBAS *B, int ref[],
      int indb[], int indn[], int j);
/* compute exact value of gamma[j] */

extern double spx_check_gvec(SPXMAT *A, SPXBAS *B, int typx[],
      int ref[], int indb[], int indn[], double gvec[]);
/* check accuracy of the vector gamma */

extern void spx_eval_cbar(SPXMAT *A, int indn[], double c[],
      double pi[], double cbar[], int _rn[], double _ak[]);
/* compute reduced costs of non-basic variables */

extern void spx_scale_mat(SPXMAT *A, int how, double R[], double S[]);
/* scale matrix of constraint coefficients */

extern SPXBAS *spx_create_b(int m);
/* create representation of the basis matrix */

extern int spx_invert_b(SPXBAS *B, SPXMAT *A, int indb[]);
/* compute representation of the basis matrix */

void spx_repair_b(SPXBAS *B, int indb[]);
/* repair the basis matrix */

extern void spx_ftran(SPXBAS *B, double z[], int save);
/* perform forward transformation */

extern void spx_btran(SPXBAS *B, double z[]);
/* perform backward transformation */

extern int spx_update_b(SPXBAS *B, int i);
/* update representation of the basis matrix */

extern void spx_delete_b(SPXBAS *B);
/* delete representation of the basis matrix */

#endif

/* eof */
