/**
 * GMyth Library
 * 
 * @file gmyth/gmyth_transcoder.c
 * 
 * @brief <p> This file contains the transcoder class.
 *
 * Copyright (C) 2007 INdT - Instituto Nokia de Tecnologia.
 * @author Artur Duque de Souza <artur.souza@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "gmyth_util.h"
#include "gmyth_debug.h"

#include "gmyth_jobqueue.h"
#include "gmyth_transcoder.h"

static void     gmyth_transcoder_class_init(GMythTranscoderClass * klass);
static void     gmyth_transcoder_init(GMythTranscoder * object);

static void     gmyth_transcoder_dispose(GObject * object);
static void     gmyth_transcoder_finalize(GObject * object);

G_DEFINE_TYPE(GMythTranscoder, gmyth_transcoder, G_TYPE_OBJECT)
    static void     gmyth_transcoder_class_init(GMythTranscoderClass *
                                                klass)
{
    GObjectClass   *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->dispose = gmyth_transcoder_dispose;
    gobject_class->finalize = gmyth_transcoder_finalize;
}

static void
gmyth_transcoder_init(GMythTranscoder * transcoder)
{
    transcoder->started = FALSE;
}

static void
gmyth_transcoder_dispose(GObject * object)
{
    GMythTranscoder *transcoder = GMYTH_TRANSCODER(object);

    g_free(transcoder->output_filename);
    g_free(transcoder->filename);
    g_free(transcoder->profile);
    g_free(transcoder->starttime);

    if (transcoder->backend_info)
        g_object_unref(transcoder->backend_info);

    G_OBJECT_CLASS(gmyth_transcoder_parent_class)->dispose(object);
}

static void
gmyth_transcoder_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);
    G_OBJECT_CLASS(gmyth_transcoder_parent_class)->finalize(object);
}

/**
 * Creates a new instance of GMythTranscoder.
 * 
 * @return a new instance of GMythTranscoder.
 **/
GMythTranscoder *
gmyth_transcoder_new(GMythBackendInfo * backend_info)
{
    GMythTranscoder *transcoder = GMYTH_TRANSCODER
        (g_object_new(GMYTH_TRANSCODER_TYPE, NULL));

    if (backend_info != NULL) {
        g_object_ref(backend_info);
        transcoder->backend_info = backend_info;
    }

    return transcoder;
}

/**
 *
 * gmyth_transcoder_date_change_format
 * @brief converts a string like YYYY-MM-DDTHH:MM:SS into YYYYMMDDHHMMSS (vice versa)
 * @param date_s gchar*
 * @return gchar* with file or iso format
 *
 **/
static gchar   *
gmyth_transcoder_date_change_format(gchar * date_s, int format)
{
    if (date_s != NULL) {
        gint            length = strlen(date_s);

        // create the right date format
        gchar          *src = (gchar *) g_malloc0(sizeof(gchar) * length);

        strncpy(src, date_s, length);

        gchar          *dst;

        if (format == DATE_FILE) {
            dst = (gchar *) g_malloc0(sizeof(gchar) * 16);
            snprintf(dst, 16, "%.4s%.2s%.2s%.2s%.2s%.2s", src, src + 5,
                     src + 7, src + 9, src + 11, src + 13);
            dst[15] = '\0';
        } else if (format == DATE_ISO) {
            dst = (gchar *) g_malloc0(sizeof(gchar) * 20);
            snprintf(dst, 20, "%.4s-%.2s-%.2sT%.2s:%.2s:%.2s", src,
                     src + 4, src + 6, src + 8, src + 10, src + 12);
            dst[19] = '\0';
        }

        gchar          *ret = g_strdup(dst);

        g_free(src);
        g_free(dst);

        return ret;
    } else
        return NULL;
}

/**
 * gmyth_transcoder_set_output
 * @brief set transcoder to use output
 * @param value gboolean
 * @param outfile filename of output
 * @return void set's up the var to value
 *
 **/
void
gmyth_transcoder_set_output(GMythTranscoder * transcoder,
                            gboolean value, const gchar * outputfile)
{
    transcoder->output = value;
    transcoder->output_filename = g_strdup(outputfile);
}

/**
 * gmyth_transcoder_set_file
 * @brief set the file to transcoder
 * @param file filename
 * @return void set's up the var to value
 *
 **/
void
gmyth_transcoder_set_filename(GMythTranscoder * transcoder,
                              const gchar * file)
{
    // fixme: if this method is called twice, memory will not be
    // dealocated
    // one transcoder can be used only for one file request?
    if (file != NULL) {
        gchar         **splited = g_strsplit(file, "_", 2);

        // Get chanid
        sscanf(splited[0], "%d", &(transcoder->chanid));

        // Get starttime
        gchar         **date = g_strsplit(splited[1], ".", 2);

        transcoder->starttime =
            gmyth_transcoder_date_change_format(date[0], DATE_ISO);

        transcoder->filename = g_strdup(file);
    }
}


/**
 *
 * gmyth_transcoder_set_profile
 * @brief set transcoder's profile
 * @param rec GMythTranscoder*
 * @param value the value
 * @return gint representing the result
 *
 **/
gint
gmyth_transcoder_set_profile(GMythTranscoder * trans, const gchar * value)
{
    g_return_val_if_fail(value != NULL, -1);

    trans->profile = g_strndup(value, strlen(value));

    return 0;
}

gboolean
gmyth_transcoder_start(GMythTranscoder * trans)
{
    g_return_val_if_fail(trans != NULL, FALSE);
    g_return_val_if_fail(trans->backend_info != NULL, FALSE);
    g_return_val_if_fail(trans->filename != NULL, FALSE);

    if (trans->started == FALSE) {  // not started yet
        if (!gmyth_util_file_exists(trans->backend_info, trans->filename)) {
            gmyth_debug("File %s does not exist", trans->filename);
        }
        trans->started = gmyth_jobqueue_add_job(trans, "JOB_TRANSCODE");
        if (trans->started == FALSE)
            gmyth_debug("Error while starting GMythTranscoder to file: %s",
                        trans->output_filename);
    } else {
        gmyth_debug("GMythTransfer already started!");
    }

    return trans->started;
}

gboolean
gmyth_transcoder_pause(GMythTranscoder * trans)
{
    g_return_val_if_fail(trans != NULL, FALSE);
    g_return_val_if_fail(trans->started == TRUE, FALSE);

    return gmyth_jobqueue_change_cmd(trans, "PAUSE", "JOB_TRANSCODE");
}

gboolean
gmyth_transcoder_resume(GMythTranscoder * trans)
{
    g_return_val_if_fail(trans != NULL, FALSE);

    return gmyth_jobqueue_change_cmd(trans, "RESUME", "JOB_TRANSCODE");
}

gboolean
gmyth_transcoder_cancel(GMythTranscoder * trans)
{
    g_return_val_if_fail(trans != NULL, FALSE);
    g_return_val_if_fail(trans->started == TRUE, FALSE);

    trans->started = FALSE;

    return gmyth_jobqueue_change_cmd(trans, "STOP", "JOB_TRANSCODE");
}

// fixme: implement this method
gint
gmyth_transcoder_get_progress(GMythTranscoder * trans)
{
    static int      fixme = 0;

    return (fixme++) % 101;
}
