/*
 * beroid.C: The implementation of the object identifier class
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later 
 * version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * See the AUTHORS file for a list of people who have hacked on 
 * this code. 
 * See the ChangeLog file for a list of changes.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <errno.h>

#include <queue>

#include "ber.h"

/* berOid - the encoding of an object identifier. The encoding of 
   this is a real hassel. First of all you have the same anoying 
   problem with length as you do with the strings and sequence. 
   Then you have to take the first two oids multiply the first 
   one by 40 and add the second one. Then you can go through the 
   rest of the oids and encode them however they are not encoded 
   as simple integers or anything they are big endian two's 
   compliment stuffed into as few bytes as possible. The way that
   you stuff them into as few bytes as possible and deal with the
   fact that that you don't know how many bytes each suboid is 
   going to be is to use the MSB to indicate if there are more 
   bits to follow. Thus you can only use the 7 LSB to represent 
   the data. So a normal long can end up being up to 5 bytes long.
 */

BerOid::BerOid(unsigned char *str)
  throw(BerOidTagException,BerLengthException){
  if(str[0]!=OID_TAG) throw BerOidTagException();
  unsigned char headerlen;
  unsigned long len=unpack_len(str,headerlen);
  encoded_oid = ustring(str+headerlen,len);
}

ustring &BerOid::encode(ustring &dest){
  start_data(OID_TAG,encoded_oid.length(),dest);
  dest+=encoded_oid;
  return dest;
}

int unpack_suboid(ustring::iterator &buf){
  /* The reason why this is sizeof(long)+1 is because when you 
     expand the bits within a 32 bit number the way that you have 
     to do for ASN.1 you can get 5 bytes worth of data. */
  unsigned char len;
  for(len=0;len<sizeof(long)+1;len++)
    if(*(buf+len)<128)
      break;
  // len now equals number of bytes-1
  long val=0;
  /* this is an amazingly clever bit of code (if I do say so 
     myself) and therefore requires a comment. (i.e. I wrote it 
     more than a year ago and then I had to spend half an hour 
     looking at it to figure out how it worked.) ;-) What it does 
     is compress the bits back together. The key to understanding 
     it is noting that there are no breaks within the switch 
     statement and that the iterator is incremented instead. That 
     way it moves through buf a byte at a time arranging the bits 
     closer together. */
  switch(len){
  case 4:
    val|=(*buf&0x7f)<<28;
    buf++;
  case 3:
    val|=(*buf&0x7f)<<21;
    buf++;
  case 2:
    val|=(*buf&0x7f)<<14;
    buf++;
  case 1:
    val|=(*buf&0x7f)<<7;
    buf++;
  default:
    val|=*buf;
  }
  buf++; //now equals the byte after this suboid
  return val;
}

void pack_suboid(long suboid, ustring &dest){
  unsigned char buf[sizeof(long)+1];
  unsigned char len=0;
  // spread out the bits
  int i;
  for(i=sizeof(long);i>=0;i--)
    buf[i]=(suboid>>7*i)&0x7f;

  // mark the bits
  for(i=sizeof(long)-1;i>=0;i--)
    if(buf[i]>0) {
      for(char j=i;j>0;j--)
	buf[j]|=0x80;
      len=i+1;
      break;
    }

  if(len==0) // deal with the oid being 0
    len=1;

  for(i=len-1;i>=0;i--)
    dest+=buf[i];
}         

BerOid::BerOid(const std::string &str)
  throw(BerOidBadSubOidException,BerNoOidsException){
  // turn the string into a list of values
  std::queue<long> oids;
  for(std::string cur=str;!cur.empty();){
    size_t end=cur.find('.');
    std::string numstr=cur.substr(0,end);
    long val=strtol(numstr.c_str(),NULL,10);    
    if(val==LONG_MAX && errno==ERANGE) throw BerOidBadSubOidException();
    oids.push(val);
    cur=(end==std::string::npos)?"":cur.substr(end+1,cur.length()-end);
  }
  if(oids.empty()) throw BerNoOidsException();

  long val=oids.front();
  oids.pop();
  if(oids.empty()){    // only one suboid
    pack_suboid(val*40,encoded_oid);
    return;
  } 

  long val2=oids.front();
  oids.pop();
  pack_suboid(val*40+val2,encoded_oid);

  while(!oids.empty()){
    val=oids.front();
    oids.pop();
    pack_suboid(val,encoded_oid);
  }
}

/* I think there is a design bug in this that I don't know how to 
   work around. In the boundry case where there is only one 
   suboid in an oid then it is impossible to tell the difference 
   between the case there is only one oid and the case where 
   second oid is exists and is zero. However, I do not believe 
   that there are any cases where SNMP would use an oid which 
   contains only one suboid. In addition the ascii_print function 
   is only used for creating human readable output at the moment 
   and so this shouldn't cause many problems. */
void BerOid::ascii_print(std::string &str){
  char buf[60];
  ustring::iterator cur=encoded_oid.begin();
  int val=unpack_suboid(cur);
  snprintf(buf,60,"%d.%d",val/40,val%40);
  str+=buf;
  while(cur!=encoded_oid.end()){
    val=unpack_suboid(cur);
    snprintf(buf,60,".%d",val);
    str+=buf;
  }
}
