/* vi:set ts=8 sts=0 sw=8:
 * $Id: win.c,v 1.5 2001/01/27 17:32:39 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <config.h>
#include <string.h>
#include <ctype.h>
#include "win.h"
#include "main.h"
#include "misc.h"
#include "msgbar.h"
#include "results.h"
#include "advopts.h"
#include "multiple.h"
#include "prefs.h"
#include "search.h"
#include "pathmax.h"


static void about_cb(GtkWidget *wgt, gpointer cbdata);
static int win_main_del_event_cb(GtkWidget *, GdkEvent *ev, gpointer cbdata);
static void win_new_cb(GtkWidget *wgt, gpointer cbdata);
static void win_menus_setup(GtkWidget *menu, int wgt_id);
static void search_page_cb(GtkWidget *wgt, gpointer cbdata);
static void locate_page_cb(GtkWidget *wgt, gpointer cbdata);
static void switch_page_cb(GtkWidget *, GtkNotebookPage *, int, gpointer);

static char *cb_signal_names[] = {
	"delete_event",	/* Ef_wgt_toplev */
	NULL,		/* Ef_wgt_dialog */
	"clicked",	/* Ef_wgt_button */
	"clicked",	/* Ef_wgt_checkbutton */
	NULL,		/* Ef_wgt_radiobutton */
	"changed",	/* Ef_wgt_entry */
	NULL,		/* Ef_wgt_combo */
	"select_row",	/* Ef_wgt_clist */
	"tree_select_row",	/* Ef_wgt_ctree */
	NULL,		/* Ef_wgt_list */
	NULL,		/* Ef_wgt_vbox */
	NULL,		/* Ef_wgt_table */
	NULL,		/* Ef_wgt_frame */
	"switch_page",	/* Ef_wgt_notebook */
	"changed",	/* Ef_wgt_spinbutton */
	"add",		/* Ef_wgt_optmenu */
	NULL,		/* Ef_wgt_toolbar */
	NULL,		/* Ef_wgt_label */
	NULL,		/* Ef_wgt_menu */
	"activate",	/* Ef_wgt_menuitem */
	NULL,		/* Ef_wgt_dont_care */
};

static ef_t ef_main_widgets[] = {
	{
		PACKAGE,
		Ef_main_toplev,
		Ef_wgt_toplev,
		GTK_SIGNAL_FUNC(win_main_del_event_cb)
	},
	{
		"main_notebook",
		Ef_main_notebook,
		Ef_wgt_notebook,
	},
	{
		"file1_menu",
		Ef_main_filemenu,
		Ef_wgt_menu,
	},
	{
		"settings1_menu",
		Ef_main_settingsmenu,
		Ef_wgt_menu,
	},
	{
		"view1_menu",
		Ef_main_viewmenu,
		Ef_wgt_menu,
	},
	{
		"help1_menu",
		Ef_main_helpmenu,
		Ef_wgt_menu,
	},
	{
		"msgbar",
		Ef_main_msgbar,		/* don't know what it is, so just */
		Ef_wgt_dont_care,	/* ignore it */
	},
	{
		"menuitem_search_new",
		Ef_main_file_new,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(search_clear_cb)
	},
	{
		"menuitem_quit",
		Ef_main_quit,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(ef_quit_cb)
	},
	{
		"menuitem_prefs",
		Ef_main_menu_prefs,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(prefs_cb)
	},
	{
		"menuitem_search_page",
		Ef_main_search_menu,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(search_page_cb)
	},
	{
		"menuitem_locate_page",
		Ef_main_locate_menu,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(locate_page_cb)
	},
	{
		"menuitem_win_new",
		Ef_main_win_new,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(win_new_cb)
	},
	{
		"menuitem_win_close",
		Ef_main_win_close,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(win_close_cb)
	},
	{
		"menuitem_search_prev",
		Ef_main_search_prev,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(search_prev_cb)
	},
	{
		"menuitem_cmd_prev",
		Ef_main_cmd_prev,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(search_cmd_prev_cb)
	},
	{
		"menuitem_about",
		Ef_main_about,
		Ef_wgt_menuitem,
		GTK_SIGNAL_FUNC(about_cb)
	},
	{
		"chkbut_adv_search",
		Ef_main_adv_search,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(search_adv_cb)
	},
	{
		"but_show_opts",
		Ef_main_show_adv_opts,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(advopts_cb)
	},
	{
		"frame_params",
		Ef_main_frame_params,
		Ef_wgt_frame
	},

	/* main toolbar */
	{
		"main_toolbar",
		Ef_main_toolbar,
		Ef_wgt_toolbar
	},
	{
		"tb_clear1",
		Ef_main_clear,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(search_clear_cb)
	},
	{
		"tb_search1",
		Ef_main_search,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(search_begin_cb)
	},
	{
		"tb_stop1",
		Ef_main_stop,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(search_stop_cb)
	},
	{
		"tb_manual1",
		Ef_main_manual,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(search_manual_cb)
	},
	{
		"tb_prefs1",
		Ef_main_prefs,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(prefs_cb)
	},
	{
		"tb_close1",
		Ef_main_close,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(win_close_cb)
	},

	/* "Basic" search page */
	{
		"entry_find",
		Ef_main_find_text,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(search_entry_file_dir_cb)
	},
	{
		"gnoentry_find",
		Ef_main_find_combo,
		Ef_wgt_dont_care,
	},
	{
		"but_mult_files",
		Ef_main_mult_files,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(mult_files_cb)
	},
	{
		"entry_directory",
		Ef_main_dir_text,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(search_entry_file_dir_cb)
	},
	{
		"fileentry_directory",
		Ef_main_dir_combo,
		Ef_wgt_dont_care,
	},
	{
		"but_mult_dirs",
		Ef_main_mult_dirs,
		Ef_wgt_button,
		GTK_SIGNAL_FUNC(mult_dirs_cb)
	},
	{
		"chkbut_search_subdir",
		Ef_main_search_subdir,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(search_subdir_cb)
	},
	{
		"chkbut_case_sens",
		Ef_main_case_sens,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(search_case_sens_cb)
	},
	{
		"chkbut_partial_fname",
		Ef_main_partial_fname,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(search_partial_fname_cb)
	},
	{
		"chkbut_regex_fname",
		Ef_main_regex_fname,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(search_regex_fname_cb)
	},

	/* Locate page */
	{
		"entry_locate",
		Ef_main_locate_text,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(search_entry_file_dir_cb)
	},
	{
		"entry_locate_db",
		Ef_main_locate_db,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(search_entry_file_dir_cb)
	},

	{ NULL }
}; /* ef_main_widgets[] */


/*
 * "About" menuitem callback.
 */
static void
about_cb(GtkWidget *wgt, gpointer cbdata)
{
	GladeXML *xml;
	GtkWidget *dlg;
	win_t *w = (win_t *)cbdata;

	xml = my_glade_xml_get("About");
	g_assert(xml);
	dlg = my_glade_widget_get(xml, "About", GLADE_MODAL);
	gnome_dialog_set_parent(GNOME_DIALOG(dlg),
				GTK_WINDOW(win_main_toplev(w)));
	gtk_object_destroy(GTK_OBJECT(xml));
	gtk_widget_show(dlg);
} /* about_cb */


static void
win_prefs_save(win_t *w)
{
	int width, height;
	GtkWidget *toplev;
	
	toplev = win_main_toplev(w);
	gdk_window_get_size(toplev->window, &width, &height);
	prefs_int_set(Prefs_main_win_w, width);
	prefs_int_set(Prefs_main_win_h, height);
} /* win_prefs_save */


/*
 * Returns TRUE if window was closed.  FALSE if close is delayed.
 */
static gboolean
win_close_common(win_t *w)
{
	w->close_window = TRUE;
	w->stop_search = TRUE;

	if (w->is_searching)
		return FALSE;

	gtk_signal_disconnect_by_func(GTK_OBJECT(w->mainwgts[Ef_main_notebook]),
				      GTK_SIGNAL_FUNC(switch_page_cb), w);

	mult_prefs_save(w);
	win_prefs_save(w);
	results_prefs_save(w);
	advopts_prefs_save(w);
	prefs_save(w);

	results_dialog_destroy(w);
	search_reset(w);
	gtk_widget_destroy(win_main_toplev(w));
	ef_win_remove(w);
	return TRUE;
} /* win_close_common */


/*
 * Callback for "close window" menuitem and toolbar button.
 */
void
win_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	(void)win_close_common(cbdata);
} /* win_close_cb */


/*
 * Callback for "delete_event" signal on main window.
 */
static int
win_main_del_event_cb(GtkWidget *wgt, GdkEvent *event, gpointer cbdata)
{
	return (!win_close_common(cbdata));
} /* win_main_del_event_cb */


/*
 * Callback for "new window" menuitem.
 */
static void
win_new_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w;

	w = win_new();
	ef_win_add(w);
} /* win_new_cb */


static void
win_widget_failed(ef_t *efp)
{
#ifdef ENABLE_DEBUG
	g_error("Widget failed (%s, %d, %d)\n",
		efp->glname, efp->wgt_id, efp->wgt_type);
#else
	g_warning("Widget failed (%s, %d, %d)\n",
		  efp->glname, efp->wgt_id, efp->wgt_type);
#endif
} /* win_widget_failed */


void
win_widgets_setup(GladeXML *xml, win_t *w, GtkWidget **wgts, ef_t *efp)
{
	GtkWidget *wgt;

	if ((wgt = my_glade_widget_get(xml, efp->glname, 0)) == NULL)
		return;

	switch (efp->wgt_type) {
	case Ef_wgt_toplev:
	case Ef_wgt_dialog:
		if (!GTK_IS_WINDOW(wgt))
			win_widget_failed(efp);
		else
			gtk_widget_queue_resize(wgt);
		break;
	case Ef_wgt_button:
		if (!GTK_IS_BUTTON(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_checkbutton:
		if (!GTK_IS_CHECK_BUTTON(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_radiobutton:
		if (!GTK_IS_RADIO_BUTTON(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_entry:
		if (!GTK_IS_ENTRY(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_combo:
		if (!GTK_IS_COMBO(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_clist:
	case Ef_wgt_ctree:
		if (!GTK_IS_CLIST(wgt) && !GTK_IS_CTREE(wgt))
			win_widget_failed(efp);
		else
			gtk_clist_column_titles_passive(GTK_CLIST(wgt));
		break;
	case Ef_wgt_list:
		if (!GTK_IS_LIST(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_vbox:
		if (!GTK_IS_VBOX(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_table:
		if (!GTK_IS_TABLE(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_frame:
		if (!GTK_IS_FRAME(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_notebook:
		if (!GTK_IS_NOTEBOOK(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_spinbutton:
		if (!GTK_IS_SPIN_BUTTON(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_optmenu:
		if (!GTK_IS_OPTION_MENU(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_label:
		if (!GTK_IS_LABEL(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_menu:
		if (!GTK_IS_MENU(wgt))
			win_widget_failed(efp);
		else
			win_menus_setup(wgt, efp->wgt_id);
		break;
	case Ef_wgt_menuitem:
		if (!GTK_IS_MENU_ITEM(wgt))
			win_widget_failed(efp);
		else if (efp->wgt_id == Ef_main_file_new)
			gtk_widget_set_sensitive(wgt, FALSE);
		break;
	case Ef_wgt_toolbar:
		if (!GTK_IS_TOOLBAR(wgt))
			win_widget_failed(efp);
		break;
	case Ef_wgt_dont_care:
		break;
	default:
#ifdef ENABLE_DEBUG
		g_error("Unknown wgt_type (n=%s, wgt_id=%d, t=%d)\n",
			efp->glname, efp->wgt_id, efp->wgt_type);
#else
		g_warning("Unknown wgt_type (n=%s, wgt_id=%d, t=%d)\n",
			  efp->glname, efp->wgt_id, efp->wgt_type);
#endif
	}

	if (efp->cb && cb_signal_names[efp->wgt_type])
		gtk_signal_connect(GTK_OBJECT(wgt),
				   cb_signal_names[efp->wgt_type],
				   GTK_SIGNAL_FUNC(efp->cb), w);

	wgts[efp->wgt_id] = wgt;
} /* win_widgets_setup */


static gboolean
win_key_press(GtkWidget *txt, GdkEventKey *event, win_t *w)
{
	char *findtxt, *dirtxt;
	int len1, len2, pagenum;

	pagenum = gtk_notebook_get_current_page(
				GTK_NOTEBOOK(w->mainwgts[Ef_main_notebook]));

	switch (event->keyval) {
	case GDK_Return:
		if (pagenum == 0) {
			g_assert(GTK_IS_ENTRY(w->mainwgts[Ef_main_find_text]));
			findtxt = gtk_entry_get_text(
				     GTK_ENTRY(w->mainwgts[Ef_main_find_text]));
			if (findtxt)
				len1 = strlen(findtxt);
			else
				return FALSE;

			g_assert(GTK_IS_ENTRY(w->mainwgts[Ef_main_dir_text]));
			dirtxt = gtk_entry_get_text(
				      GTK_ENTRY(w->mainwgts[Ef_main_dir_text]));
			if (dirtxt)
				len2 = strlen(dirtxt);
			else
				return FALSE;


			if ((GTK_WIDGET_HAS_FOCUS(
					w->mainwgts[Ef_main_find_text]) ||
			     GTK_WIDGET_HAS_FOCUS(
			     		w->mainwgts[Ef_main_dir_text])) &&
			    len1 > 0 && len2 > 0) {

				gtk_signal_emit_by_name(
					GTK_OBJECT(w->mainwgts[Ef_main_search]),
					"clicked");
				return TRUE;
			}
		} else {
			findtxt = gtk_entry_get_text(
				   GTK_ENTRY(w->mainwgts[Ef_main_locate_text]));
			if (findtxt && strlen(findtxt)) {
				gtk_signal_emit_by_name(
					GTK_OBJECT(w->mainwgts[Ef_main_search]),
					"clicked");
				return TRUE;
			}
			return FALSE;
		}

		break;
#if 0
	case GDK_KP_Enter:
	case GDK_ISO_Enter:
#endif
	case GDK_Escape:
		if (pagenum == 0 && w->is_searching) {
			w->stop_search = TRUE;
			return TRUE;
		}
		break;
	}

	return FALSE;
} /* win_key_press */


void
win_progbar_update(win_t *w)
{
	float val;

	if (w->close_window || w->stop_search)
		return;

	val = gtk_progress_get_value(w->progbar) + 0.1;
	if (val > 100)
		val = 0;
	gtk_progress_set_value(w->progbar, val);
} /* win_progbar_update */


static void
win_nb_tab_set(win_t *w)
{
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->mainwgts[Ef_main_notebook]),
				   prefs_bool_get(Prefs_show_tabs));
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->mainwgts[Ef_main_notebook]),
				 prefs_int_get(Prefs_tab_pos));
} /* win_nb_tab_set */


/*
 * Adjust window attributes according to current preferences values.
 */
void
win_prefs_set(win_t *w)
{
	prefs_toolbar_set(Prefs_main_tbstyle, w->mainwgts[Ef_main_toolbar]);
	win_nb_tab_set(w);
	results_prefs_toolbar_set(w);
	advopts_prefs_toolbar_set(w);
	mult_prefs_toolbar_set(w);

	my_prefs_window_set_size(win_main_toplev(w),
				 Prefs_save_main_win,
				 Prefs_main_win_w,
				 Prefs_main_win_h);
} /* win_prefs_set */


win_t *
win_new(void)
{
	GtkWidget *toplev;
	GtkStyle *style;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	win_t *w;
	char *fname;
	int num;

	w = g_new0(win_t, 1);
	w->advopts = AO_WPRINT;

	num = (sizeof(ef_main_widgets) / sizeof(ef_main_widgets[0])) - 1;
	w->mainwgts = my_widgets_setup(w, w->mainwgts, num, ef_main_widgets,
				       PACKAGE, Ef_main_MAX, Ef_main_MAX,
				       FALSE);
	toplev = win_main_toplev(w);
	gtk_signal_connect(GTK_OBJECT(toplev), "key_press_event",
			   GTK_SIGNAL_FUNC(win_key_press), w);
	w->msgbar = w->mainwgts[Ef_main_msgbar];
	w->progbar = gnome_appbar_get_progress(GNOME_APPBAR(w->msgbar));

	/* notebook settings */
	gtk_signal_connect_after(GTK_OBJECT(w->mainwgts[Ef_main_notebook]),
				 "switch_page",
				 GTK_SIGNAL_FUNC(switch_page_cb), w);

#if 0
	gtk_button_set_relief(GTK_BUTTON(w->mainwgts[Ef_main_mult_files]),
			      GTK_RELIEF_NONE);
	gtk_button_set_relief(GTK_BUTTON(w->mainwgts[Ef_main_mult_dirs]),
			      GTK_RELIEF_NONE);
#endif

	/* set attributes according to preferences */
	win_prefs_set(w);

	/*
	 * realize the window now to get the GdkWindow so we can get the
	 * correct style.  We'll gtk_widget_show() it later, after we're done
	 * creating the pixmap widgets.
	 */
	gtk_widget_realize(toplev);

	style = gtk_widget_get_style(toplev);

	/* set the minimized window icon */
	fname = g_strdup("./gnome-find.xpm");
	if (!g_file_exists(fname)) {
		g_free(fname);
		fname = g_concat_dir_and_file(DATADIR, "gnome-find.xpm");
	}
	if (g_file_exists(fname)) {
		pixmap = gdk_pixmap_create_from_xpm(
					toplev->window,
					&mask,
					&style->bg[GTK_STATE_NORMAL],
					fname);
		if (pixmap)
			gdk_window_set_icon(toplev->window, NULL, pixmap, mask);
	}
	g_free(fname);

	/*
	 * At one point, glade did this for me.  But suddenly, it one day
	 * decided not to.  So for now, create a pixmap widget and put it into
	 * the buttons ourselves.
	 */
	fname = g_strdup("./multiple.xpm");
	if (!g_file_exists(fname)) {
		g_free(fname);
		fname = g_concat_dir_and_file(DATADIR, "multiple.xpm");
	}
	if (g_file_exists(fname)) {
		GtkWidget *pixmapwgt;
		pixmap = gdk_pixmap_create_from_xpm(
					toplev->window,
					&mask,
					&style->bg[GTK_STATE_NORMAL],
					fname);
		if (pixmap) {
			pixmapwgt = gtk_pixmap_new(pixmap, mask);
			gtk_widget_show(pixmapwgt);
			gtk_container_add(
				GTK_CONTAINER(w->mainwgts[Ef_main_mult_files]),
				pixmapwgt);
			pixmapwgt = gtk_pixmap_new(pixmap, mask);
			gtk_widget_show(pixmapwgt);
			gtk_container_add(
				GTK_CONTAINER(w->mainwgts[Ef_main_mult_dirs]),
				pixmapwgt);
		}
	}
	g_free(fname);

	gtk_widget_show(toplev);

	/* set default options */
	search_reset(w);

	return w;
} /* win_new */


void
win_wprint(win_t *w, char *path, struct stat *sb, struct predicate *pred)
{
	results_add(w, path, NULL, 0, sb);
} /* win_wprint */


GtkWidget *
win_main_toplev(win_t *w)
{
	return (w->mainwgts ? w->mainwgts[Ef_main_toplev] : NULL);
} /* win_main_toplev */


/*
 * Sets sensitivity of either find entry box, or the directory entry box.
 */
void
win_fd_entry_sensitive(win_t *w, gboolean sensitive, gboolean is_dir)
{
	GtkWidget *wgt;

	wgt = is_dir ? w->mainwgts[Ef_main_dir_combo] :
		       w->mainwgts[Ef_main_find_combo];

	gtk_widget_set_sensitive(wgt, sensitive);
} /* win_fd_entry_sensitive */


/*
 * Sets the entry to "<Multiple %s entered>" or clears it.
 */
void
win_fd_entry_setclear(win_t *w, gboolean clear, gboolean is_dir)
{
	GtkWidget *wgt;
	char *txt;

	wgt = is_dir ? w->mainwgts[Ef_main_dir_text] :
		       w->mainwgts[Ef_main_find_text];

	if (clear) {
		txt = gtk_entry_get_text(GTK_ENTRY(wgt));
		gtk_editable_delete_text(GTK_EDITABLE(wgt), 0, strlen(txt));
		gtk_widget_set_sensitive(wgt, TRUE);
	} else {
		txt = g_strdup_printf(_("<Multiple %s entered>"),
				      is_dir ? _("directories") : _("files"));
		gtk_entry_set_text(GTK_ENTRY(wgt), txt);
		g_free(txt);
	}
} /* win_fd_entry_setclear */


/*
 * All of the help menu related code below was taken from the file
 * libgnomeui/gnome-app-helper.c, in gnome-libs.
 */

/**
 * help_display_cb:
 * @ignore: value of this is ignored.  To simplify hooking into clicked
 *          signals
 * @ref: A GnomeHelpMenuEntry.
 * 
 * Cause a help viewer to display help entry defined in @ref.
*/
static void
help_display_cb(void *ignore, GnomeHelpMenuEntry *ref)
{
	char *file, *url;

	g_assert(ref);
	g_assert(ref->path);
	g_assert(ref->name);

	file = gnome_help_file_path(ref->name, ref->path);

	if (!file)
		return;

	url = g_strdup_printf("ghelp:%s", file);
	gnome_help_goto(ignore, url);
	g_free(url);
	g_free(file);
} /* help_display_cb */


static void
help_menu_item_cb(GtkWidget *widget, gpointer data)
{
	GnomeHelpMenuEntry *entry = (GnomeHelpMenuEntry *)data;

	g_free(entry->name);
	g_free(entry->path);
	g_free(entry);
} /* help_menu_item_cb */


/* Creates  a menu item label. It will also return the underlined 
 * letter's keyval if keyval is not NULL. */
static GtkWidget *
help_create_label(char *label_text, guint *keyval)
{
	guint kv;
	GtkWidget *label;

	label = gtk_accel_label_new (label_text);

	kv = gtk_label_parse_uline (GTK_LABEL (label), label_text);
	if (keyval)
		*keyval = kv;

	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_widget_show (label);

	return label;
} /* help_create_label */


/* Creates the accelerators for the underlined letter in a menu item's label. 
 * The keyval is what gtk_label_parse_uline() returned.  If accel_group is not 
 * NULL, then the keyval will be put with MOD1 as modifier in it (i.e. for 
 * Alt-F in the _File menu).
 */
static void
help_setup_uline_accel(GtkMenuShell  *menu_shell,
		       GtkAccelGroup *accel_group,
		       GtkWidget     *menu_item,
		       guint          keyval)
{
	if (keyval != GDK_VoidSymbol) {
		if (GTK_IS_MENU(menu_shell))
			gtk_widget_add_accelerator(
				menu_item, "activate_item",
				gtk_menu_ensure_uline_accel_group(
							GTK_MENU(menu_shell)),
				keyval, 0, 0);
		if (GTK_IS_MENU_BAR(menu_shell) && accel_group)
			gtk_widget_add_accelerator(
				menu_item, "activate_item", 
				accel_group, keyval, GDK_MOD1_MASK, 0);
	}
} /* help_setup_uline_accel */


/*
 * - grabbed from create_help_entries(), in libgnomeui/gnome-app-helper.c
 * - menu_shell is just GTK_MENU_SHELL(gtkmenu)
 */
static void
my_gnome_help_menus_create(GtkMenuShell *menu_shell, gint pos)
{
	char *buf;
	char *topic_file;
	char *s;
	FILE *fp;
	GnomeHelpMenuEntry *entry;
	GtkWidget *item;
	GtkWidget *label;
	guint keyval;

	topic_file = gnome_help_file_find_file(PACKAGE, "topic.dat");
	if (!topic_file || ((fp = fopen(topic_file, "rt")) == NULL)) {
#ifdef ENABLE_DEBUG
		g_print(_("could not open '%s' (%s)\n"),
			topic_file ? topic_file : "NULL" , g_strerror(errno));
#endif
		g_free(topic_file);
		return;
	}
	g_free(topic_file);

	buf = g_new(char, PATH_MAX * 2);
	while ((fgets(buf, (PATH_MAX * 2) - 1, fp))) {
		/* Format of lines is "help_file_name whitespace* menu_title" */

		for (s = buf; *s && !isspace(*s); s++)
			;

		*s++ = '\0';

		for (; *s && isspace(*s); s++)
			;

		if (s[strlen(s) - 1] == '\n')
			s[strlen(s) - 1] = '\0';

		/* Create help menu entry */
		entry = g_new(GnomeHelpMenuEntry, 1);
		entry->name = g_strdup(PACKAGE);
		entry->path = g_strdup(buf);

		item = gtk_menu_item_new();
		label = help_create_label(s, &keyval);
		gtk_container_add(GTK_CONTAINER(item), label);
		help_setup_uline_accel(menu_shell, NULL, item, keyval);
		gtk_widget_lock_accelerators(item);

		gtk_signal_connect(GTK_OBJECT(item), "activate",
				   GTK_SIGNAL_FUNC(help_display_cb), entry);
		gtk_signal_connect(GTK_OBJECT(item), "destroy",
				   GTK_SIGNAL_FUNC(help_menu_item_cb), entry);

		gtk_menu_shell_insert(menu_shell, item, pos);
		pos++;

		gtk_widget_show(item);
	}
	g_free(buf);
} /* my_gnome_help_menus_create */


static void
win_menus_setup(GtkWidget *menu, int wgt_id)
{
	GtkWidget *mitem;

	if (wgt_id == Ef_main_helpmenu)
		my_gnome_help_menus_create(GTK_MENU_SHELL(menu), 0);

	mitem = gtk_tearoff_menu_item_new();
	gtk_menu_prepend(GTK_MENU(menu), mitem);
	gtk_widget_show(mitem);
} /* win_menus_setup */


static void
search_locate_page_common(win_t *w, int num)
{
	gtk_notebook_set_page(GTK_NOTEBOOK(w->mainwgts[Ef_main_notebook]), num);
} /* search_locate_page_common */


static void
search_page_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_locate_page_common(cbdata, 0);
} /* search_page_cb */


static void
locate_page_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_locate_page_common(cbdata, 1);
} /* locate_page_cb */


static void
switch_page_cb(GtkWidget *wgt, GtkNotebookPage *page, int num, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	search_entry_file_dir_cb(NULL, cbdata);
	if (num == 0) {
		if (!GTK_WIDGET_HAS_FOCUS(w->mainwgts[Ef_main_find_text]) &&
		    !GTK_WIDGET_HAS_FOCUS(w->mainwgts[Ef_main_dir_text])) {

			gtk_widget_grab_focus(w->mainwgts[Ef_main_find_text]);
		}
	} else {
		gtk_widget_grab_focus(w->mainwgts[Ef_main_locate_text]);
	}
} /* switch_page_cb */


/* the end */
