/* think.c - most of think's functionality, at this point.
   Copyright (C) 1999 Peter Teichman

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 03111-1307, 
   USA.  */

#include <config.h>

#include "think.h"
#include "outline.h"
#include <gnome.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include "think-outline.h"

static void new_tree (GtkWidget *widget, gpointer data);
static void file_ok_sel (GtkWidget *w, GtkFileSelection *fs);
static void save_as_tree_dialog (GtkWidget *widget, gpointer callback_data);
static void save_tree_dialog (GtkWidget *widget, gpointer callback_data);
static void think_exit (GtkWidget *widget, gpointer callback_data);
static void file_save_sel (GtkWidget *w, GtkFileSelection *fs);
static void edit_selected (GtkWidget *widget, gpointer data);
static GtkWidget *create_item_popup (ThinkOutlineContent *content);
static void edit_content (GtkWidget *widget, ThinkOutlineContent *content);
static void remove_content (GtkWidget *widget, ThinkOutlineContent *content);

static void insert_child (GtkWidget *widget, ThinkOutlineContent *content);
static void append_child (GtkWidget *widget, ThinkOutlineContent *content);
static void insert_sibling (GtkWidget *widget, ThinkOutlineContent *content);
static void append_sibling (GtkWidget *widget, ThinkOutlineContent *content);

static void free_think_tree (GNode *node, gpointer data);

static void outline_changed (void);
static void outline_clean (void);

static void change_wrapper (ThinkOutlineContent *content, 
			    ThinkOutlineContent *dragged, gpointer data);

static GNode *tree_root = NULL;
static ThinkOutline *outline = NULL;
static GtkWidget *appbar = NULL;
static gchar *outline_filename = NULL;
static gint changed = 0;
static gint making_new = 0; /* horrible, horrible hack */

static GtkWidget *main_window = NULL;

/* popup code stolen shamelessly from gnumeric */
typedef enum {
	IT_ALWAYS,
	IT_SEPARATOR,
} item_popup_types;

static struct {
	char   *name;
	void   (*fn)(GtkWidget *widget, ThinkOutlineContent *content);
	item_popup_types type;
} item_popup_menu [] = {
	{ N_("Edit node"), edit_content, IT_ALWAYS },
	{ "", NULL, IT_SEPARATOR },
	{ N_("Insert sibling"), insert_sibling },
	{ N_("Append sibling"), append_sibling },
	{ N_("Insert child"), insert_child },
	{ N_("Append child"), append_child },
	{ "", NULL, IT_SEPARATOR },
	{ N_("Remove node"), remove_content, IT_ALWAYS },
	{ NULL,       NULL }
};

ThinkNode *
think_node_new (void)
{
	ThinkNode *node = g_malloc (sizeof (ThinkNode));
	node->title = NULL;
	node->text = NULL;
	node->is_todo = 0;
	node->is_done = 0;

	return node;
}

static GtkWidget *
create_item_popup (ThinkOutlineContent *content)
{
	GtkWidget *menu, *item;
	int i;

	menu = gtk_menu_new ();
	item = NULL;
  
	for (i = 0; item_popup_menu [i].name; i++){
		switch (item_popup_menu [i].type){
		case IT_SEPARATOR:
			if (item_popup_menu [i].type == IT_SEPARATOR)
				item = gtk_menu_item_new ();
			break;

		case IT_ALWAYS:
			item = gtk_menu_item_new_with_label (_(item_popup_menu [i].name));
			gtk_signal_connect (GTK_OBJECT (item), "activate",
					    GTK_SIGNAL_FUNC (item_popup_menu [i].fn),
					    content);
			break;
		}

		gtk_widget_show (item);
		gtk_menu_append (GTK_MENU (menu), item);
	}

	return menu;
}

static gint
outline_item_event (GtkObject *item, GdkEvent *event, gpointer data)
{
	ThinkOutlineContent *content = THINK_OUTLINE_CONTENT (data);

	if (event->type == GDK_BUTTON_PRESS){
		if (event->button.button == 3){
			GtkWidget *menu;
			think_outline_select (outline, content);

			menu = create_item_popup (content);
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 3, 
					event->button.time);

			return TRUE;
		}
	}

	if (event->type == GDK_2BUTTON_PRESS){
		think_outline_select (outline, content);
		edit_selected (NULL, NULL);
		return TRUE;
	}

	return FALSE;
}

GnomeCanvasItem *
make_outline_node (GnomeCanvasGroup *group, ThinkOutlineContent *content)
{
	ThinkNode *node = (ThinkNode *)(think_outline_content_get_data (content));
	GnomeCanvasItem *new_group;
	GnomeCanvasItem *item, *check_item;
	GtkWidget *check;
	GtkWidget *label;
	GtkStyle *style;
	double x1, x2;

	new_group = gnome_canvas_item_new (group, GNOME_TYPE_CANVAS_GROUP,
					   "x", (double) 0, "y", (double) 0, NULL);

	/* nasty hack, create a label just so we can get its font */
	label = gtk_label_new ("");
	gtk_widget_ensure_style (label);
	style = gtk_widget_get_style (label);
	gtk_widget_unref (label);

	item = gnome_canvas_item_new 
		(GNOME_CANVAS_GROUP (new_group),
		 GNOME_TYPE_CANVAS_TEXT,
		 "x", (double) 0, "y", (double) 0,
		 "anchor", GTK_ANCHOR_NW,
		 "text", node->title,
		 "font_gdk", style->font,
		 NULL);

	gtk_signal_connect (GTK_OBJECT (item), "event", 
			    (void *)outline_item_event, content);

	return new_group;
}

static void
save_existing_cb (gint reply, gpointer data)
{
	if (reply == GNOME_YES){
		making_new = 1;
		save_tree_dialog (NULL, NULL);
	}

	if (reply == GNOME_NO){
		changed = 0;
		new_tree (NULL, NULL);
	}
}

static void 
new_tree (GtkWidget *widget, gpointer data)
{
	making_new = 0;

	if (changed){
		gnome_app_question (GNOME_APP (main_window), 
				    _("Save existing tree?"), 
				    (void *)save_existing_cb, NULL);
		return;
	}

	if (!tree_root)
		tree_root = g_node_new (outline);
	else
		g_node_children_foreach (tree_root, G_TRAVERSE_ALL, 
					 (void *)free_think_tree, NULL);

	if (outline_filename){
		g_free (outline_filename);
		outline_filename = NULL;
	}

	think_outline_redraw (outline);
}

static void 
open_tree_dialog (GtkWidget *widget, gpointer data)
{
	GtkWidget *file_dialog;
  
	file_dialog = gtk_file_selection_new(_("Load tree"));
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_dialog)->ok_button),
			    "clicked", (GtkSignalFunc) file_ok_sel, file_dialog);
  
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION
					       (file_dialog)->cancel_button),
				   "clicked", (GtkSignalFunc) gtk_widget_hide,
				   GTK_OBJECT (file_dialog));
  
	gtk_widget_show (file_dialog);
}

static void
save_tree_dialog (GtkWidget *widget, gpointer data)
{
	GtkWidget *dlg;

	if (outline_filename == NULL){
		save_as_tree_dialog (widget, data);
		return;
	}

	outline_save_xml (outline_filename, tree_root);
	outline_clean ();

	if (making_new)
		new_tree (NULL, NULL);
}

static void
save_as_tree_dialog (GtkWidget *widget, gpointer data)
{
	GtkWidget *dlg;

/*
	if (!changed){
		gnome_appbar_set_status (GNOME_APPBAR (appbar),
					 _("No changes to save."));
		return;
	}
*/

	dlg = gtk_file_selection_new (_("Save tree"));
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dlg)->ok_button),
			    "clicked", (GtkSignalFunc) file_save_sel, dlg);

	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION
					       (dlg)->cancel_button),
				   "clicked", (GtkSignalFunc) gtk_widget_hide,
				   GTK_OBJECT (dlg));

	gtk_widget_show (dlg);
}

static void 
file_ok_sel (GtkWidget *w, GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename (fs);

	new_tree (NULL, NULL);
	outline_load_xml (filename, &tree_root);
	outline_filename = g_strdup (filename);

	outline_clean ();
	think_outline_redraw (outline);

	gtk_widget_destroy (GTK_WIDGET (fs));
}

static void
overwrite_cb (gint reply, gpointer data)
{
	if (reply == GNOME_YES){
		outline_save_xml (data, tree_root);
		outline_clean ();

		if (making_new)
			new_tree (NULL, NULL);
	}

	g_free (data);
}


static void
file_save_sel (GtkWidget *w, GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename (fs);
	struct stat buf;

	if (outline_filename)
	{
		g_free (outline_filename);
	}
	outline_filename = g_strdup (filename);

	if (stat (filename, &buf) == 0){
		gnome_app_question (GNOME_APP (main_window), 
				    _("Overwrite existing file?"), 
				    (void *)overwrite_cb, g_strdup (filename));
	} else {
		outline_save_xml (filename, tree_root);
		outline_clean ();

		if (making_new)
			new_tree (NULL, NULL);
	}

	gtk_widget_destroy (GTK_WIDGET (fs));
}

static gboolean
free_think_node (GNode *node, gpointer data)
{
	ThinkNode *tnode = (ThinkNode *)
		(think_outline_content_get_data (THINK_OUTLINE_CONTENT (node->data)));
	g_free (tnode->title);
	g_free (tnode->text);

	return FALSE;
}

static void
free_think_tree (GNode *node, gpointer data)
{
	g_node_traverse (node, G_IN_ORDER, G_TRAVERSE_ALL, -1, 
			 (void *)free_think_node, NULL);

	think_outline_content_destroy (node);
}

static void
remove_content (GtkWidget *widget, ThinkOutlineContent *content)
{
	GNode *node;
	ThinkNode *tnode;
	g_return_if_fail (content != NULL);

	think_outline_unselect (outline);
	node = content->gnode;

	free_think_tree (node, NULL);

	think_outline_redraw (outline);
}

static void
remove_selected (GtkWidget *widget, gpointer data)
{
	ThinkOutlineContent *content;

	content = think_outline_selection (outline);
	think_outline_unselect (outline);

	if (!content)
		return;

	remove_content (NULL, content);
	outline_changed ();
}

static void
dialog_button_events (GnomeDialog *dialog, gint button_number, gpointer data)
{
	ThinkOutlineContent *content;
	ThinkNode *node;
	GtkWidget *widget;
	gchar *tmp;

	switch (button_number){
	case 0: /* cancel button */
		break;

	case 1: /* ok */
		content = THINK_OUTLINE_CONTENT (data);
		node = (ThinkNode *)think_outline_content_get_data (content);

		widget = gtk_object_get_data (GTK_OBJECT (dialog), "think_title");
		node->title = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));

		widget = gtk_object_get_data (GTK_OBJECT (dialog), "think_text");

		tmp = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);

		if (strlen (tmp) > 0){
			node->text = g_strdup (tmp);
		}
		else{
			node->text = NULL;
		}

		think_outline_content_set_dirty (content);
		think_outline_redraw (outline);

		outline_changed ();
		break;
	}
}

static void
edit_content (GtkWidget *widget, ThinkOutlineContent *content)
{
	GtkWidget *window, *title_entry, *text;
	ThinkNode *node;

	g_return_if_fail (content != NULL);

	window = gnome_dialog_new(_("Edit node"), GNOME_STOCK_BUTTON_CANCEL,
				  GNOME_STOCK_BUTTON_OK, NULL);

	node = (ThinkNode *)think_outline_content_get_data (content);

	title_entry = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (title_entry), node->title);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (window)->vbox), 
			    title_entry, TRUE, TRUE, 0);
	gtk_widget_show (title_entry);
  
	text = gtk_text_new (NULL, NULL);
	gtk_text_set_editable (GTK_TEXT (text), TRUE);

	if (node->text)
		gtk_text_insert (GTK_TEXT (text), NULL, NULL, NULL, node->text,
				 strlen (node->text));

	gtk_object_set_data (GTK_OBJECT (window), "think_title", title_entry);
	gtk_object_set_data (GTK_OBJECT (window), "think_text", text);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (window)->vbox),
			    text, TRUE, TRUE, 0);
	gtk_widget_show (text);

	gtk_signal_connect (GTK_OBJECT (window), "clicked", dialog_button_events,
			    content);
	gnome_dialog_set_close (GNOME_DIALOG (window), TRUE);

	gtk_widget_show (window);
}

static void
edit_selected (GtkWidget *widget, gpointer data)
{
	if (!think_outline_selection (outline))
		return;

	edit_content (NULL, think_outline_selection (outline));
}

static GNode *
new_think_gnode (void)
{
	gchar *title = _("- empty node -");
	GNode *gnode;
	ThinkOutlineContent *content;
	ThinkNode *node;
  
	node = think_node_new ();
	node->title = g_strdup (title);

	gnode = think_outline_content_new_with_data (node,
						    (void *)make_outline_node);

	content = THINK_OUTLINE_CONTENT (gnode->data);
	
	gtk_signal_connect (GTK_OBJECT (content), "item_drop",
			    (void *)change_wrapper, NULL);
	gtk_signal_connect (GTK_OBJECT (content), "item_drop_after",
			    (void *)change_wrapper, NULL);
	gtk_signal_connect (GTK_OBJECT (content), "item_drop_before",
			    (void *)change_wrapper, NULL);

	return gnode;
}

static void 
insert_sibling (GtkWidget *widget, ThinkOutlineContent *content)
{
	GNode *new_one, *parent, *sibling;

	g_return_if_fail (content != NULL);

	new_one = new_think_gnode ();

	sibling = content->gnode;
	parent = sibling->parent;
	g_node_insert_before (parent, sibling, new_one);

	think_outline_redraw (outline);

	outline_changed ();
}

static void 
append_sibling (GtkWidget *widget, ThinkOutlineContent *content)
{
	GNode *new_one, *parent, *sibling;
	ThinkNode *node;

	g_return_if_fail (content != NULL);

	new_one = new_think_gnode ();

	sibling = content->gnode;
	parent = sibling->parent;
	g_node_insert_before (parent, sibling->next, new_one);

	think_outline_redraw (outline);

	outline_changed ();
}

static void
insert_child (GtkWidget *widget, ThinkOutlineContent *content)
{
	GNode *new_one, *parent;
	ThinkNode *node;

	g_return_if_fail (content != NULL);

	new_one = new_think_gnode ();

	parent = content->gnode;
	g_node_insert (parent, 0, new_one);
	think_outline_redraw (outline);

	outline_changed ();
};

static void
append_child (GtkWidget *widget, ThinkOutlineContent *content)
{
	GNode *new_one, *parent;
	ThinkNode *node;

	g_return_if_fail (content != NULL);
  
	new_one = new_think_gnode ();

	parent = content->gnode;
	g_node_append (parent, new_one);
	think_outline_redraw (outline);

	outline_changed ();
};

static void
maybe_create_node (void)
{
	GNode *gnode;

	gnode = new_think_gnode ();

	g_node_insert (tree_root, 0, gnode);
	think_outline_redraw (outline);
}

static void
sel_append_sibling (GtkWidget *widget, gpointer *data)
{
	if (think_outline_selection (outline))
		append_sibling (NULL, think_outline_selection (outline));
	else
		maybe_create_node ();

	outline_changed ();
}

static void
sel_append_child (GtkWidget *widget, gpointer data)
{
	if (think_outline_selection (outline))
		append_child (NULL, think_outline_selection (outline));
	else
		maybe_create_node ();

	outline_changed ();
}

static void
about_cb (GtkWidget *widget, gpointer data)
{
	GtkWidget *about;
	const gchar *authors[] = {
		"Peter Teichman",
		NULL
	};

	about = gnome_about_new
		(_("Think"),
		 VERSION,
		 "(C) 1999 Peter Teichman",
		 authors,
		 _("Think is an outline thingy."),
		 NULL);

	gtk_widget_show (about);

	return;
}

/* menu-making stuff */
static GnomeUIInfo filemenu[] = {
	GNOMEUIINFO_MENU_NEW_ITEM (N_("_New outline"), N_("Create a new outline"),
				   new_tree, NULL),
	GNOMEUIINFO_MENU_OPEN_ITEM (open_tree_dialog, NULL),
	GNOMEUIINFO_MENU_SAVE_ITEM (save_tree_dialog, NULL),
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (save_as_tree_dialog, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_EXIT_ITEM (think_exit, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo helpmenu[] = {
	GNOMEUIINFO_MENU_ABOUT_ITEM (about_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo mainmenu[] = {
	GNOMEUIINFO_MENU_FILE_TREE (filemenu),
	GNOMEUIINFO_MENU_HELP_TREE (helpmenu),
	GNOMEUIINFO_END
};

static GnomeUIInfo toolbar[] = {
	GNOMEUIINFO_ITEM_STOCK(N_("New sibling"),
			       N_("Insert a new node as a sibling of selected node"),
			       sel_append_sibling, GNOME_STOCK_PIXMAP_NEW),
	GNOMEUIINFO_ITEM_STOCK(N_("New child"),
			       N_("Insert a new node as a child of selected node"),
			       sel_append_child, GNOME_STOCK_PIXMAP_COPY),
	GNOMEUIINFO_ITEM_STOCK(N_("Remove"),
			       N_("Remove the selected node"),
			       remove_selected, GNOME_STOCK_PIXMAP_TRASH),
	GNOMEUIINFO_ITEM_STOCK(N_("Edit"),
			       N_("Edit the selected node"),
			       edit_selected, GNOME_STOCK_PIXMAP_PROPERTIES),
	GNOMEUIINFO_END
};

static void
item_select_cb (ThinkOutline *outline, ThinkOutlineContent *content)
{
	ThinkNode *tnode;

	if (content == NULL){   /* selection has been cleared */
		return;
	}

	tnode = (ThinkNode *)think_outline_content_get_data (content);

	if (tnode->text){
		gchar **lines;
		gchar *tmp;

		/* we only want to display the first line on the status thingy */
		tmp = g_strdup (tnode->text);
		lines = g_strsplit (tmp, "\n", 1);
		gnome_appbar_set_status (GNOME_APPBAR (appbar), lines[0]);

		g_strfreev (lines);
		g_free (tmp);
	}
	else{
		gnome_appbar_clear_stack (GNOME_APPBAR (appbar));
	}
}

void
outline_changed (void)
{
	changed = 1;
}

void
outline_clean (void)
{
	changed = 0;
}

void
change_wrapper (ThinkOutlineContent *content, ThinkOutlineContent *dragged,
		gpointer data)
{
	outline_changed ();
}

static void
reply_cb (gint reply, gpointer callback_data)
{
	if (reply == GNOME_YES)
		gtk_main_quit ();
}

static void 
think_exit (GtkWidget *widget, gpointer callback_data)
{
	if (changed){
		gnome_app_question (GNOME_APP (main_window), 
				    _("Outline modified. Exit without saving changes?"), 
				    (void *)reply_cb, NULL);
	}
	else{
		gtk_main_quit ();
	}
}

GtkWidget *
think_window_new (gchar *filename)
{
	GtkWidget *tree_window, *tree;

	main_window = gnome_app_new ("think", _("Think") );
	gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
			    GTK_SIGNAL_FUNC (think_exit), NULL);

	tree_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (tree_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);

	tree_root = g_node_new (GINT_TO_POINTER (tree_window));

	outline = THINK_OUTLINE (think_outline_new_with_tree (tree_root));
	gtk_signal_connect (GTK_OBJECT (outline), "select", 
			    (void *)item_select_cb, NULL);
  
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (tree_window), 
					       GTK_WIDGET (outline));

	if (filename){
		outline_filename = g_strdup (filename);
		outline_load_xml (filename, &tree_root);
	}
  
	gnome_app_set_contents (GNOME_APP (main_window), tree_window);
	gnome_app_create_menus (GNOME_APP (main_window), mainmenu);
	gnome_app_create_toolbar (GNOME_APP (main_window), toolbar);

	appbar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_USER);
	gnome_app_set_statusbar (GNOME_APP (main_window), appbar);

	gtk_widget_show (GTK_WIDGET (outline));
	gtk_widget_show (tree_window);

	outline_clean ();
	think_outline_redraw (outline);

	return main_window;
}
