#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000, 2001 Free Software Foundation
#
# FILE:
# GConditions.py
#
# DESCRIPTION:
#
#
# NOTES:
#
# HISTORY:
#

from GObjects import GObj
import GTypecast

class ConditionError (StandardError):
  pass

class ConditionNotSupported (ConditionError):
  pass

#
# Build a condition tree using a dict
# as a source.  Assumes keys are field
# names and values are constants.
#
def buildConditionFromDict (dict, comparison=None):
  cond = GCondition()
  lastParent = cond

  if len(dict.keys()):
    lastParent = GCand(lastParent)

  for key in dict.keys():
    eq = (comparison or GCeq)(lastParent)
    GCField(eq, key)
    GCConst(eq, dict[key])

  return cond

#
# Combine two conditions with an and clause.
# NOTE: This modifies cond1 (and also returns it)
#
def combineConditions (cond1, cond2):
  if cond1 == None or cond1 == {}:
    return cond2
  elif cond2 == None or cond2 == {}:
    return cond1

  if type(cond1) == type({}):
    cond1 = buildConditionFromDict(cond1)
  if type(cond2) == type({}):
    cond1 = buildConditionFromDict(cond2)

  if not len(cond1._children):
    cond1._children = cond2._children
    return cond1
  elif len(cond2._children):
    children = cond1._children[:]
    cond1._children = []
    _and = GCand(cond1)
    _and._children = children
    if len(cond2._children) > 1:
      _and2 = GCand(cond1)
      _and2._children = cond2._children[:]
    else:
      cond1._children.append(cond2._children[0])

  return cond1

class GCondition(GObj):
  def __init__(self, parent=None, type="GCCondition"):
    GObj.__init__(self, parent, type=type)

class GConditions(GCondition):
  def __init__(self, parent=None, type="GCConditions"):
    GCondition.__init__(self, parent, type=type)

class GConditionElement (GCondition) :
  def __init__(self, parent=None, type="GConditionElement"):
    GCondition.__init__(self, parent, type=type)

class GCField (GConditionElement):
  def __init__(self, parent, name=None, datatype="char"):
    GConditionElement.__init__(self, parent, 'GCCField')
    self.type = datatype
    self.name = name

class GCParam (GConditionElement):
  def __init__(self, parent, name=None, datatype="char"):
    GConditionElement.__init__(self, parent, 'GCParam')
    self.type = datatype
    self.name = name

  def getValue(self):
    return ""

class GCConst (GConditionElement):
  def __init__(self, parent, value=None, datatype="char"):
    GConditionElement.__init__(self, parent, 'GCCConst')
    self.type = datatype
    self.value = value

class GCand(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCand')

class GCor(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCor')

class GCnot(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCnot')

class GCnegate(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCnegate')

class GCeq(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCeq')

class GCne(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCne')

class GCgt(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCgt')

class GCge(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCge')

class GClt(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GClt')

class GCle(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCle')

class GClike(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GClike')

class GCnotlike(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCnotlike')

class GCbetween(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCbetween')

class GCnotbetween(GConditionElement):
  def __init__(self, parent):
    GConditionElement.__init__(self, parent, 'GCnotbetween')



def getXMLelements(updates={}):
  xmlElements = {
      'conditions':       {
         'BaseClass': GConditions,
         'SingleInstance': 1,
         'ParentTags':  None },
      'condition':       {
         'BaseClass': GCondition,
         'ParentTags':  ('conditions','and','or','not','negate') },
      'cfield':       {
         'BaseClass': GCField,
         'Attributes': {
            'name':     {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'cparam':       {
         'BaseClass': GCParam,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Unique':   1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'cconst':       {
         'BaseClass': GCConst,
         'Attributes': {
            'value':     {
               'Required': 1,
               'Typecast': GTypecast.text } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'and':       {
         'BaseClass': GCand,
         'ParentTags':  ('condition','and','or','not','negate') },
      'or':       {
         'BaseClass': GCor,
         'ParentTags':  ('condition','and','or','not','negate') },
      'not':       {
         'BaseClass': GCnot,
         'ParentTags':  ('condition','and','or','not','negate') },
      'negate':       {
         'BaseClass': GCnegate,
         'ParentTags':  ('condition','and','or','not','negate') },
      'eq':       {
         'BaseClass': GCeq,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'ne':       {
         'BaseClass': GCne,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'gt':       {
         'BaseClass': GCgt,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'ge':       {
         'BaseClass': GCge,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'lt':       {
         'BaseClass': GClt,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'le':       {
         'BaseClass': GCle,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'like':       {
         'BaseClass': GClike,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'notlike':       {
         'BaseClass': GCnotlike,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'between':       {
         'BaseClass': GCbetween,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') },
      'notbetween':       {
         'BaseClass': GCnotbetween,
         'ParentTags':  ('eq','ne','lt','le','gt','ge',
                         'like','notlike','between','notbetween') } }

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  return xmlElements
