#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000, 2001 Free Software Foundation
#
# FILE:
# _dbsig/DBdriver.py
#
# DESCRIPTION:
# Generic implementation of dbdriver using Python DB-SIG v2 
# specification.
#
# NOTES:
# The classes below are meant to be extended
#
# HISTORY:
#

from gnue.common import GDataObjects, GDebug, GConditions
import string
from gnue.common import GDebug
import types

class DBSIG_RecordSet(GDataObjects.RecordSet): 
  def _postChanges(self): 
    if not self.isPending(): return
    if self._deleteFlag:
      statement = self._buildDeleteStatement()
    elif self._insertFlag:
      statement = self._buildInsertStatement()
    elif self._updateFlag:
      statement = self._buildUpdateStatement()

    GDebug.printMesg(5, "_postChanges: statement=%s" % statement)

    try: 
      self._parent._update_cursor.execute(statement)

      # Set _initialData to be the just-now posted values
      if not self._deleteFlag: 
        self._initialData = {}
        for key in self._fields.keys(): 
          self._initialData[key] = self._fields[key]

    except self._parent._dataObject._DatabaseError, err: 
      raise GDataObjects.ConnectionError, err

    self._updateFlag = 0
    self._insertFlag = 0
    self._deleteFlag = 0

    return 1
        

  # If a vendor can do any of these more efficiently (i.e., use a known
  # PRIMARY KEY or ROWID, then override these methods. Otherwise, leave
  # as default.  Note that these functions are specific to DB-SIG based
  # drivers (i.e., these functions are not in the base RecordSet class)

  def _buildDeleteStatement(self):
    if self._initialData.has_key(self._parent._primaryIdField):
      where = [self._parent._primaryIdFormat % \
          self._initialData[self._parent._primaryIdField]  ]
    else:
      where = []
      for field in self._initialData.keys():
        if self._parent.isFieldBound(field):
          if self._initialData[field] == None:
            where.append ("%s IS NULL" % field)
          else:
            where.append ("%s='%s'" % (field, self._initialData[field]))

    statement = "DELETE FROM %s WHERE %s" % \
       (self._parent._dataObject.table, string.join(where,' AND ') )
    return statement

  def _buildInsertStatement(self): 
    vals = []
    fields = []

    for field in self._modifiedFlags.keys():
      if self._parent.isFieldBound(field):
        fields.append (field)
        if self._fields[field] == None or self._fields[field] == '':
          vals.append ("NULL") #  % (self._fields[field]))
        else:
          vals.append ("'%s'" % (self._fields[field]))

    return "INSERT INTO %s (%s) VALUES (%s)" % \
       (self._parent._dataObject.table, string.join(fields,','), \
        string.join(vals,',') )

  def _buildUpdateStatement(self):
    updates = []
    for field in self._modifiedFlags.keys():
      updates.append ("%s='%s'" % (field, self._fields[field]))

    if self._initialData.has_key(self._parent._primaryIdField): 
      where = [self._parent._primaryIdFormat % \
          self._initialData[self._parent._primaryIdField]  ]
    else: 
      where = []
      for field in self._initialData.keys(): 
        if self._initialData[field] == None: 
          where.append ("%s IS NULL" % field)
        else: 
          where.append ("%s='%s'" % (field, self._initialData[field]))

    return "UPDATE %s SET %s WHERE %s" % \
       (self._parent._dataObject.table, string.join(updates,','), \
        string.join(where,' AND ') )


class DBSIG_ResultSet(GDataObjects.ResultSet): 
  def __init__(self, dataObject, cursor=None, \
        defaultValues={}, masterRecordSet=None):
    GDataObjects.ResultSet.__init__(
           self,dataObject,cursor,defaultValues,masterRecordSet)
    self._recordSetClass = DBSIG_RecordSet
    self._fieldNames = []
    
    if self._cursor:
      for t in(self._cursor.description):
        self._fieldNames.append(t[0])
      GDebug.printMesg(5, "Field names set to %s" % self._fieldNames)
                
    self._recordCount = cursor.rowcount or 0

    # If a DB driver supports a unique identifier for rows,
    # list it here.  _primaryIdField is the field name (lower case)
    # that would appear in the recordset (note that this can be
    # a system generated format). If a primary id is supported,
    # _primaryIdFormat is the WHERE clause to be used. It will have
    # the string  % (fieldvalue) format applied to it.
    self._primaryIdField = None
    self._primaryIdFormat = "__gnue__ = '%s'"

    GDebug.printMesg(5, 'ResultSet created')

  def _loadNextRecord(self):
    if self._cursor:
      rs = None

      try:
        rsets = self._cursor.fetchmany()
      except self._dataObject._DatabaseError, err:
        raise GDataObjects.ConnectionError, err
# TODO: It seems that popy does what the other drivers don't
# TODO: and raises this error ALOT need to find out why
# TODO: BTW - This should not be in here :(
#
# TODO: Since popy fails the fetchmany() with
      if rsets and len(rsets):
        for rs in(rsets):
          if rs:
            i = 0
            dict = {}
            for f in (rs):
              dict[self._fieldNames[i]] = f
              i += 1
            self._cachedRecords.append (self._recordSetClass(parent=self, \
                                                             initialData=dict))
          else:
            return 0
        return 1
      else:
        return 0
    else:
      return 0


class DBSIG_DataObject(GDataObjects.DataObject):

  conditionElements = {
       'add':             (2, 999, '(%s)',                   '+'      ),
       'sub':             (2, 999, '(%s)',                   '-'      ),
       'mul':             (2, 999, '(%s)',                   '*'      ),
       'div':             (2, 999, '(%s)',                   '/'      ),
       'and':             (1, 999, '(%s)',                   ' AND '  ),
       'or':              (2, 999, '(%s)',                   ' OR '   ), 
       'not':             (1,   1, '(NOT %s)',               None     ), 
       'negate':          (1,   1, '-%s',                    None     ),
       'eq':              (2,   2, '(%s = %s)',              None     ),
       'ne':              (2,   2, '(%s != %s)',             None     ),
       'gt':              (2,   2, '(%s > %s)',              None     ),
       'ge':              (2,   2, '(%s >= %s)',             None     ),
       'lt':              (2,   2, '(%s < %s)',              None     ),
       'le':              (2,   2, '(%s <= %s)',             None     ),
       'like':            (2,   2, '%s LIKE %s',             None     ),
       'notlike':         (2,   2, '%s NOT LIKE %s',         None     ),
       'between':         (3,   3, '%s BETWEEN %s AND %s',   None     ) }

  def __init__(self, strictQueryCount=1):
    GDataObjects.DataObject.__init__(self)

    GDebug.printMesg (1,"DB-SIG database driver backend initializing")

    self._resultSetClass = DBSIG_ResultSet
    self._DatabaseError = None
    self._strictQueryCount = strictQueryCount


  # This should be over-ridden only if driver needs more than user/pass
  def getLoginFields(self):
    return [['_username', 'User Name',0],['_password', 'Password',1]]


  def _createResultSet(self, conditions={}, readOnly=0, masterRecordSet=None):
    try:
      cursor = self._dataConnection.cursor()

      # pull a record count for the upcomming query
      if self._strictQueryCount:
        recordCount = self._getQueryCount(conditions)

      cursor.arraysize = self.cache
      cursor.execute(self._buildQuery(conditions))
      
    except self._DatabaseError, err: 
      raise GDataObjects.ConnectionError, err
    rs = self._resultSetClass(self, cursor=cursor, masterRecordSet=masterRecordSet)
    if self._strictQueryCount:
      rs._recordCount = recordCount
    if readOnly: 
      rs._readonly = readOnly
    
    return rs

  def _getQueryCount(self,conditions={}):
    cursor = self._dataConnection.cursor()

    cursor.execute(self._buildQueryCount(conditions))
    rs = cursor.fetchone()
    return rs[0]

  def commit(self):
    GDebug.printMesg (5,"DB-SIG database driver: commit()")

    try:
      self._dataConnection.commit()
    except self._DatabaseError, value:
      raise GDataObjects.ConnectionError, value

    self._beginTransaction()

  def rollback(self):
    GDebug.printMesg (5,"DB-SIG database driver: rollback()")

    try: 
      self._dataConnection.rollback()
    except: 
      pass	# I'm SURE this isn't right (jcater)
                # But not all db's support transactions

    self._beginTransaction()


  # Used to convert a condition tree to an sql where clause
  def _conditionToSQL (self, condition):
    if condition == {} or condition == None:
      return ""
    elif type(condition) == types.DictType:
      cond = GConditions.buildConditionFromDict(condition)
    else:
      cond = condition

    if not len(cond._children):
      return ""
    elif len(cond._children) > 1:
      chillun = cond._children[:]
      cond._children = []
      _and = GConditions.GCand(cond)
      _and._children = chillun

    where = " WHERE (%s)" % (self.__conditionToSQL (cond._children[0]))
    GDebug.printMesg(5, where)
    return where

  #
  # Used internally by _conditionToSQL
  #
  # This code recursively travels down a condition tree replacing the objects
  # with a strings representation
  def __conditionToSQL (self, element):
    if type(element) != types.InstanceType:
      return "%s" % element
    else:
      # Note that we strip the GC from the object types and lowercase the rest
      otype = string.lower(element.getObjectType()[2:])
      #print "Otype: ",otype
      if otype == 'cfield':
        return "%s" % element.name
      elif otype == 'cconst':
        if element.value == None:
          return "NULL"
        elif element.type == 'number':
          return "%s" % element.value
        else: 
          return "'%s'" % element.value
      elif otype == 'param': 
        v = element.getValue()
        return (v == None and "NULL") or ("'%s'" % element.getValue())
      elif self.conditionElements.has_key(otype): 
        for i in range(0, len(element._children)): 
          element._children[i] = self.__conditionToSQL(element._children[i])
        if len(element._children) < self.conditionElements[otype][0]: 
          raise GConditions.ConditionError, \
            'Condition element "%s" expects at least %s arguments; found %s' % \
                (otype, self.conditionElements[otype][0], len(element._children))
        if len(element._children) > self.conditionElements[otype][1]: 
          raise GConditions.ConditionError, \
            'Condition element "%s" expects at most %s arguments; found %s' % \
                (otype, self.conditionElements[otype][0], len(element._children))
        if self.conditionElements[otype][3] == None: 
          return self.conditionElements[otype][2] % tuple(element._children)
        else: 
          return self.conditionElements[otype][2] % \
            (string.join(element._children, self.conditionElements[otype][3]))
      else: 
        raise GConditions.ConditionNotSupported, \
          'Condition clause "%s" is not supported by this db driver.' % otype

  # Code necessary to force the connection into transaction mode... 
  # this is usually not necessary (MySQL is one of few DBs that must force)
  def _beginTransaction(self): 
    pass      


class DBSIG_DataObject_Object:
  def __init__(self):
   # TODO: A dummy placeholder to let old db driver work
   # TODO: each can have their *_DataObject_Object object from __init__ing
   # TODO: DBSIG_DataObject_Object
   GDebug.printMesg(0,
     "Database driver needs updated to not initialize DBSIG_DataObject_Object")

  def _buildQuery(self, conditions={}):
    GDebug.printMesg(7,'Implicit Fields: %s' % self._fieldReferences)
    if len(self._fieldReferences):
      q = "SELECT %s FROM %s%s" % \
           (string.join(self._fieldReferences.keys(),","), self.table,
            self._conditionToSQL(conditions))
    else:
      q = "SELECT * FROM %s%s" % (self.table, self._conditionToSQL(conditions))

    if hasattr(self,'order_by'):
     q = "%s ORDER BY %s " % (q, self.order_by)

    GDebug.printMesg(5,q)

    return q

  def _buildQueryCount(self, conditions={}):
    q = "SELECT count(*) FROM %s%s" % (self.table, self._conditionToSQL(conditions))

    GDebug.printMesg(5,q)

    return q

class DBSIG_DataObject_SQL:
  def _buildQuery(self, conditions={}): 
    # Obviously, this is in a pre-alpha state :)
    return "select zipcode, city, state from zipcode order by zipcode desc"

supportedDataObjects = { 
  'object': DBSIG_DataObject_Object,
  'sql':    DBSIG_DataObject_SQL
}









