#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# DataObject.py
#
# DESCRIPTION:
#
# NOTES:
#
# $Id :$

__all__ = ['DataObject']

from ResultSet import ResultSet

from gnue.common.apps import GDebug
from gnue.common.datasources import GConditions, Exceptions
import string

from ResultSet import ResultSet

###########################################################
#
#
#
###########################################################
class DataObject:

  _resultSetClass = ResultSet

  def __init__(self, connection):
    self._connection = connection

    self.masterlink = ""
    self.detaillink = ""

    self._masterfields = []
    self._detailfields = []
    self._staticCondition = None

    self._masterObject = None
    self._detailObjects = []
    self._dataConnection = None
    self._fieldReferences = {}  # Set by GDataSource; lists all fields
                                # a client explicitly references

    self._unboundFieldReferences = {}  # Contains names of all unbound
                                       # field references

    self._defaultValues = {}

    self._unicodeMode = 1 # if set to true, datasources will unicode
                          # strings instead of strings in local encoding
                          # and if non unicode strings are passed to the
                          # db driver a warning is raised (in 0.6.0 an
                          # exception will be raised)

    # TODO: This is to keep old code from breaking.
    # TODO: 0.5.1 was last version to support
    # TODO triggerExtensions (functionality is now
    # TODO: encapsulated in Connection objects.
    self.triggerExtensions = connection

  # Do we have a master datasource?
  def hasMaster(self):
    return self._masterObject != None

  # Do not over-ride by vendor code
  def createResultSet(self, conditions={}, readOnly=0, masterRecordSet=None, sql=""):
    return self._createResultSet(
       GConditions.combineConditions(conditions, self._staticCondition),
       readOnly=readOnly, masterRecordSet=masterRecordSet, sql=sql)

  # Designed to be replaced by vendor-specific code
  def _createResultSet(self, conditions={}, readOnly=0, masterRecordSet=None, \
                       sql=""):
    pass

  # Do not over-ride by vendor code
  def getQueryString(self,conditions={},forDetailSQL=None,additionalSQL=""):
    return self._buildQuery(conditions,forDetailSQL,additionalSQL)

  def createEmptyResultSet(self, readOnly=0, masterRecordSet=None):
    return self._createEmptyResultSet(readOnly, masterRecordSet)

  # Designed to be replaced by vendor-specific code
  def _createEmptyResultSet(self, readOnly=0, masterRecordSet=None):
    cond = GConditions.GCondition()
    ceq = GConditions.GCeq(cond)
    GConditions.GCConst(ceq,1,"number")
    GConditions.GCConst(ceq,0,"number")
    return self.createResultSet(conditions=cond, readOnly=readOnly,
                                masterRecordSet=masterRecordSet)


  # Add a detail data object.  This dataobject will create a new resultset
  # everytime this dataobject changes (new record, etc).  The optional
  # handler will be called after the detail dataobject is notified.  The
  # client application may wish to add a handler to know when the detail
  # has been requeried.  handler is a method that takes two arguments:
  # the master ResultSet and the detail ResultSet
  def addDetailDataObject(self, dataObject, handler=None, **params):

    dataObject.__dict__.update(params)

    GDebug.printMesg (8,"Adding a master/detail relationship to DataObject")
    dataObject._masterObject = self

    dataObject._masterfields = string.split (  \
        hasattr (dataObject, 'masterlink') and \
          dataObject.masterlink or "", ',')

    dataObject._detailfields = string.split (  \
        hasattr (dataObject, 'detaillink') and \
          dataObject.detaillink or "", ',')

    if len(dataObject._masterfields) != len(dataObject._detailfields):
      tmsg = u_("master=%s; detail=%s") % (dataObject._masterfields, dataObject._detailfields)
      raise Exceptions.MasterDetailFieldMismatch, tmsg

    # Make sure "master" fields will be in our future query
    for field in dataObject._masterfields:
      self._fieldReferences[string.strip(field)] = ""

    for field in dataObject._detailfields:
      dataObject._fieldReferences[string.strip(field)] = ""

    self._detailObjects.append ([dataObject, handler])

  #
  # Connect to database from a GConnection object
  #
  def connect(self):
    self._dataConnection = self._connection.native

  def commit(self):
    GDebug.printMesg(1,'WARNING: The use of DataObject.commit() is deprecated')
    self._connection.commit()

  def rollback(self):
    GDebug.printMesg(1,'WARNING: The use of DataObject.rollback() is deprecated')
    self._connection.rollback()


  #
  # Schema (metadata) functions
  #

  # TODO: DEPRECATED!!!!!
  def getSchemaTypes(self):
    print "WARNING: Your app is calling GDataObject.getSchemaTypes()"
    return self._connection.schema.types[:]

  # TODO: DEPRECATED!!!!!
  def getSchemaList(self, type=None):
    print "WARNING: Your app is calling GDataObject.getSchemaList()"
    return self._connection.schema.find(type=type)

  # TODO: DEPRECATED!!!!!
  def getSchemaByName(self, name, type=None):
    print "WARNING: Your app is calling GDataObject.getSchemaByName()"
    return self._connection.schema.findfirst(name=name, type=type)




  # Called when new record master in master/detail is queried
  def _masterRecordChanged(self, master):
    GDebug.printMesg (8, 'Master Record Changed')
    criteria = {}

    # If a detail result set has already been created for a particular
    # master record set, then just return/reuse this old set (after all,
    # it may contain uncommitted changes)
    if (not master.current._cachedDetailResultSets.has_key(self)) or \
        ( not int(gConfig('CacheDetailRecords')) and \
          not master.current._cachedDetailResultSets[self].isPending() ):
      doQuery = None
      for i in range(0, len(self._masterfields)):
        GDebug.printMesg(8,"Adding criteria")
        criteria[string.strip(self._detailfields[i])] = \
            master.current.getField(string.strip(self._masterfields[i]))

        #If all are set to None then this will prevent the details
        #from being queried.  This happens are startup with blank master
        #datasources.
        doQuery = doQuery or master.current.getField(string.strip(self._masterfields[i]))

        GDebug.printMesg(8,master.current.getField(self._masterfields[i]))
      if doQuery:
        master.current.addDetailResultSet(self.createResultSet(\
             conditions=criteria, masterRecordSet=master.current))
      else:
        master.current.addDetailResultSet(self.createEmptyResultSet())
##      master.current._cachedDetailResultSets[self] = \
##          self.createResultSet(conditions=criteria, masterRecordSet=master.current)
    return master.current._cachedDetailResultSets[self]

