# GNU Enterprise Common - MaxDB/SAP-DB Driver - Schema Creation
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Creation.py 6851 2005-01-03 20:59:28Z jcater $

import sapdb.dbm

from gnue.common.apps import errors
from gnue.common.datasources.GLoginHandler import BasicLoginHandler
from gnue.common.datasources.drivers.DBSIG2.Schema.Creation import \
    Creation as Base


# =============================================================================
# This class implements the schema creation for MaxDB/SAP-DB
# =============================================================================

class Creation (Base.Creation):

  END_COMMAND     = ""
  MAX_NAME_LENGTH = 32
  MAX_VARCHAR_LEN = 3999

  # ---------------------------------------------------------------------------
  # Create a new database instance
  # ---------------------------------------------------------------------------

  def createDatabase (self):
    """
    This function creates a new database instance as defined by the
    connection's parameters. The user will be askt for a username and password
    which is member of the SDBA group on the backend system and theirfore
    allowed to create new instances. If the database already exists no action
    takes place.
    """

    host     = self.connection.parameters.get ('host', 'localhost')
    dbname   = self.connection.parameters.get ('dbname', None)
    username = self.connection.parameters.get ('username', 'gnue')
    password = self.connection.parameters.get ('password', 'gnue')

    res = BasicLoginHandler ().getLogin (['OS User', host, \
                                    [['_username', u_('User Name'), False],
                                     ['_password', u_('Password'), True]]], '')

    try:
      session = sapdb.dbm.DBM (host, '', '',
                              "%s,%s" % (res ['_username'], res ['_password']))

    except sapdb.dbm.CommunicationError, err:
      raise errors.AdminError, \
          u_("Unable to establish session: %s") % errors.getException () [2]

    try:
      result = session.cmd ('db_enum')

      for entry in result.split ('\n'):
        if entry.split ('\t') [0] == dbname.upper ():
          return

      print o(u_("Creating database instance %s") % dbname)
      session.cmd ("db_create %s %s,%s" % (dbname, res ['_username'],
                                                   res ['_password']))

      print o(u_("Setting up parameters ..."))
      session.cmd ("param_startsession")
      session.cmd ("param_init OLTP")
      session.cmd ("param_put MAXUSERTASKS 10")
      session.cmd ("param_put CACHE_SIZE 20000")
      session.cmd ("param_put _UNICODE YES")
      session.cmd ("param_checkall")
      session.cmd ("param_commitsession")

      print o(u_("Adding log- and data-volumes ..."))
      session.cmd ("param_adddevspace 1 LOG  LOG_001 F 1000")
      session.cmd ("param_adddevspace 1 DATA DAT_001 F 2500")

      print o(u_("Entering administration mode"))
      session.cmd ("db_admin")

      print o(u_("Activating instance with initial user %s") % (username))
      session.cmd ("db_activate %s,%s" % (username, password))

      print o(u_("Loading system tables ..."))
      session.cmd ("load_systab -ud domp")

      print o(u_("Database instance created."))

    finally:
      session.release ()



  # ---------------------------------------------------------------------------
  # Handle special defaults
  # ---------------------------------------------------------------------------

  def _defaultwith (self, code, tableName, fieldDef, forAlter):
    """
    This function set's the defaults for 'serial'- and 'timestamp'-defaults
            
    @param code: code-tuple to merge the result in
    @param tableName: name of the table
    @param fieldDef: dictionary describing the field with the default
    @param forAlter: TRUE if the definition is used in a table modification
    """

    if fieldDef ['defaultwith'] == 'serial':
      fieldDef ['default'] = 'SERIAL'

    elif fieldDef ['defaultwith'] == 'timestamp':
      fieldDef ['default'] = 'TIMESTAMP'


  # ---------------------------------------------------------------------------
  # A key is an integer
  # ---------------------------------------------------------------------------

  def key (self, fieldDefinition):
    """
    Native datatype for a key field is INTEGER

    @param fieldDefinition: dictionary describing the field

    @return: 'INTEGER'
    """

    return "INTEGER"


  # ---------------------------------------------------------------------------
  # Create an apropriate datatype for numbers
  # ---------------------------------------------------------------------------

  def number (self, fieldDefinition):
    """
    This function returns an apropriate numeric type

    @param fieldDefinition: dictionary describing the field

    @return: numeric type (smallint, integer, fixed)
    """

    scale  = fieldDefinition.get ('precision', 0)
    length = fieldDefinition.get ('length', 0)

    if not scale:
      if length <= 5:
        return "SMALLINT"

      elif length <= 10:
        return "INTEGER"

      else:
        return "FIXED (%d)" % length

    else:
      return "FIXED (%d,%d)" % (length, scale)


  # ---------------------------------------------------------------------------
  # Native datatype for booleans
  # ---------------------------------------------------------------------------

  def boolean (self, fieldDefinition):
    """
    This funciton returns the native data type for a boolean, which is 'boolean'

    @param fieldDefinition: dictionary describing the field
    @return: 'BOOLEAN'
    """

    return "BOOLEAN"


  # ---------------------------------------------------------------------------
  # Native datatype for datetime is timestamp
  # ---------------------------------------------------------------------------

  def datetime (self, fieldDefinition):
    """
    This function returns the native type for a datetime value

    @param fieldDefinition: dictionary describing the field
    @return: 'TIMESTAMP'
    """

    return "TIMESTAMP"


  # ---------------------------------------------------------------------------
  # Get an apropriate type for strings
  # ---------------------------------------------------------------------------

  def string (self, fieldDefinition):
    """
    This function returns a native type for strings

    @param fieldDefinition: dictionary describing the field
    @return: 'VARCHAR (?)' or 'LONG'
    """

    length = fieldDefinition.get ('length', self.MAX_VARCHAR_LEN)

    if length <= self.MAX_VARCHAR_LEN:
      return "VARCHAR (%d)" % length

    else:
      return "LONG"
