# GNU Enterprise RPC interface - Py-XMLRPC client adapter
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: ServerAdapter.py 7007 2005-02-11 16:18:59Z reinhard $

# Server Parameters:
#
#    port        The port that the service is located on
#    loglevel    Loglevel of the internal server
#

from types import *

import string
import sys
import thread
import traceback

from gnue.common.apps import errors, GDebug
from gnue.common.rpc import server
from gnue.common.rpc.drivers._helpers import ObjectLibrarian, ObjectEnabler

try:
  import xmlrpc
except ImportError:
  tmsg = u_("\nUnable to load xmlrpc.  To use the XML-RPC interface, \n"
            "please install xmlrpc from:\n"
            "    http://sourceforge.net/projects/py-xmlrpc\n"
            "the appropriate DEBIAN package is python-xmlrpc.deb")
  raise server.AdapterInitializationError, tmsg

import typeconv

# =============================================================================
# Server driver
# =============================================================================

class ServerAdapter (ObjectEnabler.ObjectEnabler):

  _default_port = 8765

  # ---------------------------------------------------------------------------
  # Initialize object
  # ---------------------------------------------------------------------------

  def __init__ (self, rpcdef, bindings, params):

    # Create Subserver
    self.server = XMLRPC_Server ()

    # Build up Service Directory and enable object dispatching
    ObjectEnabler.ObjectEnabler.__init__ (self, rpcdef, bindings, params)

    if hasattr (params,'loglevel'):
      self._loglevel = params ['loglevel']
    else:
      self._loglevel = 0
    xmlrpc.setLogLevel(self._loglevel)

    if params.has_key ('httpbind'):
      GDebug.printMesg(0,"The Py-XMLRPC RPC Driver doesn't support http access")

    self.server.setParent (self)
    self.server.bindAndListen (self._port)
    self.server.setOnErr (self.OnErr)

  # ---------------------------------------------------------------------------
  # Main loop
  # ---------------------------------------------------------------------------

  def serve (self):

    quit = False
    while not quit:
      try:
	self.server.work ()
	# you could set a timeout if desired
      except:
        e = sys.exc_info ()
        if e [0] in (KeyboardInterrupt, SystemExit):
          # shut down server without showing error message
          quit = True
        else:
          print "*" * 60
          print "Error occured on server:"
          print "*" * 60
          traceback.print_exc()
          print "*" * 60
          print "\n\n"

  # ---------------------------------------------------------------------------
  # Add an attribute to the dictionary
  # ---------------------------------------------------------------------------

  def OnErr (self, server, source):

    print u_("Internal server error occured:\n"
             " server %(server)s \n on source %(source)s.") \
          % {'server': server,
             'source': source}
    return xmlrpc.ONERR_KEEP_WORK
    

  # ---------------------------------------------------------------------------
  # Call a method
  # ---------------------------------------------------------------------------

  def call (self, method, params):

    __params = typeconv.rpc_to_python (params, server.InvalidParameter)

    try:
      result = ObjectEnabler.ObjectEnabler.call (self, method, __params)
    except SystemExit:
      raise
    except:
      (group, name, message, detail) = errors.getException (3)
      raise errors.RemoteError, (group, name, message, detail)

    return typeconv.python_to_rpc (result, server.InvalidParameter)

  # ---------------------------------------------------------------------------
  # Add a method to the dictionary
  # ---------------------------------------------------------------------------

  def addRpMethod (self, object, parent, binding):    

    ObjectEnabler.ObjectEnabler.addRpMethod (self, object, parent, binding)

    # Tell the server that the method exists
    self.server.addMethods ({force_str (object._path): None})

  # ---------------------------------------------------------------------------
  # Add an attribute to the dictionary
  # ---------------------------------------------------------------------------

  def addRpAttribut (self, object, parent, binding):

    ObjectEnabler.ObjectEnabler.addRpAttribut (self, object, parent, binding)

    if not object.readonly:
      # Add the set_* directory entry
      self.server.addMethods ({force_str ('%s.set_%s' % \
                                          (parent._path, object.name)): None})

    if not object.writeonly:
      # Add the get_* directory entry
      self.server.addMethods ({force_str ('%s.get_%s' % \
                                          (parent._path, object.name)): None})

  # ---------------------------------------------------------------------------
  # Register a newly created dynamic object
  # ---------------------------------------------------------------------------

  def registerDynamicObject (self, objhandle, type):
    #
    #  add new methods to the xmlrpc server
    #
    tl = len (type)
    for i in self.directory.keys ():
      if i [0:tl] == type:
        method = force_str ("[" + objhandle + "]" + i [tl:])
#       GDebug.printMesg (1, 'Method %s registered to py-xmlrpc ' \
        print 'Method %s registered to py-xmlrpc ' \
                             % method +\
                             ' internal directory.'
        self.server.addMethods ({method: None})
     
    # add a method to close this object
    self.server.addMethods ({"["+objhandle+"]._close": None})

  # ---------------------------------------------------------------------------
  # Unregister a dynamic object
  # ---------------------------------------------------------------------------

  def deregisterDynamicObject (self, object):
    #
    #  remove the new methods from the xmlrpc server
    #
    objhandle = ObjectLibrarian.getObjectReference (object)
    type = object._type
    tl = len (type)
    for i in self.directory.keys ():
      if i[0:tl] == type:
        method = force_str ("[" + objhandle + "]" + i [tl:])
        GDebug.printMesg (9, 'Method %s is deleted from py-xmlrpc ' \
                             % method + \
                             ' internal directory.')
        self.server.removeMethods ({method:None})

    # remove the method to close this object
    self.server.removeMethods ({"["+objhandle+"]._close": None}) 

# =============================================================================
# Subclass of the xmlrpc server
# =============================================================================

class XMLRPC_Server (xmlrpc.server):

  # Remember parent
  def setParent (self, parent):
    self._parent = parent

  # Do the actual call
  def dispatch (self, servp, srcp, uri, method, params):
    return self._parent.call (method, params)

  # Remove methods of dynamic objects
  def removeMethods (self, dict):
    for (name, func) in dict.items ():
      del self.comtab [name]
      # TODO: remove method from internal comtab too
      # del self._o.comtab [name]
      # or
      # self._o.removeMethods(d)

def force_str (input):
  """converts the parameter @input into a string
     UnicodeTypes are converted to a utf-8 representation
     """
  if type(input)==UnicodeType:
    return input.encode('utf-8')
  else:
    return str(input)
