# GNU Enterprise Datasource Library - Driver for GNUe-AppServer
#
# Copyright 2000-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: RecordSet.py 6851 2005-01-03 20:59:28Z jcater $

from gnue.common.apps import errors

from gnue.common.datasources.drivers import Base

# =============================================================================
# RecordSet class
# =============================================================================

class RecordSet (Base.RecordSet):
  """
  Handles a record (i.e. an instance) in the GNUe-AppServer backend.
  """

  # ---------------------------------------------------------------------------
  # Initialization
  # ---------------------------------------------------------------------------

  def __init__ (self, parent, sm, session_id, classname, initialData = {}):

    Base.RecordSet.__init__ (self, parent, initialData = initialData)

    self.__sm         = sm
    self.__session_id = session_id
    self.__classname  = classname

    if not self._fields.has_key ('gnue_id'):
      self.__initRecord ()


  # ---------------------------------------------------------------------------
  # Delete
  # ---------------------------------------------------------------------------

  def _postDelete (self):
    """
    This function deletes the current record.
    """

    self.__sm.delete (self.__session_id,
                      self.__classname,
                      [self._fields ['gnue_id']])

  # ---------------------------------------------------------------------------
  # Insert
  # ---------------------------------------------------------------------------

  def _postInsert (self, fields):
    """
    This function updates the current record with the given fields.

    @param fields: dictionary with the new fields and their values
    """

    self.__updateRecord (fields)


  # ---------------------------------------------------------------------------
  # Update
  # ---------------------------------------------------------------------------

  def _postUpdate (self, fields):
    """
    This function updates the current reocrd with the given fields.

    @param fields: dictionary with the new fields and their values
    """

    self.__updateRecord (fields)


  # ---------------------------------------------------------------------------
  # Call a server-side function
  # ---------------------------------------------------------------------------

  def callFunc (self, methodname, parameters):
    """
    This function calls a procedure of the current record (object). All pending
    changes of the record will be posted first.

    @param methodname: name of the procedure to be called
    @param parameters: dictionary with parameters for the procedure

    @return: return value of the procedure
    """

    if self._fields.get ('gnue_id') is None:
      raise errors.ApplicationError, u_("Function call for empty record")

    # Before calling the function, post all pending changes to the server
    empty  = self._emptyFlag
    insert = self._insertFlag
    update = self._updateFlag

    # Find out the top level record set regarding to master-/detail and call
    # this recordset's post (). This will do implicit post's on the details
    current = self._parent

    while current._masterRecordSet is not None:
      current = current._masterRecordSet._parent

    current.post ()

    try:
      result = self.__sm.call (self.__session_id,
                               self.__classname,
                               [self._fields ['gnue_id']],
                               methodname,
                               parameters) [0]

      self.__updateFields ()

      self._updateFlag = True
      self._emptyFlag  = False

    except:
      self._emptyFlag  = empty
      self._insertFlag = insert
      self._updateFlag = update

      raise

    return result


  # ---------------------------------------------------------------------------
  # Update the current record set
  # ---------------------------------------------------------------------------

  def updateRecordSet (self):
    """
    This function searches for the top level master record and post's it to the
    backend. Doing this it makes sure all calculated fields have an up to date
    value. As a side effect this will fire all 'OnChange' triggers bound to the
    record's class definition.
    """

    if self._fields.get ('gnue_id') is None:
      return

    # Before calling the function, post all pending changes to the server
    empty  = self._emptyFlag
    insert = self._insertFlag
    update = self._updateFlag

    # Find out the top level record set regarding to master-/detail and call
    # this recordset's post (). This will do implicit post's on the details
    current = self._parent

    while current._masterRecordSet is not None:
      current = current._masterRecordSet._parent

    try:
      current.post ()

      self.__updateFields ()

      # Should we check if the _fields really have changed during the post to
      # set the flags ?
      self._updateFlag = True
      self._emptyFlag  = False

    except:
      self._emptyFlag  = empty
      self._insertFlag = insert
      self._updateFlag = update

      raise



  # ---------------------------------------------------------------------------
  # Initialize a record 
  # ---------------------------------------------------------------------------

  def __initRecord (self):
    """
    This function creates a new instance of a record and loads all fields from
    the backend. The store () creates a new gnue_id, calls all OnInit-triggers
    of the class and loads all fields afterwards, where the state of the record
    is still 'clean'.
    """

    self._fields ['gnue_id'] = self.__sm.store (self.__session_id,
                                                self.__classname,
                                                [None], [], [[]]) [0]
    for field in self._parent._dataObject._fieldReferences.keys ():
      if not self._fields.has_key (field.lower ()):
        self._fields [field.lower ()] = None

    self.__updateFields ()


  # ---------------------------------------------------------------------------
  # Update a record with the given fields
  # ---------------------------------------------------------------------------

  def __updateRecord (self, fields):
    """
    This function updates the record with the given fields dictionary where the
    keys are the field names and the values are the field values. After calling
    the store () function all fields will be reloaded so implicit changes will
    be reflected in the _fields-dictionary.

    @param fields: dictionary with fields (name = key, data = value)
    """

    self.__sm.store (self.__session_id,
                     self.__classname,
                     [self._fields ['gnue_id']],
                     fields.keys (),
                     [fields.values ()])
    self.__updateFields ()


  # ---------------------------------------------------------------------------
  # Update all fields of a record after update/inserts
  # ---------------------------------------------------------------------------

  def __updateFields (self):
    """
    This function loads all fields from the backend and updates the record's
    _field dictionary with the new values.
    """

    propertylist = []
    for item in self._fields.keys ():
      if self._parent.isFieldBound (item):
        propertylist.append (item)

    res = self.__sm.load (self.__session_id, self.__classname,
                          [self._fields ['gnue_id']], propertylist)

    for ix in range (0, len (propertylist)):
      self._fields [propertylist [ix]] = res [0][ix]
