#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# forms/wizards/CreateSchema.py
#
# DESCRIPTION:
# Implements a basic form template
#
# NOTES/TODO:
#  A similar tool should be build as a SchemaTemplate
#  which would take multiple GFD files as an argument to
#  create a merged schema out of them

from gnue.designer.forms.TemplateSupport import *
from gnue.designer import VERSION
from gnue.common.schema.Objects import *
import string


class CreateSchemaWizard(FormTemplate):


  # The first step in our wizard.
  # The template parser will initially
  # call GetStep(FIRST_STEP).
  FIRST_STEP = '0'


  ###############
  #
  # Initialize any runtime variables
  #
  def Start(self, root, current):
    self.form = root
    self.current = current

    return 0


  ###############
  #
  # Return the markup for a specific page
  #
  def GetStep(self, step):

    # 0 - Get name / Source (existing datasource, new
    #     bound datasource, fixed set of choices)
    # 1 - Select / Create Source

    #
    # Step #0 / Get Name
    #
    if step == '0':

      content = [WizardText(_('This will create a GNUe schema definition '+
                            'according to the needs of this form.')),
                 WizardInput('file',label=_('Write to File:'),required=1,size=20)]

      return   { 'title': 'Create Schema Definition',
                 'content': content,
                 'prev': None,
                 'next': None }


  ###############
  #
  # Verify contents of current step
  # Return None if no problems, otherwise
  # return a tuple of error message strings
  #
  def ValidateStep(self, stepCode):

    # The Simple wizard uses basic "required"
    # settings in the page markup, so doesn't
    # need any special validation.
    return None



  ###############
  #
  # We have all the data, so generate our form. This
  # is called after the user clicks the "Finish" button.
  # No more user input is allowed at this point.
  #
  def Finalize(self):

    self.tablelist = {}
    self.schema = GSSchema()
    self.gstables = GSTables(self.schema)

    # Create Tables out of <datasource> tags
    self.form.walk(self.CreateTables)

    # Populate tables with fields, primary and foreign keys
    self.form.walk(self.PopulateSchema)

    dest = open(self.variables['file'],"w")

    dest.write("""<?xml version="1.0"?>
<!-- Schema definition created by GNUe Designer's
     "Create Schema from Form" plugin -->
""")

    dest.write(self.schema.dumpXML().encode('utf-8'))
    dest.close()

    return 1

  def CreateTables(self,obj):
    if obj._type=='GFDataSource':

      if hasattr(obj,'table'):

        #
        # add the table
        #

        newtable = GSTable(self.gstables)
        newtable.name=obj.table
        GSFields(newtable)
        GSConstraints(newtable)
        self.tablelist[string.lower(obj.name)]=newtable

      else:
        print "WARNING: You defined a datasource without an 'table' "+\
              "attribute."


  def PopulateSchema(self,obj):
    # TODO: * care for multiple fields in fk_descriptions
    #       * add error messages in case of missing/wrong datasources
    #         for a fk_source, or a datasource field (in <block ...>)
    #       * add a way to check for a datasource name with right case before
    #         just don't care for the case
    #       * add a function for guessing row types

    if obj._type=='GFEntry':

      myblock=obj.findParentOfType('GFBlock')

      #
      #  add a simple field to the table
      #
      if hasattr(obj,'field') and hasattr(myblock,'datasource') and \
             self.tablelist.has_key(string.lower(myblock.datasource)):

        table=self.tablelist[string.lower(myblock.datasource)]

        type = self.DEFAULT_TYPE # standart type

        if hasattr(obj,'style') and obj.style == 'checkbox':
          type = self.BOOLEAN_TYPE

        if hasattr(obj,'numeric') and obj.numeric==1:
          type = self.NUMERIC_TYPE

        if hasattr(obj,'height') and obj.height>1:
          type = self.TEXT_TYPE

        if hasattr(obj,'typecast'):
          if obj.typecast=='number':
            type = self.NUMERIC_TYPE

          if obj.typecast=='date':
            type = self.DATE_TYPE

          if obj.typecast=='text' and hasattr(obj,'numeric') and \
                 obj.numeric==1: # reset previous numeric fields
            type = self.STRING_TYPE

        # string type needs a definit size
        if type == self.STRING_TYPE:

          size = self.DEFAULT_SIZE

          if hasattr(obj,'max_length'):
            size = obj.max_length

          elif hasattr(obj,'width'):
            size = obj.width

          self.AddField(table, obj.field, self.STRING_TYPE, size)

        else:

          self.AddField(table, obj.field, type)

      #
      #  add fields to the table referenced in dropdowns
      #
      if hasattr(obj,'fk_key') and hasattr(obj,'fk_source') and\
             self.tablelist.has_key(string.lower(obj.fk_source)):

        table=self.tablelist[string.lower(obj.fk_source)]

        # add 'fk_key' field
        if type == self.STRING_TYPE:
          self.AddField(table, obj.fk_key, type, size)

        else:
          self.AddField(table, obj.fk_key, type)

        if hasattr(obj,'fk_description'):

          # add 'fk_description' field
          self.AddField(table, obj.fk_description, self.STRING_TYPE)


    elif obj._type=='GFDataSource':

      if hasattr(obj,'table'):

        #
        # check for master/detail relation ship
        #

        if hasattr(obj,'master') and hasattr(obj,'masterlink') and \
               hasattr(obj,'detaillink'):

          table=self.tablelist[string.lower(obj.name)]

          # create a foreign key on this table
          newconstraint=GSConstraint(table.findChildOfType('GSConstraints'))
          newconstraint.name="fk_%s_%s" % (obj.table,obj.detaillink)
          newconstraint.type="foreignkey"

          nfk_field=GSConstraintField(newconstraint)
          nfk_field.name=obj.detaillink
          nfk_ref=GSConstraintRef(newconstraint)
          nfk_ref.name=obj.masterlink

          # create a field for that key, use string type for now
          # TODO: Check if detaillink or masterlink type is defined
          # if it one of the two is defined, then use the definition for both
          self.AddField(table, obj.detaillink, self.DEFAULT_TYPE)

          if self.tablelist[string.lower(obj.master)]:

            mtable=self.tablelist[string.lower(obj.master)]
            # set table name for foreign key
            nfk_ref.table=mtable.name

            oldpkey=mtable.findChildOfType("GSPKField")
            if oldpkey==None:
              pkey=GSPrimaryKey(mtable)
              pkey.name="pk_%s_%s" % (mtable.name,obj.masterlink)
              pkeyf=GSPKField(pkey)
              pkeyf.name=obj.masterlink

              # add field to master table, using default type
              self.AddField(mtable, obj.masterlink, self.DEFAULT_TYPE)

            elif oldpkey.name!=obj.masterlink:
              print "WARNING: different primary keys computed out of "+\
                    "master detail relationship ('%s'!='%s')." % \
                    (oldpkey.name!=obj.masterlink)
          else:
            print "WARNING: master datasource is not defined in this file."


    else:
      pass

  def AddField(self,table,fieldname,type,length=None):

    fields = table.findChildOfType('GSFields')

    # check if field already exists
    for child in fields._children:
      if child._type == 'GSField' and child.name==fieldname:
        # field already exists, so returning
        # (possibly updating fieldtype here)
        if type!=self.DEFAULT_TYPE:
          child.type=type

        if length!=None:
          child.size=int(length)
        return

    # create field
    newfield = GSField(fields)
    newfield.name=fieldname
    newfield.type=type
    if length!=None:
      newfield.size=int(length)
    elif type==self.STRING_TYPE:
      newfield.size=self.DEFAULT_SIZE



############
#
# Basic information about this template
#
# TODO: This is temporarily disabled for 0.5.0 release
# TODO: Needs some overhauling to be compatable w/0.5
__DISABLE__TemplateInformation = {
    'Product': 'forms',
    'BaseID' : 'CreateSchema',
    'BaseClass' : CreateSchemaWizard,
    'Name' : _('Create Schema Definition'),
    'Description' : _('Create a gnue schema definition file'),
    'Version' : VERSION,
    'Author' : 'The GNUe Designer Team',
    'Behavior': WIZARD,
    'MenuLocation' : ('Tools|Extras',_('Create Schema from Form'))
}

