/* Copyright (C) 2002,2003 Pascal Haakmat */

#ifndef SHELL_H
#define SHELL_H

#include <sys/time.h>
#include <unistd.h>
#include <gnome.h>
#include <pthread.h>
#include <audiofile.h>
#include "snd.h"
#include "mixer.h"
#include "modstate.h"
#include "marker.h"
#include "grid.h"

#define SLOPE_ENABLED(track, type) \
  ((shl->markers[track]->marker_types_enabled & type) == type)
#define PIXEL_TO_FRAME(x) (shl->hadjust->value + ((x) * shl->hres))
#define LOOP_IS_ACTIVE(shl) ((shl)->loop && (shl)->loop_start != (shl)->loop_end)
#define DISTANCE(x, y) (((x) - (y)) < 0 ? -((x) - (y)) : ((x) - (y)))

#define DEBUG_FLAG_STEP_MODE   0x01
#define DEBUG_FLAG_DRAW_BLOCKS 0x02

#define WHEEL_FORWARD 0
#define WHEEL_BACKWARD 1

typedef struct {
    char *widget_name;
    char *signal_name;
    GtkSignalFunc signal_func;
} signal_bindings;

typedef struct {
    GtkObject *object;
    //    int ref;
} action_bindings;

struct properties {
    float sample_rate;
    int sample_width;
    GtkToggleButton *resample;
    GtkSpinButton *rate;
    GtkLabel *new_label;
    GtkLabel *cur_label;
    GtkWidget *dialog;
};

typedef struct {
    pthread_mutex_t player_display_lock;
    pthread_cond_t player_display_cond;
    int player_display_draw_requested;
    int player_display_cancel_requested;
    pthread_t player_display_thread;
    pthread_t player_thread;
    pthread_mutex_t player_running_lock;
    pthread_cond_t player_running_cond;
    int player_running;
    int player_cancel_honored;
    int player_cancel_requested;
    int play_audio_fd;
    int rec_audio_fd;
    int record_discarded_frames;
    int record_mode;
    int record_replace;
    unsigned int draws_requested;
    unsigned int draws_prevented;
    frame_bits_t muxbuf;
    frame_bits_t srcbufs[MAX_TRACKS];
    AFframecount player_pos;
    AFframecount undo_start_offset;
    AFframecount undo_end_offset;
    int undo_channel_map;
    snd *undo_sr;
} player;

typedef struct {
    char magic[2];
    snd *sr;
    GList *undo_stack;
    player player;
    action_bindings *bindings;

    /* The grid. */

    struct grid grid;

    /* The id of the module that is currently being executed or -1 if
       none. */

    int active_module_id;

    /* Shell is about to close. */

    int close_requested;

    /* Current action recursion depth, when user closes this shell
       we wait for this to become zero before closing. */

    int use;

    /* This flag is set when the user requests that a module cancels
       the current operation (this flag only makes sense when
       active_module_id is not -1). */

    int module_cancel_requested;

    /* This flag is set when a file loading operation should be
       canceled. */

    int file_load_cancel_requested;

    /* Bitfield specifying which actions we are currently
       executing. */

    int action_state;

    /* State for the modules in this shell. Every loaded module gets a
       slot in this array to store their state. */

    mod_state module_state[MAX_MODULES];

    /* Loop start and end. */

    AFframecount loop_start;
    AFframecount loop_end;

    /* Selection; select_channel_map is a bitfield specifying which
       channels are selected. */

    int select_channel_map;
    AFframecount select_start;
    AFframecount select_end;
    AFframecount select_pivot;
    AFframecount select_flex;

    /* Horizontal resolution, power of 2. E.g. 128 specifies a scale
       of 1:128, while 0.25 specifies 4:1 magnification. */

    float hres;

    /* Vertical resolution, e.g. 128 or 256. */

    int vres;

    /* Misc flags. */

    unsigned int loop: 1;
    unsigned int has_changed: 1;
    unsigned int has_name: 1;
    unsigned int follow_playback: 1;
    unsigned int envelope_enabled: 1;
    unsigned int show_zero: 1;
    unsigned int show_envelope: 1;
    unsigned int show_grid: 1;
    unsigned int snap_to_grid: 1;
    unsigned int snap_to_cuepoints: 1;
    unsigned int record_mode: 1;
    unsigned int record_replace: 1;
    unsigned int debug_flags: 1;
    unsigned int scroll_drag: 1;
    unsigned int scrubbing: 1;
    unsigned int being_drawn: 1;

    /* Buffers. */

    graph_bits_unit_t *graph_bits_buffer_low;
    graph_bits_unit_t *graph_bits_buffer_high;
    GdkPixmap *pixmap;
    GdkPixmap *waveformpixmap;
    GdkPixmap *mixerpixmap;

    /* Properties dialog. */
    
    struct properties props;

    /* UI elements. */

    GnomeApp *appwindow;
    GnomeAppBar *appbar;
    GtkDrawingArea *canvas;
    GtkDrawingArea *infocanvas;
    GtkDrawingArea *mixercanvas;
    GtkAdjustment *hadjust;
    GtkAdjustment *vadjust;
    GtkRange *hscrollbar;
    GtkRange *vscrollbar;
    GtkHRuler *hruler;
#ifdef HAVE_GNOME2
    GtkProgressBar *progress;
#else
    GtkProgress *progress;
#endif
    GtkSpinButton *gridunits;
    GtkSpinButton *gridbpm;
    GtkOptionMenu *gridmeasurement;
    GdkDrawable *drawable;
    GtkMenuItem *cancelmodule;
    GtkWidget *marker_dialog;
    GtkEntry *marker_label;
    GdkGC *gc;

    /* Need to keep track of mouse position over waveform. */

    gdouble last_mouse_x;
    gdouble last_mouse_y;
    gdouble last_mouse_x_root;

    /* Tracks last mouse wheel movement so that rapid succession of
       mouse wheel movement means to zoom in/out on same frame. */

    struct timeval last_wheel_movement;
    AFframecount last_wheel_pos;
    double last_wheel_x;

    /* Markers. */

    struct marker *marker_being_dragged;
    int marker_dragged_on_track;
    int marker_next_label;

    /* Mixer. */

    mixer *mixer;
    int target_channel_being_dragged;
    int source_channel_being_dragged;

    /* Action. */
    
    struct _action *tail_action;
} shell;


shell *
shell_new();

void
shell_destroy(shell *shl);

void 
shell_cursor_set(shell *shl, 
                 GdkCursorType type);

void
shell_grid_bpm_set(shell *shl,
                   float bpm);

void
shell_grid_units_set(shell *shl,
                     float bpm);

void
shell_grid_measurement_set(shell *shl,
                           enum grid_measurement measure);

char *
shell_path_strip(const char *p);

void
shell_status_default_set(shell *shl);

void
shell_status_push(shell *shl, 
                  const char *format, 
                  ...);

void
shell_status_pop(shell *shl);

void
shell_redraw(shell *shl);

void
shell_viewport_center(shell *shl, 
                      AFframecount start,
                      AFframecount end);

GtkObject *
shell_bindings_find(shell *shl,
                    const char *name);

void
shell_bindings_disable(shell *shl,
                       char *names[]);

int
shell_canvas_width_get(shell *shl);

#endif /* ! SHELL_H */
