/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * FIXME: I don't know anything about filtering.
 * Portions Copyright 1991 Lance Norskog And Sundry Contributors.
 * Some of this code was stolen and slightly rewritten from the
 * soxgamma distribution, file band.c.
 */

/*
 * July 5, 1991
 * Copyright 1991 Lance Norskog And Sundry Contributors
 * This source code is freely redistributable and may be used for
 * any purpose.  This copyright notice must be maintained. 
 * Lance Norskog And Sundry Contributors are not responsible for 
 * the consequences of using this software.
 */

/*
 * Sound Tools Bandpass effect file.
 *
 * Algorithm:  2nd order recursive filter.
 * Formula stolen from MUSIC56K, a toolkit of 56000 assembler stuff.
 * Quote:
 *   This is a 2nd order recursive band pass filter of the form.                
 *   y(n)= a * x(n) - b * y(n-1) - c * y(n-2)   
 *   where :    
 *        x(n) = "IN"           
 *        "OUT" = y(n)          
 *        c = EXP(-2*pi*cBW/S_RATE)             
 *        b = -4*c/(1+c)*COS(2*pi*cCF/S_RATE)   
 *   if cSCL=2 (i.e. noise input)               
 *        a = SQT(((1+c)*(1+c)-b*b)*(1-c)/(1+c))                
 *   else       
 *        a = SQT(1-b*b/(4*c))*(1-c)            
 *   endif      
 *   note :     cCF is the center frequency in Hertz            
 *        cBW is the band width in Hertz        
 *        cSCL is a scale factor, use 1 for pitched sounds      
 *   use 2 for noise.           
 */

#include <math.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include "../modutils.h"

module modinfo;

typedef struct {
    module_id id;
    shell *shl;
    double frequency;
    double width;
    double gain;
    GtkSpinButton *frequency_control;
    GtkSpinButton *width_control;
    GtkWidget *dialog;
} filter_params;

#ifdef HAVE_GNOME2
#define BANDPASS_GLADE_FILE "bandpass-2.glade"
#else
#define BANDPASS_GLADE_FILE "bandpass.glade"
#endif

/*
 * Slightly rewritten version of the band_start() function from the
 * soxgamma distribution.
 */
inline void
calc_coeffs(double rate,
            double frequency,
            double width,
            double *ap,
            double *bp,
            double *cp) {
    double a, b, c;
    int noise = 0;

    c = exp(-2 * M_PI * width / rate);
    b = -4 * c / (1 + c) * cos(2 * M_PI * frequency / rate);
    if(noise)
        a = sqrt(((1 + c) * (1 + c) - b * b) * (1 - c) / (1 + c));
    else
        a = sqrt(1 - b * b / (4 * c)) * (1 - c);
    *ap = a;
    *bp = b;
    *cp = c;
}

void
bandpass(shell *shl,
         int track,
         AFframecount start_offset,
         AFframecount end_offset,
         double frequency,
         double width) {
    int32_t *out_buffer = mem_calloc(1, EFFECT_BUF_SIZE * sizeof(int32_t));
    int32_t out, out1, out2;
    double a, b, c;
    int i;
    ITERATOR_INIT(start_offset, end_offset - start_offset);

    if(!out_buffer) {
        FAIL("not enough memory for out buffer (%d bytes)\n",
             EFFECT_BUF_SIZE * sizeof(int32_t));
        ITERATOR_EXIT();
        return;
    }

    out1 = out2 = 0;
    ITERATOR(shl, shl->sr->tracks[track],
             /*
              * This for-loop is a slightly rewritten version of the
              * band_flow() function in the soxgamma distribution.
              */
             for(i = 0; i < iter_read; i++) {
                 calc_coeffs(shl->sr->rate, frequency, width, &a, &b, &c);
                 out = (a * int32_frame_bits[i] - b * out1) - c * out2;
                 out2 = out1;
                 out1 = out;
                 out_buffer[i] = out;
             }
             track_int32_frames_replace(shl->sr->tracks[track],
                                        out_buffer,
                                        iter_frame_offset,
                                        iter_read));
    free(out_buffer);
    ITERATOR_EXIT();
    return;
}

module *
module_new() {
    MODULE_INIT(&modinfo, 
		"Bandpass",
		"Pascal Haakmat",
		"Copyright (C) 2002,2003");
    return &modinfo;
}

void
on_apply_clicked(GtkWidget *w,
                 gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    filter_params *p = (filter_params *)module_state->data;
    p->frequency = gtk_spin_button_get_value(p->frequency_control);
    p->width = gtk_spin_button_get_value(p->width_control);
    action_do(ACTION_MODULE_EXECUTE_NEW(WITH_UNDO, 
                                        p->shl,
                                        p->id));
}

void
on_ok_clicked(GtkWidget *w,
              gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    filter_params *p = (filter_params *)module_state->data;
    on_apply_clicked(w, NULL);
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_close_clicked(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    filter_params *p = (filter_params *)module_state->data;
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_dialog_destroy(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    module_state->is_open = 0;
    free(module_state->data);
}

void
module_open(module_id id,
            shell *shl, 
            int undo) {
    GladeXML *xml;
    GtkWidget *w;
    char path[4096];
    filter_params *p = mem_calloc(1, sizeof(filter_params));
    
    if(!p) {
        gui_alert("Not enough memory.");
        return;
    }

    snprintf(path, 4096, "%s/%s", dirname(modules[id].fname), BANDPASS_GLADE_FILE);
    xml = glade_xml_new(path, "dialog", NULL);
    
    if(!xml) {
        gui_alert("Bandpass: could not load interface %s.", path);
        free(p);
        return;
    }

    shl->module_state[id].is_open = 1;
    shl->module_state[id].data = p;
    p->id = id;
    p->shl = shl;
    p->frequency_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "frequency"));
    p->width_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "width"));
    p->dialog = GTK_WIDGET(glade_xml_get_widget(xml, "dialog"));
    g_signal_connect(GTK_OBJECT(p->dialog), "destroy", 
                     G_CALLBACK(on_dialog_destroy), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "ok"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_ok_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "close"));
    g_signal_connect(GTK_OBJECT(w), "clicked",
                     G_CALLBACK(on_close_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "apply"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_apply_clicked), shl);
    g_object_set_data(G_OBJECT(p->dialog), "user_data",
                      GINT_TO_POINTER(&shl->module_state[id]));
    g_object_unref(G_OBJECT(xml));
}

void
module_close(mod_state *module_state) {
    gtk_object_destroy(GTK_OBJECT(((filter_params *)module_state->data)->dialog));
}

action_group *
module_execute(shell *shl, 
               int undo) {
    AFframecount start = shl->select_start,
        end = shl->select_end;
    int map = shl->select_channel_map;
    int t, id = shl->active_module_id;
    double frequency, width;
    filter_params *p = NULL;
    action_group *undo_ag = NULL;

    if(!shl->module_state[id].is_open) { 
        gui_alert("Bandpass does not have defaults.");
        return NULL;
    }
    p = shl->module_state[id].data;

    frequency = p->frequency;
    width = p->width;

    /* FIXME: deny this in interface (as well). */

    if(frequency > shl->sr->rate / 2) {
        gui_alert("Bandpass: frequency must be < %f.", shl->sr->rate / 2);
        return NULL;
    }
    
    rwlock_rlock(&shl->sr->rwl);
    if(undo) 
        undo_ag = action_group_undo_create(shl,
                                           map, 
                                           start,
                                           end - start,
                                           start,
                                           end - start);

    for(t = 0; t < snd_track_count(shl->sr); t++) 
        if((1 << t) & map)
            bandpass(shl,
                     t,
                     start,
                     end,
                     frequency,
                     width);
    rwlock_runlock(&shl->sr->rwl);

    return undo_ag;

}
