# gozerbot/redispatcher.py
#
#

""" implement RE (regular expression) dispatcher """

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import rlog, calledfrom, handle_exception, lockdec
import gozerbot.thr as thr
import sys, re, copy, types, thread

relock = thread.allocate_lock()
locked = lockdec(relock)

class Recallback(object):

    """ a regular expression callback """

    def __init__(self, index, regex, func, perm, plugname, speed=5, \
threaded=True, allowqueue=True, options={}):
        # index of place of RE callback in list
        self.name = "noname"
        self.index = index
        # the RE to match
        self.regex = regex
        self.compiled = re.compile(regex)
        # the function to call if RE matches
        self.func = func
        # make sure perms is a list
        if type(perm) == types.ListType:
            self.perms = list(perm)
        else:
            self.perms = [perm, ]
        # plug name
        self.plugname = plugname
        # speed at which the functin is to be dispatched
        self.speed = copy.deepcopy(speed)
        # flag to see if RE callback is to be run in a thread or in the main
        # loop
        self.threaded = copy.deepcopy(threaded)
        self.allowqueue = copy.deepcopy(allowqueue)
        self.options = dict(options)

class Redispatcher(object):

    """ this is were the regexs callbacks live """

    def __init__(self):
        self.relist = []
        
    def size(self):
        """ nr of callbacks """
        return len(self.relist)

    def whatperms(self):
        """ return possible permissions """
        result = []
        for i in self.relist:
            for j in i.perms:
                if j not in result:
                    result.append(j)
        return result

    def list(self, perm):
        """ list re with permission perm """
        result = []
        perm = perm.upper()
        for recom in self.relist:
            if perm in recom.perms:
                result.append(recom)
        return result

    def getfuncnames(self, plug):
        """ return function names in plugin """
        result = []
        for i in self.relist:
            if i.plugname == plug:
                result.append(i.func.func_name)
        return result
        
    def permoverload(self, funcname, perms):
        """ overload permission of function with funcname  """
        perms = [perm.upper() for perm in perms]
        got = 0
        for nr in range(len(self.relist)):
            try:
                if self.relist[nr].func.func_name == funcname:
                    self.relist[nr].perms = list(perms)
                    rlog(0, 'redispatcher', '%s function overloaded with %s' \
% (funcname, perms))
                    got = 1
            except AttributeError:
                rlog(10, 'redispatcher', 'permoverload: no %s function' % \
funcname)
        if got:
            return 1

    def add(self, index, regex, func, perm, speed=5, threaded=True, \
allowqueue=True, options={}):
        """ add a command """
        try:
            # get plugin name from where callback is added
            plugname = calledfrom(sys._getframe())
            # add Recallback
            self.relist.append(Recallback(index, regex, func, perm, plugname, \
speed, threaded, allowqueue, options))
            # sort of index number
            self.relist.sort(lambda a, b: cmp(a.index, b.index))
            rlog(0, 'redispatcher', 'added %s (%s) ' % (regex, plugname))
        finally:
            pass

    def unload(self, plugname):
        """ unload regexs commands """
        got = 0
        try:
            for i in range(len(self.relist)-1, -1 , -1):
                if self.relist[i].plugname == plugname:
                    rlog(1, 'redispatcher', 'unloading %s (%s)' % \
(self.relist[i].regex, plugname))
                    del self.relist[i]
                    got = 1
        finally:    
            pass
        if got:
            return 1

    def getcallback(self, txt):
        """ get re callback if txt matches """
        for i in self.relist:
            try:
                result = re.search(i.compiled, txt)
                if result:
                    return i
            except:
                pass

    def dispatch(self, callback, txt):
        """ dispatch callback on txt """
        try:
            result = re.search(callback.compiled, txt)
            if result:
                if callback.threaded:
                    thr.start_new_thread(callback.func, (txt, result.groups()))
                else:
                    callback.func(txt, result.groups())
                    return 1
        except Exception, ex:
            handle_exception()

class Botredispatcher(Redispatcher):

    """ dispatcher on ircevent """

    def dispatch(self, callback, bot, ievent):
        """ dispatch callback on ircevent """
        try:
            result = re.search(callback.compiled, ievent.txt)
            if result:
                ievent.groups = list(result.groups())                
                if callback.threaded:
                    thr.start_bot_command(callback.func, (bot, ievent))
                else:
                    callback.func(bot, ievent)
                return 1
        except Exception, ex:
            handle_exception(ievent)

rebefore = Botredispatcher()
reafter = Botredispatcher()
