/* This is for emacs: -*-Mode: C++;-*- */
#if !defined(__INC_GQL_DRIVER_MANAGER_H)
#define __INC_GQL_DRIVER_MANAGER_H

#include <string>
#include <list>

#include <yehia/plugin.h>

#include <gql++/connection.h>
#include <gql++/driver.h>

namespace GQL
{

/** The DriverManager class.
 *
 *  The DriverManager is responsible for opening connections and
 *  managing the drivers. It uses an uC::PluginManager object for the
 *  driver plugin management.
*/
class DriverManager : public SigC::Object
{
  public:
    /** Contructor.
     * \param mgr PluginManager instance to use for handling of the driver
     * plugins. If 0, PluginManager::instance() will be used. 
     * \param area area of uCXX plugin namespace used. Don't specify a 
     * value unless you have installed the driver plugins in an 
     * unusual place. */
    DriverManager(Yehia::PluginManager& mgr = Yehia::PluginManager::instance(), 
                  const std::string& area = "gql");

    /** Destructor. */
    ~DriverManager();
    
    /** Try to open a connection.
     * If is_auto_load() is \c true, the right driver will be loaded 
     * and registered.
     * \param url GQL database URL
     * \param name Database username.
     * \param passwd Password. 
     * \exception SQLException opening failed. 
     * \return Opened connection. */
    Connection *get_connection(const std::string &url,
                               const std::string &name = std::string(),
                               const std::string &passwd = std::string());

    /** Register a driver.
     * \param id Driver ID. If the driver is not already registered, 
     *   attempt to load it. 
     * \return \c true if the driver is now registered. */
    bool register_driver(const std::string& id);
    
    /** Remove a driver.
     * \param id Driver ID.
     * \exception SQLException driver \p id was not registered. */
    void deregister_driver(const std::string& id);
    
    /** Get a list of registered drivers.
     * \return List of driver IDs. */
    std::list<std::string> get_registered_drivers() const;
    
    /** Get the parent node of the drivers.
     * \return The Yehia::PluginNode which contains the driver plugins. */
    const Yehia::PluginNode& get_drivers() const { return *drivers_; }
    
    /* Get a driver by ID.
     * \return Driver with ID \p id, 0 if this driver is not registered. */
    Driver *get_driver(const std::string& id);

    /** Get autoload status.
     * \return \c true if drivers are autoloaded. */
    bool is_auto_load() const { return auto_load_; }
    /** Set autoload status.
     * \param do_auto Wether to autoload drivers. */
    void set_auto_load(bool do_auto) { auto_load_ = do_auto; }

    /** Get underlying PluginManager object.
     * \return The PluginManager object that does the real work. */
    Yehia::PluginManager& plugin_manager() { return pm_; }
  private:
    Yehia::PluginManager& pm_;
    Yehia::PluginNode *drivers_;
    std::string area_name_;
    bool auto_load_;
};

}

#endif
