/*  
  Copyright 2002, Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <stdio.h>
#include <unistd.h>

#include "app.h"

namespace GQLShell
{

using namespace std;

// We assume a URL like gql:<driver>:[//<host>[:<port>]/]<dbname>.
static void url_split(const string& url, 
                      string& driver, 
                      string& host, 
                      string& port,
                      string& dbname)
{
  string::size_type newpos, pos = 4;
  
  driver = "";
  host = "";
  port = "";
  dbname = "";
  
  if (url.substr(0, 4) != "gql:")
    return;

  // Driver
  if ((newpos = url.find(':', pos)) == string::npos)
    return;
  
  driver = url.substr(pos, newpos - pos);
  pos = newpos + 1;
  
  if (url[pos] == '/' && url[pos + 1] == '/')
  {
    // we probably have a hostname next
    if ((newpos = url.find(':', pos)) != string::npos)
    {
      // we probably have host:port
      host = url.substr(pos + 2, newpos - (pos + 2));
    
      // hostnumber must be decimal integer
      while (isdigit(url[++newpos]))
        port += url[newpos];
      
      if (url[newpos] == '/') 
        dbname = url.substr(newpos + 1); // ok, rest is dbname
      else
      {
        // we haven't the correct format (host[:port]/dbname), so
        // we'll assume the whole thing is the dbname
        host = "";
        port = "";
        dbname = url.substr(pos);
      }
    }
    else if ((newpos = url.find('/', pos + 2)) != string::npos)
    {
      // no port
      host = url.substr(pos + 2, newpos - (pos + 2));
      dbname = url.substr(newpos + 1);
    }
    else
    {
      // no host, the whole thing is dbname
      dbname = url.substr(pos);
    }
  }
  else
    dbname = url.substr(pos);
}

/*--------------------------
 * Return a prompt made by interpolating certain
 * tcsh style escape sequences into prompt[i]
 *
 * Defined interpolations are:
 * %M - database server "hostname.domainname", "[local]" for AF_UNIX
 *              sockets, "[local:/dir/name]" if not default
 * %m - like %M, but hostname only (before first dot), or always "[local]"
 * %> - database server port number
 * %n - database user name
 * %d - current database driver
 * %/ - current database
 * %~ - like %/ but "~" when database name equals user name
 * %# - "#" if superuser, ">" otherwise
 * %R - in prompt1 normally =, or ^ if single line mode,
 *			or a ! if session is not connected to a database;
 *		in prompt2 -, *, ', or ";
 *		in prompt3 nothing
 * %? - the error code of the last query (not yet implemented)
 * %% - a percent sign
 *
 * %[0-9]		   - the character with the given decimal code
 * %0[0-7]		   - the character with the given octal code
 * %0x[0-9A-Fa-f]  - the character with the given hexadecimal code
 *
 * %`command`	   - The result of executing command in /bin/sh with trailing
 *					 newline stripped.
 * %:name:		   - The value of the psql variable 'name'
 * (those will not be rescanned for more escape sequences!)
 *
 *--------------------------
 */
std::string App::get_prompt(PromptType type) const
{
  bool esc = false;
  const char *prompt_string;
  string driver, host, port, dbname;
  string result;
  
  if (type == PROMPT_READY)
    prompt_string = prompt_[0].c_str();
  else if (type == PROMPT_CONTINUE || 
           type == PROMPT_SINGLEQUOTE || 
           type == PROMPT_DOUBLEQUOTE || 
           type == PROMPT_COMMENT || 
           type == PROMPT_PAREN)
    prompt_string = prompt_[1].c_str();
  else if (type == PROMPT_COPY)
    prompt_string = prompt_[2].c_str();
  else
    prompt_string = "? ";
  
  url_split(url_, driver, host, port, dbname);
  
  for (const char *cp = prompt_string; cp && *cp; cp++)
  {
    if (esc)
    {
      switch (*cp)
      {
        case '%':
          result += "%";
          break;
        case '/': /* Current database */
          if (conn_)
            result += dbname;
          break;
        case '~':
        {
          if (conn_)
          {
            if (dbname == username_)
              result += "~";
            else
              result += dbname;
          }
          break;
        }
        case 'M': /* DB server hostname (long/short) */
        case 'm':
          if (conn_)
          {
            if (!host.empty())
            {
              if (*cp == 'm')
                result += host.substr(0, host.find('.'));
              else
                result += host;
            }
            else
              result += "[local]";
          }
          break;
        case '>': /* DB server port number */
          if (conn_ && !port.empty())
            result += port;
          break;
        case 'n': /* DB server user name */
          if (conn_)
            result += username_;
          break;
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
        {
          long int l;
          char *end;
          char buf[32];
          
          l = strtol(cp, &end, 0);
          sprintf(buf, "%c", (unsigned char) l);
          result += buf;
          cp = end - 1;
          break;
        }
        case 'R':
          switch (type)
          {
            case PROMPT_READY:
              if (!conn_)
                result += '!';
#if 0 // FIXME
              else if (GetVariableBool(pset.vars, "SINGLELINE"))
                result += '^';
#endif
              else
                result += '=';
              break;
            case PROMPT_CONTINUE:
              result += '-';
              break;
            case PROMPT_SINGLEQUOTE:
              result += '\'';
              break;
            case PROMPT_DOUBLEQUOTE:
              result += '"';
              break;
            case PROMPT_COMMENT:
              result += '*';
              break;
            case PROMPT_PAREN:
              result += '(';
              break;
            default:
              break;
          }
          break;
        case '?':
          /* FIXME: implement */
          break;

        case '#':
        {
          if (0) // FIXME
            result += '#';
          else
            result += '>';

          break;
        }

        case '`': /* execute command */
        {
          FILE *fp = NULL;
          string filename = cp + 1;
          string::size_type cmdend = filename.find('`');
          
          if (cmdend == string::npos)
            cmdend = filename.size();
          filename.resize(cmdend);
          fp = popen(filename.c_str(), "r");
          if (fp)
          {
            char buf[128];
            fgets(buf, sizeof(buf), fp);
            pclose(fp);
            result += buf;
          }
          if (!result.empty() && result[result.size() - 1] == '\n')
            result.resize(result.size() - 1);
          cp += cmdend + 1;
          break;
        }
#if 0 // FIXME
        /* interpolate variable */
        case ':':
        {
          char	   *name;
          const char *val;
          int			nameend;

          name = strdup(p + 1);
          nameend = strcspn(name, ":");
          name[nameend] = '\0';
          val = GetVariable(pset.vars, name);
          if (val)
            strncpy(buf, val, MAX_PROMPT_SIZE);
          free(name);
          cp += nameend + 1;
          break;
        }
#endif

        default:
          result += *cp;
      }
      esc = false;
    }
    else if (*cp == '%')
      esc = true;
    else
    {
      result += *cp;
      esc = false;
    }
  }

  return result;
}

}
