#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "xmlparse.h"

#include "dataset.h"

static XML_Parser parser = NULL;
static void ItemFree(Item *item);
static void GroupFree(Group *group);
static void DataFree(Data *data);

static Dataset *DatasetCreate()
{
  Dataset *t = (Dataset *) (malloc(sizeof(Dataset)));
  t->id = NULL;
  t->groupCount = 0;
  t->itemCount = 0;
  t->groups = NULL;
  t->complete = 0;
  return t;
}

void freeDataset(Dataset *top)
{
  int i;
  free(top->id);
  for (i = 0; i < top->groupCount; i++) {
    GroupFree(top->groups[i]);
  }
  free(top->groups);
  free(top);
}

static void DatasetAddGroup(Dataset *t, Group *g)
{
  int i = t->groupCount++;
  t->groups = realloc(t->groups, sizeof(Group *) * t->groupCount);
  t->groups[i] = g;  
}

static Group *GroupCreate()
{
  Group *g = (Group *) (malloc(sizeof(Group)));
  g->id = NULL;
  g->itemCount = 0;
  g->items = NULL;
  return g;
}

static void GroupFree(Group *group)
{
  int i;
  free(group->id);
  for (i = 0; i < group->itemCount; i++) {
    ItemFree(group->items[i]);
  }
  free(group->items);
  free(group);
}

static void GroupAddItem(Group *g, Item *n)
{
  int i = g->itemCount++;
  g->items = realloc(g->items, sizeof(Item *) * g->itemCount);
  g->items[i] = n;
}

static Item *ItemCreate()
{
  Item *i = (Item *) (malloc(sizeof(Item)));
  i->id = NULL;
  i->dataCount = 0;
  i->datums = NULL;
  return i;
}

static void ItemFree(Item *item)
{
  int i;
  free(item->id);
  for (i = 0; i < item->dataCount; i++) {
    DataFree(item->datums[i]);
  }
  free(item->datums);
  free(item);
}

static void ItemAddData(Item *n, Data *d)
{
  int i = n->dataCount++;
  n->datums = realloc(n->datums, sizeof(Data *) * n->dataCount);
  n->datums[i] = d;
}

static Data *DataCreate()
{
  Data *i = (Data *) (malloc(sizeof(Data)));
  i->name = NULL;
  i->value = 0.0;
  return i;
}

static void DataFree(Data *data)
{
  int i;
  free(data->name);
  free(data);
}

static void startElement(void *userData, const char *name, const char **atts)
{
  int i;
  Dataset *top = (Dataset *) userData;

  if (strcmp(name, "datastream") == 0) {
  }
  else if (strcmp(name, "dataset") == 0) {
    for (;*atts; atts++) {
      if (strcmp(*atts, "id") == 0) {
	top->id = strdup(*(atts + 1));
      }
    }
  }
  else if (strcmp(name, "item") == 0) {
    Group *group = top->groups[top->groupCount - 1];
    Item *newItem = ItemCreate();
    
    for (;*atts; atts++) {
      if (strcmp(*atts, "id") == 0) {
	newItem->id = strdup(*(atts + 1));
      }
    }
    GroupAddItem(group, newItem);
    top->itemCount++;
  }
  else if (strcmp(name, "group") == 0) {
    Group *newGroup = GroupCreate();
    
    for (;*atts; atts++) {
      if (strcmp(*atts, "id") == 0) {
	newGroup->id = strdup(*(atts + 1));
      }
    }
    DatasetAddGroup(top, newGroup);
  }
  else {
    Group *group = top->groups[top->groupCount - 1];
    Item *item = group->items[group->itemCount - 1];
    Data *newData = DataCreate();
    newData->name = strdup(name);
    for (;*atts; atts++) {
      if (strcmp(*atts, "data") == 0) {
	newData->value = strtod(*(atts + 1), NULL);
      }
      else if (strcmp(*atts, "id") == 0) {
	free(newData->name);
	newData->name = strdup(*(atts +1));
      }
    }
    ItemAddData(item, newData);
  }
}

static void endElement(void *userData, const char *name)
{
  if (strcmp(name, "dataset") == 0) {
    Dataset *top = (Dataset *) userData;
    top->complete = 1;
  }
}

Dataset *getDataset(FILE *inputfp) {
  int done = 0;
  char buf[1000];
  static XML_Parser parser = NULL;
  Dataset *top = DatasetCreate();
  if (parser == NULL) {
    parser = XML_ParserCreate(NULL);
    XML_SetElementHandler(parser, startElement, endElement);
  }
  XML_SetUserData(parser, top);

  do {
    fgets(buf, sizeof(buf), inputfp);
    if (!XML_Parse(parser, buf, strlen(buf), done)) {
      fprintf(stderr,
              "%s at line %d\n%s\n",
              XML_ErrorString(XML_GetErrorCode(parser)),
              XML_GetCurrentLineNumber(parser),
	      buf);
      return NULL;
    }
  } while (!top->complete);
  return top;
}

#ifdef TESTING
int main(int argc, char **argv)
{
  int g, i, j;
  while (1) {
    Dataset *top = getDataset(stdin);
    printf("top %s\n", top->id);
    for (g = 0; g < top->groupCount; g++) {
      printf("  group %s\n", top->groups[g]->id);
      for (i = 0; i < top->groups[g]->itemCount; i++) {
	printf("    item %s\n",
	       top->groups[g]->items[i]->id);
	for (j = 0; j < top->groups[g]->items[i]->dataCount; j++) {
	  printf("      data %s %f\n", 
		 top->groups[g]->items[i]->datums[j]->name, 
		 top->groups[g]->items[i]->datums[j]->value);
	}
      }
    }
  }
}
#endif
