/*
 * Copyright 2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.codehaus.groovy.transform.powerassert;

import org.codehaus.groovy.runtime.typehandling.DefaultTypeTransformation;

import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

/**
 * Verifies that assertions hold, throwing a PowerAssertionError
 * otherwise. Only meant to be called from code generated by the transformation.
 *
 * @author Peter Niederwieser
 */
public abstract class AssertionVerifier {
    // used for code generation
    public static final String VERIFY_METHOD_NAME = "verify";

    // file to which debug output will be written; if null, no debug output will be written
    // NOTE: if debug output is enabled, a few tests in the Groovy project will fail
    // (e.g. due to missing JVM permission to write to the generated file)
    // it would ne nice to use a system property for this, but I didn't manage
    // to set system properties for tests in the Groovy build
    private static final String LOG_FILE = null;

    /**
     * Verifies that the given value is true according to Groovy truth,
     * throwing a PowerAssertionError with a cleaned-up stacktrace and
     * helpful detail message otherwise.
     *
     * @param truthValue the value of an assertion's truth expression
     * @param text       the source text of the assertion
     * @param recorder   a ValueRecorder used for rendering failed assertions
     */
    public static void verify(Object truthValue, String text, ValueRecorder recorder) {
        if (LOG_FILE != null) log(text, recorder);

        if (!DefaultTypeTransformation.castToBoolean(truthValue)) {
            String msg = AssertionRenderer.render(text, recorder);
            Error error = new PowerAssertionError(msg);
            filterStackTrace(error);
            throw error;
        }
    }

    private static void log(String text, ValueRecorder recorder) {
        Writer out = null;
        try {
            out = new FileWriter(LOG_FILE, true);
            out.write(AssertionRenderer.render(text, recorder));
            out.write("\n\n");
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            try {
                if (out != null) out.close();
            } catch (IOException ignored) {
            }
        }
    }

    /*
     * Removes all stacktrace elements prior to the element which points to the
     * failed assertion statement.
     */
    private static void filterStackTrace(Throwable error) {
        StackTraceElement[] stackTrace = error.getStackTrace();

        for (int i = 0; i < stackTrace.length; i++) {
            StackTraceElement elem = stackTrace[i];
            if (!elem.getClassName().startsWith("org.codehaus.groovy.transform.powerassert")
                    && !elem.getClassName().startsWith("org.codehaus.groovy.runtime.callsite")) {
                StackTraceElement[] newStackTrace = new StackTraceElement[stackTrace.length - i];
                System.arraycopy(stackTrace, i, newStackTrace, 0, newStackTrace.length);
                error.setStackTrace(newStackTrace);
                return;
            }
        }
    }
}
