//*****************************************************************************
//                              SimnNgSpice.cpp                               *
//                             -----------------                              *
// Started     : 07/05/2008                                                   *
// Last Update : 12/10/2009                                                   *
// Copyright   : (C) 2008 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "netlist/SimnNgSpice.hpp"

//*****************************************************************************
// Constructor.

SimnNgSpice::SimnNgSpice( void ) : SimnBase( )
{
  m_eSimEng = eSIMR_NGSPICE;

  // Initialize all object attributes
  bClear( );
}

//*****************************************************************************
// Destructor.

SimnNgSpice::~SimnNgSpice( )
{
}

//*****************************************************************************
// Extract the simulator engine and return TRUE if it's NG-Spice.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bExtractSimrEng( void )
{
  wxString  os1;
  size_t    sz1;

  // Scan the circuit description for simulator type
  for( sz1=0; sz1<NetList::m_osaNetLst.GetCount( ); sz1++ )
  {
    os1 = NetList::m_osaNetLst.Item( sz1 );
    if( os1.IsEmpty( ) ) continue;
    if( os1.Upper( ).Contains( wxT("NG-SPICE") ) ) return( TRUE );
  }

  return( FALSE );
}

//*****************************************************************************
// Extract all the simulator command lines from the circuit description.
// The following commands are currently detected and extracted :
//
//   .OPTIONS
//   .IC
//   .DC
//   .AC
//   .TRANSIENT
//   .PRINT
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bExtractSimCmds( void )
{
  bool      bRtnValue=FALSE;
  wxString  os1;
  size_t    sz1;

  // Scan the circuit description for simulation commands
  for( sz1=0; sz1<NetList::m_osaNetLst.GetCount( ); sz1++ )
  {
    os1 = NetList::m_osaNetLst.Item( sz1 );

    if( os1.IsEmpty( ) )               continue;
    if( ! os1.StartsWith( wxT(".") ) ) continue;

    os1.MakeUpper( );

    if(      os1.StartsWith( wxT(".OPT") ) ) // OPTIONS command
      m_oCmdOPT.bSetString( NetList::m_osaNetLst.Item( sz1 ) );
    else if( os1.StartsWith( wxT(".DC")  ) ) // DC command
      m_oCmdDC .bSetString( NetList::m_osaNetLst.Item( sz1 ) );
    else if( os1.StartsWith( wxT(".AC")  ) ) // AC command
      m_oCmdAC .bSetString( NetList::m_osaNetLst.Item( sz1 ) );
    else if( os1.StartsWith( wxT(".TR")  ) ) // TRANSIENT command
      m_oCmdTR .bSetString( NetList::m_osaNetLst.Item( sz1 ) );
    else if( os1.StartsWith( wxT(".PR")  ) ) // PRINT command
      m_oCmdPR .bSetString( NetList::m_osaNetLst.Item( sz1 ) );
    else continue;

    bRtnValue = TRUE;
  }

  return( bRtnValue );
}

//*****************************************************************************
// Extract the source component.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bExtractSigSrc( void )
{
  // Search for and extract a signal source component
  if( ! SimnBase::bExtractSigSrc( ) ) return( TRUE );

  // Set the NG-Spice Independent Source structure
  (Component &) m_oCpntIndSrc = m_oCpntSwpSrc;
  if( ! m_oCpntIndSrc.bParse( ) )     return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Clear all object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bClear( void )
{
  // Clear the command object pointer attributes
  bClrCmds( );

  // Clear the base class
  return( SimnBase::bClear( ) );
}

//*****************************************************************************
// Clear all simulator command objects.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bClrCmds( void )
{
  m_oCmdOPT.bClear( );
  m_oCmdDC .bClear( );
  m_oCmdAC .bClear( );
  m_oCmdTR .bClear( );
  m_oCmdPR .bClear( );

  return( TRUE );
}

//*****************************************************************************
// Clear all test points ie. nodes and components.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bClrTstPts( void )
{
  m_oCmdPR.m_osaNodes.Empty( );
  m_oCmdPR.m_osaCpnts.Empty( );

  return( TRUE );
}

//*****************************************************************************
// Do the current simulation settings constitute a valid simulation?
//
// Return Values :
//   TRUE  - Valid
//   FALSE - Not valid

bool  SimnNgSpice::bValidate( void )
{
  // Validate the base class
  SimnBase::bValidate( );

  if( ! m_oCmdOPT.bIsValid( ) )    SetErrMsg( m_oCmdOPT.rosGetErrMsg( ) );
  if( ! m_oCmdPR .bIsValid( ) )    SetErrMsg( m_oCmdPR .rosGetErrMsg( ) );

  switch( eGetAnaType( ) )
  {
    case eCMD_DC :
      if( ! m_oCmdDC.bIsValid( ) ) SetErrMsg( m_oCmdDC .rosGetErrMsg( ) );
      break;

    case eCMD_AC :
      if( ! m_oCmdAC.bIsValid( ) ) SetErrMsg( m_oCmdAC .rosGetErrMsg( ) );
      break;

    case eCMD_TR :
      if( ! m_oCmdTR.bIsValid( ) ) SetErrMsg( m_oCmdTR .rosGetErrMsg( ) );
      break;

    default :
      SetErrMsg( wxT("No valid analysis command exists.") );
  }

  return( bIsValid( ) );
}

//*****************************************************************************
// Load a simulation to file.
//
// Argument List :
//   rosFName - The name of the file to be loaded
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bLoadFile( const wxString & rosFName )
{
  // Envoke the base class load operations
  return( SimnBase::bLoadFile( rosFName ) );
}

//*****************************************************************************
// Save (or resave) the simulation to file.
//
// Argument List :
//   rosFName - The name of the file to be saved
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bSaveFile( const wxString & rosFName )
{
  wxString  os1;

  // Save the netlist to file
  if( ! SimnBase::bSaveFile( rosFName ) ) return( FALSE );

  // Open the file
  wxTextFile  oFileCct( m_ofnSaveFile.GetFullPath( ) );
  if( ! oFileCct.Open( ) )               return( FALSE );

  // Temporarily remove the circuit description terminator ie. ".END"
  oFileCct.RemoveLine( oFileCct.GetLineCount( )-1 );

  // Append the simulation command lines to the end of the file
  oFileCct.AddLine( wxT("* NG-Spice Simulation Commands") );
  oFileCct.AddLine( m_oCmdOPT );
  oFileCct.AddLine( m_oCmdPR );
  if( m_oCmdDC.bIsValid( ) ) oFileCct.AddLine( m_oCmdDC );
  if( m_oCmdAC.bIsValid( ) ) oFileCct.AddLine( m_oCmdAC );
  if( m_oCmdTR.bIsValid( ) ) oFileCct.AddLine( m_oCmdTR );
  oFileCct.AddLine( wxT("") );

  // Add the circuit description terminator
  oFileCct.AddLine( wxT(".END") );
  oFileCct.AddLine( wxT("") );

  // Save the changes to disk
  oFileCct.Write( );
  oFileCct.Close( );

  return( TRUE );
}

//*****************************************************************************
// Add a node to the list of test points.
//
// Argument List :
//   rosNode - The name of the node to be added to the list of test points
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bAddTstNode( const wxString & rosName )
{
  // Is the node name valid?
  if( rosName.IsEmpty( ) )                                 return( FALSE );
  // Is the node name already in the test node list?
  if( rosaGetTstNodes( ).Index( rosName ) != wxNOT_FOUND ) return( TRUE );

  // Add the node name to the list of test points
  m_oCmdPR.m_osaNodes.Add( rosName );
  m_oCmdPR.m_osaNodes.Sort( &iStrCmp );

  return( TRUE );
}

//*****************************************************************************
// Add a component to the list of test points.
//
// Argument List :
//   rosName - The name of the component to be added to the list of test points
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnNgSpice::bAddTstCpnt( const wxString & rosName )
{
  Component  oCpnt;
  size_t     sz1;
  wxString   os1;

  // Is the component name valid?
  if( rosName.IsEmpty( ) )                             return( FALSE );

  // Get the component object with this name
  for( sz1=0; sz1<NetList::m_oaCpnts.GetCount( ); sz1++ )
  {
    oCpnt = NetList::m_oaCpnts.Item( sz1 );
    if( oCpnt.m_osName == rosName ) break;
  }
  if( sz1 >= NetList::m_oaCpnts.GetCount( ) )          return( FALSE );

  // Is the node pair already in the test component list?
  if( oCpnt.m_osaNodes.GetCount( ) != 2 )              return( FALSE );
  os1 = oCpnt.m_osaNodes.Item( 0 ) + wxT(",") + oCpnt.m_osaNodes.Item( 1 );
  if( rosaGetTstCpnts( ).Index( os1 ) != wxNOT_FOUND ) return( TRUE );

  // Add the node pair to the list of test points
  m_oCmdPR.m_osaCpnts.Add( os1 );
  m_oCmdPR.m_osaCpnts.Sort( &iStrCmp );

  return( TRUE );
}

//*****************************************************************************
// Get an wxArrayString object containing a list of the test component labels.
//
// Return Values :
//   A reference to a list of test components name

const wxArrayString & SimnNgSpice::rosaGetTstCpnts( void )
{
  static  wxArrayString  osaTstCpnts;
  Component  oCpnt1;
  wxString   os1, os2;
  size_t     sz1, sz2;

  // Clear the test component list
  osaTstCpnts.Clear( );

  // Convert the PRINT command node pairs to component labels
  for( sz1=0; sz1<m_oCmdPR.m_osaCpnts.GetCount( ); sz1++ )
  {
    // Get a node pair from the list of test components
    os1 = m_oCmdPR.m_osaCpnts.Item( sz1 );

    // Match the node pair to a component
    for( sz2=0; sz2<NetList::m_oaCpnts.GetCount( ); sz2++ )
    {
      oCpnt1 = NetList::m_oaCpnts.Item( sz2 );
      os2 = oCpnt1.m_osaNodes.Item(0) + wxT(',') + oCpnt1.m_osaNodes.Item(1);
      if( os1 == os2 )
      {
        // Add the component name to the test component list
        osaTstCpnts.Add( oCpnt1.m_osName );
        break;
      }
    }
  }

  return( osaTstCpnts );
}

//*****************************************************************************
// Derive a list of column labels for the NG-Spice results file.
// NG-Spice only accepts node/s as probe points. To specify a component a node
// pair must be provided. To improve the readability of the simulation results
// gSpiceUI substitutes these node pairs with component labels.
//
// Return Values :
//   A reference to a wxString containing the results file column labels

const wxString & SimnNgSpice::rosGetColLbls( void )
{
  static  wxString  osColLbls;
  CmdNgSpicePR  oCmdPR;
  wxString      osNodes;
  size_t        sz1, sz2;

  osColLbls.Empty( );

  // Check if anything can be done
  if( !m_oCmdPR.m_osaCpnts.IsEmpty( ) && !NetList::m_oaCpnts.IsEmpty( ) )
  {
    oCmdPR = m_oCmdPR;

    // Replace PRINT command node pairs with corresponding component names
    for( sz1=0; sz1<oCmdPR.m_osaCpnts.GetCount( ); sz1++ )
    {
      osNodes = oCmdPR.m_osaCpnts.Item( sz1 );

      // Find a component name for a node pair
      for( sz2=0; sz2<NetList::m_oaCpnts.GetCount( ); sz2++ )
      {
        Component & roCpnt1 = NetList::m_oaCpnts.Item( sz2 );

        if( roCpnt1.m_osaNodes.GetCount( ) != 2 ) continue;

        // Test if a component's nodes match the PRINT command node pair
        if( roCpnt1.rosGetNodes( ) == osNodes )
        {
          oCmdPR.m_osaCpnts.Item( sz1 ) = roCpnt1.m_osName;
          break;
        }
      }
    }

    // Create the results file column labels
    oCmdPR.bFormat( );
    osColLbls = oCmdPR   .AfterFirst( wxT(' ') );
    osColLbls = osColLbls.AfterFirst( wxT(' ') );
  }

  return( osColLbls );
}

//*****************************************************************************
// Copy the contents of a SimnGnuCap object.
//
// Argument List :
//   roSimn - A reference to a SimnGnuCap object
//
// Return Values :
//   A reference to this object

SimnNgSpice & SimnNgSpice::operator = ( const SimnGnuCap & roSimn )
{
  (SimnBase &) *this = (SimnBase &) roSimn;

  m_oCmdOPT     = roSimn.m_oCmdOPT;
  m_oCmdPR      = roSimn.m_oCmdPR;
  m_oCpntIndSrc = roSimn.m_oCmdGEN;

  switch( roSimn.eGetAnaType( ) )
  {
    case eCMD_OP :
      m_oCmdDC = roSimn.m_oCmdOP;
      break;

    case eCMD_DC :
      m_oCmdDC = roSimn.m_oCmdDC;
      break;

    case eCMD_AC :
      m_oCmdAC = roSimn.m_oCmdAC;
      break;

    case eCMD_TR :
      m_oCmdTR = roSimn.m_oCmdTR;
      break;

    default :
      break;
  }

  return( *this );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  SimnNgSpice::Print( const wxString & rosPrefix )
{
  SimnBase::Print( rosPrefix + wxT("SimnBase::")  );
  m_oCmdOPT.Print( rosPrefix + wxT("m_oCmdOPT::") );
  m_oCmdDC .Print( rosPrefix + wxT("m_oCmdDC::")  );
  m_oCmdAC .Print( rosPrefix + wxT("m_oCmdAC::")  );
  m_oCmdTR .Print( rosPrefix + wxT("m_oCmdTR::")  );
  m_oCmdPR .Print( rosPrefix + wxT("m_oCmdPR::")  );
}

//*****************************************************************************
