//*****************************************************************************
//                             PnlNgSpiceTR.cpp                               *
//                            ------------------                              *
// Started     : 05/08/2004                                                   *
// Last Update : 20/08/2009                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/panels/PnlNgSpiceTR.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlNgSpiceTR, PnlAnaBase )

  EVT_CHOICE( ID_CHO_SRCNAME, PnlNgSpiceTR::OnChoSrcName )
  EVT_BUTTON( ID_BTN_SETUP,   PnlNgSpiceTR::OnBtnSetup   )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlNgSpiceTR::PnlNgSpiceTR( wxWindow * poWin )
                          : PnlAnaBase( poWin ), m_oDlgCfgSrc( poWin )
{
  bSetSimrEng( eSIMR_NGSPICE );
  bSetAnaType( eCMD_TR );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlNgSpiceTR::~PnlNgSpiceTR( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlNgSpiceTR::Create( void )
{
  wxStaticBox * poSbxSigSrc;
  int           ix, iy;

  // Disable the checkboxes for the parameters NG-Spice cannot calculate
  m_oCbxCurrent.Disable( );
  m_oCbxPower  .Disable( );
  m_oCbxResist .Disable( );

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" Transient Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Time") );
  m_oPnlStop   .bSetName( wxT("Stop Time") );
  m_oPnlStep   .bSetName( wxT("Step Increment") );

  // Set sweep parameter units
  m_oPnlStart.bSetUnitsType( eUNITS_TIME );
  m_oPnlStop .bSetUnitsType( eUNITS_TIME );
  m_oPnlStep .bSetUnitsType( eUNITS_TIME );

  // Create and add generator controls
  poSbxSigSrc = new wxStaticBox( this, ID_UNUSED, wxT(" Signal Source "),
                        wxPoint( 6, 175 ),   wxSize( 235, 55 ) );
  m_oChoSrcName.Create( this, ID_CHO_SRCNAME,
                        wxPoint( 18, 195 ),  wxSize( 107, -1 ) );
  m_oBtnSetup  .Create( this, ID_BTN_SETUP, wxT("Setup ..."),
                        wxPoint( 130, 195 ), wxSize( 100, 29 ) );
  poSbxSigSrc->SetOwnFont( FONT_SLANT );

  PnlAnaBase::CreateInitC( ); // Create and add the initial conditions controls
  PnlAnaBase::CreateTemp( );  // Create the analysis temperature controls
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceTR::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )    bRtn = FALSE;

  // Set the sweep parameters to their defaults
  m_oPnlStart.bSetValue( (float) 0.0 );
  m_oPnlStep .bSetValue( (float) 1.0 );
  m_oPnlStart.bSetUnits( wxT("msec") );
  m_oPnlStop .bSetUnits( wxT("msec") );
  m_oPnlStep .bSetUnits( wxT("msec") );

  // Set default initial condition values
  if( ! bSetInitC( eINITC_WARM ) ) bRtn = FALSE;

  // Set input source default values
  m_oChoSrcName.Clear( );
  m_oChoSrcName.Append( wxT("None") );
  m_oChoSrcName.SetSelection( 0 );

  // Clear the generator dialog
  m_oDlgCfgSrc.bClear( );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSimn - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceTR::bLoad( SimnNgSpice & roSimn )
{
  bool  bRtn=TRUE;

  // Load the components into the signal source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VI") );

  // Set the source component name
  if( ! PnlAnaBase::bSetSrcCpnt( roSimn.m_oCpntSwpSrc ) )    bRtn = FALSE;

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdTR.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdTR.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdTR.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  // Set the initial conditions
  if( ! bSetInitC( (eInitCType) roSimn.m_oCmdTR.m_eInitC ) ) bRtn = FALSE;

  // Set the initial state of the source setup dialog
  m_oDlgCfgSrc.bSetValues( roSimn.m_oCpntIndSrc );

  // Set the analysis temperature
  if( ! m_oPnlTemp.bSetValue( roSimn.m_oCmdOPT.m_osTEMP ) )  bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
// (Prior to it being passed to a simulator object.)
//
// Argument List :
//   roSimn - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlNgSpiceTR::bSave( SimnNgSpice & roSimn )
{
  wxString  os1;

  m_osErrMsg.Empty( );

  // Set the sweep values
  roSimn.m_oCmdTR.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdTR.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdTR.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the initial conditions
  roSimn.m_oCmdTR.m_eInitC = (eInitCType) m_oRbxInitC.GetSelection( );

  // Set the component to be used as a signal source
  if( m_oChoSrcName.GetStringSelection( ) != wxT("None") )
  {
    // Get the unmodified signal source component
    os1 = m_oChoSrcName.GetStringSelection( );
    (Component &) roSimn.m_oCpntIndSrc = roSimn.roGetCpnt( os1 );
    // Add the modifications from the independent source configuration dialog
    m_oDlgCfgSrc.bGetValues( roSimn.m_oCpntIndSrc );
    if( roSimn.m_oCpntIndSrc.bFormat( ) )
      roSimn.m_oCpntSwpSrc = roSimn.m_oCpntIndSrc;
    else
    {
      os1 = wxT("The signal source ")
          + roSimn.m_oCpntIndSrc.rosGetErrMsg( ) + wxT(".   ");
      SetErrMsg( os1 );
    }
  }
  else
  {
    roSimn.m_oCpntSwpSrc.bClear( );
    roSimn.m_oCpntIndSrc.bClear( );
  }

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_TR;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Set the analysis temperature
  roSimn.m_oCmdOPT.m_osTEMP = m_oPnlTemp.rosGetValue( );

  // Create the command strings
  roSimn.m_oCmdTR.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdTR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdTR.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Generator component choice control event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceTR::OnChoSrcName( wxCommandEvent & roEvtCmd )
{
  if( m_oChoSrcName.GetStringSelection( ) != wxT("None") )
  {
    wxString  os1 = m_oChoSrcName.GetStringSelection( );
    m_oDlgCfgSrc.bSetName( os1 );
  }
}

//*****************************************************************************
// Generator component setup button control event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  PnlNgSpiceTR::OnBtnSetup( wxCommandEvent & roEvtCmd )
{
  m_oDlgCfgSrc.CenterOnParent( );
  m_oDlgCfgSrc.ShowModal( );
}

//*****************************************************************************
