/* CleanIce theme for gtk, based on the ThinIce and Clean GTK+ Engines.
   Author:  Rodney Dawes <dobez@fnmail.com>
   ThinIce Authors: Tim Gerla <timg@means.net>
            Tomas gren <stric@ing.umu.se>
   Clean Author: dengen40@yahoo.com
*/ 

#include "cleanice_theme.h"
#include <math.h>

#define DETAIL(xx) ((detail) && (!strcmp(xx, detail)))

#define BROKEN_MACH64

#define SET_GC_OUT(th_data)				\
    if (((ThemeStyleData *)th_data)->black_and_white)	\
    {							\
	gc1 = style->white_gc;				\
	gc2 = style->black_gc;				\
    }							\
    else						\
    {							\
	gc1 = style->light_gc[state_type];		\
	gc2 = style->dark_gc[state_type];		\
    }

#define SET_GC_IN(th_data)				\
    if (((ThemeStyleData *)th_data)->black_and_white)	\
    {							\
	gc1 = style->black_gc;				\
	gc2 = style->white_gc;				\
    }							\
    else						\
    {							\
	gc1 = style->dark_gc[state_type];		\
	gc2 = style->light_gc[state_type];		\
    }

extern GtkStyleClass thinice_default_class;

/* internal functions */
static void         draw_hline(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x1,
			       gint x2,
			       gint y);
static void         draw_vline(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint y1,
			       gint y2,
			       gint x);
static void         draw_shadow(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height);

static void         draw_polygon(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 GdkPoint * point,
				 gint npoints,
				 gint fill);
static void         draw_arrow(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       GtkArrowType arrow_type,
			       gint fill,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_diamond(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 gint x,
				 gint y,
				 gint width,
				 gint height);
static void         draw_oval(GtkStyle * style,
			      GdkWindow * window,
			      GtkStateType state_type,
			      GtkShadowType shadow_type,
			      GdkRectangle * area,
			      GtkWidget * widget,
			      gchar * detail,
			      gint x,
			      gint y,
			      gint width,
			      gint height);
static void         draw_string(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				const gchar * string);
static void         draw_box(GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x,
			     gint y,
			     gint width,
			     gint height);
static void         draw_flat_box(GtkStyle * style,
				  GdkWindow * window,
				  GtkStateType state_type,
				  GtkShadowType shadow_type,
				  GdkRectangle * area,
				  GtkWidget * widget,
				  gchar * detail,
				  gint x,
				  gint y,
				  gint width,
				  gint height);
static void         draw_check(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_option(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height);
static void         draw_cross(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_ramp(GtkStyle * style,
			      GdkWindow * window,
			      GtkStateType state_type,
			      GtkShadowType shadow_type,
			      GdkRectangle * area,
			      GtkWidget * widget,
			      gchar * detail,
			      GtkArrowType arrow_type,
			      gint x,
			      gint y,
			      gint width,
			      gint height);
static void         draw_tab(GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x,
			     gint y,
			     gint width,
			     gint height);
static void         draw_shadow_gap(GtkStyle * style,
				    GdkWindow * window,
				    GtkStateType state_type,
				    GtkShadowType shadow_type,
				    GdkRectangle * area,
				    GtkWidget * widget,
				    gchar * detail,
				    gint x,
				    gint y,
				    gint width,
				    gint height,
				    GtkPositionType gap_side,
				    gint gap_x,
				    gint gap_width);
static void         draw_box_gap(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 gint x,
				 gint y,
				 gint width,
				 gint height,
				 GtkPositionType gap_side,
				 gint gap_x,
				 gint gap_width);
static void         draw_extension(GtkStyle * style,
				   GdkWindow * window,
				   GtkStateType state_type,
				   GtkShadowType shadow_type,
				   GdkRectangle * area,
				   GtkWidget * widget,
				   gchar * detail,
				   gint x,
				   gint y,
				   gint width,
				   gint height,
				   GtkPositionType gap_side);
static void         draw_focus(GtkStyle * style,
			       GdkWindow * window,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_slider(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height,
				GtkOrientation orientation);
static void         draw_handle(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height,
				GtkOrientation orientation);

static void thinice_slash_one(GdkWindow *window,
				GdkGC *gc1,
				GdkGC *gc2,
				int width,
				int height,
				int x,
				int y);
static void thinice_slash_two(GdkWindow *window,
				GdkGC *gc1,
				GdkGC *gc2,
				int width,
				int height,
				int x,
				int y);
static void	draw_dimple(GtkStyle *style,
				GdkWindow *window,
				GtkStateType state_type,
				int width,
				int height,
				int x,
				int y,
				gboolean horizontal);
static void thinice_tab(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height);

/* internal data structs */

GtkStyleClass       thinice_default_class =
{
  2,
  2,
  draw_hline,
  draw_vline,
  draw_shadow,
  draw_polygon,
  draw_arrow,
  draw_diamond,
  draw_oval,
  draw_string,
  draw_box,
  draw_flat_box,
  draw_check,
  draw_option,
  draw_cross,
  draw_ramp,
  draw_tab,
  draw_shadow_gap,
  draw_box_gap,
  draw_extension,
  draw_focus,
  draw_slider,
  draw_handle
};

static void
draw_hline(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x1,
	   gint x2,
	   gint y)
{
  gint                thickness_light;
  gint                thickness_dark;
  gint                i;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  thickness_light = style->klass->ythickness / 2;
  thickness_dark = style->klass->ythickness - thickness_light;

  if (area)
  {
    gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
    gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
  }
  for (i = 0; i < thickness_dark; i++)
  {
    gdk_draw_line(window, style->light_gc[state_type], x2 - i - 1, y + i, x2, y + i);
    gdk_draw_line(window, style->dark_gc[state_type], x1, y + i, x2 - i - 1, y + i);
  }

  y += thickness_dark;
  for (i = 0; i < thickness_light; i++)
  {
    gdk_draw_line(window, style->dark_gc[state_type], x1, y + i, x1 + thickness_light - i - 1, y + i);
    gdk_draw_line(window, style->light_gc[state_type], x1 + thickness_light - i - 1, y + i, x2, y + i);
  }
  if (area)
  {
    gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
  }
}

static void
draw_vline(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint y1,
	   gint y2,
	   gint x)
{
  gint                thickness_light;
  gint                thickness_dark;
  gint                i;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  thickness_light = style->klass->xthickness / 2;
  thickness_dark = style->klass->xthickness - thickness_light;

  if (area)
  {
    gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
    gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
  }
  for (i = 0; i < thickness_dark; i++)
  {
    gdk_draw_line(window, style->light_gc[state_type], x + i, y2 - i - 1, x + i, y2);
    gdk_draw_line(window, style->dark_gc[state_type], x + i, y1, x + i, y2 - i - 1);
  }

  x += thickness_dark;
  for (i = 0; i < thickness_light; i++)
  {
    gdk_draw_line(window, style->dark_gc[state_type], x + i, y1, x + i, y1 + thickness_light - i);
    gdk_draw_line(window, style->light_gc[state_type], x + i, y1 + thickness_light - i, x + i, y2);
  }
  if (area)
  {
    gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
    gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
  }
}

static void
draw_shadow(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height)
{
  GdkGC              *gc1 = NULL;
  GdkGC              *gc2 = NULL;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  switch (shadow_type)
    {
    case GTK_SHADOW_NONE:
      return;
    case GTK_SHADOW_IN:
    case GTK_SHADOW_ETCHED_IN:
      gc1 = style->light_gc[state_type];
      gc2 = style->dark_gc[state_type];
      break;
    case GTK_SHADOW_OUT:
    case GTK_SHADOW_ETCHED_OUT:
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      break;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, area);
      gdk_gc_set_clip_rectangle(gc2, area);
      if ((shadow_type == GTK_SHADOW_IN) ||
	  (shadow_type == GTK_SHADOW_OUT))
	{
	  gdk_gc_set_clip_rectangle(style->black_gc, area);
	  gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
	}
    }
  switch (shadow_type)
    {
    case GTK_SHADOW_NONE:
      break;

    case GTK_SHADOW_ETCHED_IN:
	gdk_draw_rectangle(window, gc2, FALSE, x, y, width-1, height-1);
/*	gdk_draw_rectangle(window, gc2, FALSE, x+1, y+1, width-3, height-3); */

	break;
    case GTK_SHADOW_IN:
	gdk_draw_line(window, gc2,
		      x, y, x + width-1, y);
	gdk_draw_line(window, gc2,
		      x, y, x, y + height-1);
	
	gdk_draw_line(window, gc1,
		      x, y + height-1, x + width-1, y + height-1);
	gdk_draw_line(window, gc1,
		      x + width-1, y, x + width-1, y + height-1);
	break;
	
    case GTK_SHADOW_ETCHED_OUT:
	gdk_draw_rectangle(window, gc2, FALSE, x, y, width-1, height-1);

	break;
    case GTK_SHADOW_OUT:
	gdk_draw_line(window, gc2,
		      x, y, x + width-1, y);
	gdk_draw_line(window, gc2,
		      x, y, x, y + height-1);
	
	gdk_draw_line(window, gc1,
		      x, y + height-1, x + width-1, y + height-1);
	gdk_draw_line(window, gc1,
		      x + width-1, y, x + width-1, y + height-1);
	
	break;
    }
  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, NULL);
      gdk_gc_set_clip_rectangle(gc2, NULL);
      if ((shadow_type == GTK_SHADOW_IN) ||
	  (shadow_type == GTK_SHADOW_OUT))
	{
	  gdk_gc_set_clip_rectangle(style->black_gc, NULL);
	  gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
	}
    }
}

static void
draw_polygon(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     GdkPoint * points,
	     gint npoints,
	     gint fill)
{
#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif /* M_PI */
#ifndef M_PI_4
#define M_PI_4  0.78539816339744830962
#endif /* M_PI_4 */

  static const gdouble pi_over_4 = M_PI_4;
  static const gdouble pi_3_over_4 = M_PI_4 * 3;

  GdkGC              *gc1;
  GdkGC              *gc2;
  GdkGC              *gc3;
  GdkGC              *gc4;
  gdouble             angle;
  gint                xadjust;
  gint                yadjust;
  gint                i;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
  g_return_if_fail(points != NULL);

  switch (shadow_type)
    {
    case GTK_SHADOW_IN:
      gc1 = style->light_gc[state_type];
      gc2 = style->dark_gc[state_type];
      gc3 = style->light_gc[state_type];
      gc4 = style->dark_gc[state_type];
      break;
    case GTK_SHADOW_ETCHED_IN:
      gc1 = style->light_gc[state_type];
      gc2 = style->dark_gc[state_type];
      gc3 = style->dark_gc[state_type];
      gc4 = style->light_gc[state_type];
      break;
    case GTK_SHADOW_OUT:
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      gc3 = style->dark_gc[state_type];
      gc4 = style->light_gc[state_type];
      break;
    case GTK_SHADOW_ETCHED_OUT:
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      gc3 = style->light_gc[state_type];
      gc4 = style->dark_gc[state_type];
      break;
    default:
      return;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, area);
      gdk_gc_set_clip_rectangle(gc2, area);
      gdk_gc_set_clip_rectangle(gc3, area);
      gdk_gc_set_clip_rectangle(gc4, area);
    }

  if (fill)
    gdk_draw_polygon(window, style->bg_gc[state_type], TRUE, points, npoints);

  npoints--;

  for (i = 0; i < npoints; i++)
    {
      if ((points[i].x == points[i + 1].x) &&
	  (points[i].y == points[i + 1].y))
	{
	  angle = 0;
	}
      else
	{
	  angle = atan2(points[i + 1].y - points[i].y,
			points[i + 1].x - points[i].x);
	}

      if ((angle > -pi_3_over_4) && (angle < pi_over_4))
	{
	  if (angle > -pi_over_4)
	    {
	      xadjust = 0;
	      yadjust = 1;
	    }
	  else
	    {
	      xadjust = 1;
	      yadjust = 0;
	    }

	  gdk_draw_line(window, gc1,
			points[i].x - xadjust, points[i].y - yadjust,
			points[i + 1].x - xadjust, points[i + 1].y - yadjust);
	  gdk_draw_line(window, gc3,
			points[i].x, points[i].y,
			points[i + 1].x, points[i + 1].y);
	}
      else
	{
	  if ((angle < -pi_3_over_4) || (angle > pi_3_over_4))
	    {
	      xadjust = 0;
	      yadjust = 1;
	    }
	  else
	    {
	      xadjust = 1;
	      yadjust = 0;
	    }

	  gdk_draw_line(window, gc4,
			points[i].x + xadjust, points[i].y + yadjust,
			points[i + 1].x + xadjust, points[i + 1].y + yadjust);
	  gdk_draw_line(window, gc2,
			points[i].x, points[i].y,
			points[i + 1].x, points[i + 1].y);
	}
    }
  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, NULL);
      gdk_gc_set_clip_rectangle(gc2, NULL);
      gdk_gc_set_clip_rectangle(gc3, NULL);
      gdk_gc_set_clip_rectangle(gc4, NULL);
    }
}

static void
thinice_slash_two(GdkWindow *window, GdkGC *gc1, GdkGC *gc2,
                  int width, int height, int x, int y)
{
    int centerx, centery, thick;
    int ax1=0,ax2=0,ay1=0,ay2=0;
    
    centerx = (width - 1)/2 + x;
    centery = (height - 1)/2 + y;
    if (width > height) {
        ax1 = -2; ax2 = 1;
    } else {
        ay1 = -2; ay2 = 1;
    }
    
    thick = ((width < height?width-1:height-1) >> 1) - 2;
    gdk_draw_line(window, gc2,
            centerx - thick + ax1, centery + thick + ay1,
            centerx + thick + ax1, centery - thick + ay1);
    gdk_draw_line(window, gc1,
            centerx - thick + ax1 + ax2, centery + thick + ay1 + ay2,
            centerx + thick + ax1 + ax2, centery - thick + ay1 + ay2);
    if (width > height) {
        ax1 = 2; /* ax2 = 1; */
    } else {
        ay1 = 2; /* ay2 = 1; */
    }
    gdk_draw_line(window, gc2,
            centerx - thick + ax1, centery + thick + ay1,
            centerx + thick + ax1, centery - thick + ay1);
    gdk_draw_line(window, gc1,
            centerx - thick + ax1 + ax2, centery + thick + ay1 + ay2,
            centerx + thick + ax1 + ax2, centery - thick + ay1 + ay2);
}

static void
thinice_slash_one(GdkWindow *window, GdkGC *gc1, GdkGC *gc2,
                  int width, int height, int x, int y)
{
    int centerx, centery, thick;
    
    centerx = (width - 1)/2 + x;
    centery = (height - 1)/2 + y;
    
    thick = ((width < height?width-1:height-1) >> 1) - 2;
    gdk_draw_line(window, gc2,
            centerx - thick, centery + thick,
            centerx + thick, centery - thick);
    gdk_draw_line(window, gc1,
            centerx - thick, centery + thick - 1,
            centerx + thick - 1, centery - thick);
}

static void
draw_dimple(GtkStyle * style,
		   GdkWindow * window,
		   GtkStateType state_type,
		   int width,
		   int height,
		   int x,
		   int y,
		   gboolean horizontal)
{
    int centerx;
    int centery;
    GdkGC *gc1 = NULL;
    GdkGC *gc2 = NULL;

    centerx = (width - 1) / 2 + x;
    centery = (height - 1) / 2 + y;

    SET_GC_OUT(style->engine_data);
    if (horizontal)
    {
	gdk_draw_point(window, gc2, centerx, centery);
	gdk_draw_point(window, gc1, centerx + 1, centery);
    }
    else
    {
	gdk_draw_point(window, gc2, centerx, centery);
	gdk_draw_point(window, gc1, centerx, centery + 1);
    }
}

static void
draw_arrow(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   GtkArrowType arrow_type,
	   gint fill, gint x, gint y, gint width, gint height)
{
    GdkGC *gc1;
    GdkGC *gc2;
    gint as;			// arrow size
    gint xl, xm, xr;		// left, mid, right
    gint yt, ym, yb;		// top,  mid, bottom
    GdkPoint points[4];
    ThemeStyleData *th_data = style->engine_data;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

#ifdef DEBUG
    printf("%-17s %-15s\n", "draw_arrow", detail);
#endif
    if ((width == -1) && (height == -1))
	gdk_window_get_size(window, &width, &height);
    else if (width == -1)
	gdk_window_get_size(window, &width, NULL);
    else if (height == -1)
	gdk_window_get_size(window, NULL, &height);

    if (DETAIL("menuitem"))
    {
	/* draw indented triangle */
	if (th_data->small_menu_arrows)
	{
	    points[0].x = x;
	    points[0].y = y + 1;
	    points[1].x = x + (height - 2) / 2;
	    points[1].y = y + height / 2;
	    points[2].x = x;
	    points[2].y = y + height - 1;
	}
	else
	{
	    points[0].x = x;
	    points[0].y = y;
	    points[1].x = x + (height) / 2;
	    points[1].y = y + height / 2;
	    points[2].x = x;
	    points[2].y = y + height;
	}
	gdk_draw_polygon(window, style->bg_gc[GTK_STATE_ACTIVE], TRUE, points,
			 3);
	gdk_draw_line(window, style->black_gc, points[0].x, points[0].y,
		      points[2].x, points[2].y);
	gdk_draw_line(window, style->black_gc, points[0].x, points[0].y,
		      points[1].x, points[1].y);
	gdk_draw_line(window, style->white_gc, points[2].x, points[2].y,
		      points[1].x, points[1].y);
    }
    else
    {
      if(shadow_type && !DETAIL("spinbutton")
	 && !DETAIL("scrollbar") &&
	 !(detail && !strcmp(&detail[1], "scrollbar"))) {
	shadow_type = GTK_SHADOW_NONE;
      }
      /*
      if(DETAIL("spinbutton")) {
	shadow_type = GTK_SHADOW_ETCHED_OUT;
      }
      */
      switch (shadow_type)
	{
	case GTK_SHADOW_IN:
	case GTK_SHADOW_ETCHED_IN:
	  gc1 = style->white_gc;
	  gc2 = style->dark_gc[state_type];
	  break;
	
	case GTK_SHADOW_OUT:
	case GTK_SHADOW_ETCHED_OUT:
	  gc1 = style->black_gc;
	  gc2 = style->light_gc[state_type];
	  break;

	case GTK_SHADOW_NONE:
	  gc1 = style->black_gc;
	  gc2 = style->black_gc;
	  break;

	default:
	  return;
	}
	 
      if (area)
	{
	  gdk_gc_set_clip_rectangle(gc1, area);
	  gdk_gc_set_clip_rectangle(gc2, area);
	}

      if (area)
	{
	    gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
	}
	/* fill with background color */
	gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
			   x, y, width, height);
	if (area)
	{
	    gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
	}

	as = (width < height) ? (width - 2) / 3 : (height - 2) / 3;
	if ((arrow_type == GTK_ARROW_UP) || (arrow_type == GTK_ARROW_DOWN))
	{
	    xl = x + width / 2 - as / 2 - 1;
	    xm = xl + as - 1;
	    xr = xm + as - 1;
	    yt = y + as + 1;
	    yb = yt + as - 1;
	}
	else
	{
	    xl = x + as + 1;
	    xr = xl + as - 1;
	    yt = y + height / 2 - as / 2 - 1;
	    ym = yt + as - 1;
	    yb = ym + as - 1;
	}

	switch (arrow_type)
	{
	    case GTK_ARROW_UP:
		gdk_draw_line(window, gc1, xl, yb, xm, yt);
		gdk_draw_line(window, gc1, xl+1, yb, xm, yt+1);

		gdk_draw_line(window, gc1, xr, yb, xm, yt);
		gdk_draw_line(window, gc1, xr-1, yb, xm, yt+1);
#ifdef BROKEN_MACH64
		gdk_draw_point(window, gc1, xm, yt);
#endif
		break;

	    case GTK_ARROW_DOWN:
		gdk_draw_line(window, gc1, xl,   yt, xm, yb);
		gdk_draw_line(window, gc1, xl+1, yt, xm, yb-1);

		gdk_draw_line(window, gc1, xr,   yt, xm, yb);
		gdk_draw_line(window, gc1, xr-1, yt, xm, yb-1);
#ifdef BROKEN_MACH64
		gdk_draw_point(window, gc1, xm, yb);
#endif
		break;

	    case GTK_ARROW_LEFT:
		gdk_draw_line(window, gc1, xr, yt,   xl,   ym);
		gdk_draw_line(window, gc1, xr, yt+1, xl+1, ym);

		gdk_draw_line(window, gc1, xr, yb,   xl,   ym);
		gdk_draw_line(window, gc1, xr, yb-1, xl+1, ym);
#ifdef BROKEN_MACH64
		gdk_draw_point(window, style->bg_gc[state_type], xl, ym-1);
		gdk_draw_point(window, style->bg_gc[state_type], xl, ym+1);
		gdk_draw_point(window, gc1, xl, ym);
#endif
		break;

	    case GTK_ARROW_RIGHT:
		gdk_draw_line(window, gc1, xl, yt,   xr,   ym);
		gdk_draw_line(window, gc1, xl, yt+1, xr-1, ym);

		gdk_draw_line(window, gc1, xl, yb,   xr,   ym);
		gdk_draw_line(window, gc1, xl, yb-1, xr-1, ym);
#ifdef BROKEN_MACH64
		gdk_draw_point(window, style->bg_gc[state_type], xr, ym-1);
		gdk_draw_point(window, style->bg_gc[state_type], xr, ym+1);
		gdk_draw_point(window, gc1, xr, ym);
#endif
		break;
	}

	if (DETAIL("spinbutton") ||
	    !th_data->flat_arrows ||
	    ((shadow_type == GTK_SHADOW_OUT) ||
	     (shadow_type == GTK_SHADOW_ETCHED_OUT)) &&
	    (detail && !strcmp(&detail[1], "scrollbar"))) {
	  gtk_paint_shadow(style, window, state_type, shadow_type, area,
			   widget, detail, x, y, width, height);
	}
	if (area)
	{
	    gdk_gc_set_clip_rectangle(gc1, NULL);
	    gdk_gc_set_clip_rectangle(gc2, NULL);
	}
    }
}

static void
draw_diamond(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail, gint x, gint y, gint width, gint height)
{
    gint half_width;
    gint half_height;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);
#ifdef DEBUG
    printf("%-17s %-15s\n", "draw_diamond", detail);
#endif

    if ((width == -1) && (height == -1))
	gdk_window_get_size(window, &width, &height);
    else if (width == -1)
	gdk_window_get_size(window, &width, NULL);
    else if (height == -1)
	gdk_window_get_size(window, NULL, &height);

    half_width = width / 2;
    half_height = height / 2;

    if (area)
    {
	gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
	gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
	gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
	gdk_gc_set_clip_rectangle(style->black_gc, area);
    }
    switch (shadow_type)
    {
	case GTK_SHADOW_IN:
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + 2, y + half_height,
			  x + half_width, y + height - 2);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + half_width, y + height - 2,
			  x + width - 2, y + half_height);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + 1, y + half_height,
			  x + half_width, y + height - 1);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + half_width, y + height - 1,
			  x + width - 1, y + half_height);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x, y + half_height, x + half_width, y + height);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + half_width, y + height,
			  x + width, y + half_height);

	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + 2, y + half_height, x + half_width, y + 2);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + half_width, y + 2,
			  x + width - 2, y + half_height);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + 1, y + half_height, x + half_width, y + 1);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + half_width, y + 1,
			  x + width - 1, y + half_height);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x, y + half_height, x + half_width, y);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + half_width, y, x + width, y + half_height);
	    break;
	case GTK_SHADOW_OUT:
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + 2, y + half_height,
			  x + half_width, y + height - 2);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + half_width, y + height - 2,
			  x + width - 2, y + half_height);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + 1, y + half_height,
			  x + half_width, y + height - 1);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + half_width, y + height - 1,
			  x + width - 1, y + half_height);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x, y + half_height, x + half_width, y + height);
	    gdk_draw_line(window, style->dark_gc[state_type],
			  x + half_width, y + height,
			  x + width, y + half_height);

	    gdk_draw_line(window, style->light_gc[state_type],
			  x + 2, y + half_height, x + half_width, y + 2);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + half_width, y + 2,
			  x + width - 2, y + half_height);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + 1, y + half_height, x + half_width, y + 1);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + half_width, y + 1,
			  x + width - 1, y + half_height);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x, y + half_height, x + half_width, y);
	    gdk_draw_line(window, style->light_gc[state_type],
			  x + half_width, y, x + width, y + half_height);
	    break;
	default:
	    break;
    }
    if (area)
    {
	gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
	gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
	gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
	gdk_gc_set_clip_rectangle(style->black_gc, NULL);
    }
}

static void
draw_oval(GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail,
	  gint x,
	  gint y,
	  gint width,
	  gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
}

static void
draw_string(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    const gchar * string)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->white_gc, area);
      gdk_gc_set_clip_rectangle(style->fg_gc[state_type], area);
    }
  if (state_type == GTK_STATE_INSENSITIVE)
    gdk_draw_string(window, style->font, style->white_gc, x + 1, y + 1, string);
  gdk_draw_string(window, style->font, style->fg_gc[state_type], x, y, string);
  if (area)
    {
      gdk_gc_set_clip_rectangle(style->white_gc, NULL);
      gdk_gc_set_clip_rectangle(style->fg_gc[state_type], NULL);
    }
}

static void
draw_box(GtkStyle * style,
	 GdkWindow * window,
	 GtkStateType state_type,
	 GtkShadowType shadow_type,
	 GdkRectangle * area,
	 GtkWidget * widget,
	 gchar * detail,
	 gint x,
	 gint y,
	 gint width,
	 gint height)
{
  GdkGC *light_gc, *dark_gc;
  GtkOrientation      orientation;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  light_gc = style->light_gc[state_type];
  dark_gc = style->dark_gc[state_type];

  orientation = GTK_ORIENTATION_HORIZONTAL;
  if (height > width)
    orientation = GTK_ORIENTATION_VERTICAL;

#ifdef DEBUG
     printf("%p %s %i %i state_type = %d (%d)\n", detail, detail, width, height, state_type, orientation);
#endif

  if (DETAIL("through")) {
      if (area) {
          gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
      }
      gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
              x, y, width, height);
      if (area) {
          gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
      }
      gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
              detail, x, y, width, height);
  } else if (DETAIL("slider")) {
        GdkPoint pointsh[7];
        int i, rect = 0, midlines = 1;
        ThemeStyleData *data = style->engine_data;
        pointsh[0].x = x;          pointsh[0].y = y+height-1;
        pointsh[1].x = x;          pointsh[1].y = y+6;
        pointsh[2].x = x+6;        pointsh[2].y = y;
        pointsh[3].x = x+width-1;  pointsh[3].y = y;
        pointsh[4].x = x+width-1;  pointsh[4].y = y+height-7;
        pointsh[5].x = x+width-7;  pointsh[5].y = y+height-1;
        pointsh[6].x = x;          pointsh[6].y = y+height-1;

#ifdef DEBUG
        printf("Slider... x,y=%d,%d width = %d, height = %d (%d)\n",x,y,width,height, state_type);
#endif
        if (data) {
            if (data->scrollbar_marks == MARKS_OFF) {
                midlines = 0;
            }
            if (data->scrollbar_type == SCROLL_RECT) {
                rect = 1;
            }
        }
        /* too small, use rect & no midlines */
        if ((width <= 13) && (height <= 13)) {
            midlines = 0;
            rect = 1;
        }

        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
            gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
            gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
        }
        if (!rect && area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_ACTIVE], area);
        }

        if (rect) {
            /* Just a plain rectangle with shadow ... */
            if ((!style->bg_pixmap[state_type]) ||
                    (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP)) {
                gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
                        x, y, width, height);
            } else {
                gtk_style_apply_default_background(style, window,
                        widget && !GTK_WIDGET_NO_WINDOW(widget),
                        state_type, area,
                        x, y, width, height);
            }
            gtk_paint_shadow(style, window, state_type, shadow_type, area,
                    widget, detail, x, y, width, height);
        } else {
            /* Fill it with the "correct" background first */
            gdk_draw_rectangle(window, style->bg_gc[GTK_STATE_ACTIVE], TRUE,
                    x, y, width, height);
            /* Fill the polygon */
            gdk_draw_polygon(window, style->bg_gc[state_type],
                    TRUE, pointsh, 6);
            /* Draw the light border */
            for (i=0;i<3;i++) {
                gdk_draw_line(window, style->light_gc[state_type],
                        pointsh[i].x,pointsh[i].y,
                        pointsh[i+1].x,pointsh[i+1].y);
            }
            /* Draw the dark border */
            for (i=3;i<6;i++) {
                gdk_draw_line(window, style->dark_gc[state_type],
                        pointsh[i].x,pointsh[i].y,
                        pointsh[i+1].x,pointsh[i+1].y);
            }
            
        }
	draw_dimple(style, window, state_type, width, height, x, y,
			   (orientation == GTK_ORIENTATION_HORIZONTAL));
        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
            gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
            gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
        }
        if (!rect && area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_ACTIVE], NULL);
        }
    } else if (DETAIL("buttondefault")) {
      /* I don't want no background on default buttons..
         Let's add that cute triangle (see below) instead... */
    } else if (DETAIL("button")) {
        GdkPoint points1[3]; /* dark */
        GdkPoint points2[3]; /* light */
        points1[0].x = x+2;  points1[0].y = y+2;
        points1[1].x = x+10; points1[1].y = y+2;
        points1[2].x = x+2;  points1[2].y = y+10;
        points2[0].x = x+3;  points2[0].y = y+3;
        points2[1].x = x+10; points2[1].y = y+3;
        points2[2].x = x+3;  points2[2].y = y+10;

        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
        }
        if ((!style->bg_pixmap[state_type]) ||
                (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP)) {
            gdk_draw_rectangle(window, style->bg_gc[state_type],
                    TRUE, x + 1, y + 1, width - 2, height - 2);
        } else {
            gtk_style_apply_default_background(style, window,
                    widget && !GTK_WIDGET_NO_WINDOW(widget),
                    state_type, area,
                    x, y, width, height);
        }
        /* Paint a triangle here instead of in "buttondefault"
           which is drawn _behind_ the current button */
        if (GTK_WIDGET_HAS_DEFAULT (widget)) {
            gdk_draw_polygon(window, dark_gc,
                    FALSE, points1, 3);
            gdk_draw_polygon(window, light_gc,
                    FALSE, points2, 3);
            gdk_draw_polygon(window, style->bg_gc[GTK_STATE_SELECTED],
                    TRUE, points2, 3);
        }
        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
        }
        gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
                detail, x, y, width, height);
    } else if (DETAIL("bar")) {
        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], area);
        }
        gdk_draw_rectangle(window, style->bg_gc[GTK_STATE_SELECTED],
                TRUE, x + 1, y + 1, width - 2, height - 2);
        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], NULL);
        }
    } else if (DETAIL("handlebox_bin")) {
        if ((!style->bg_pixmap[state_type]) ||
                (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP)) {
            if (area) {
                gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
            }
            gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
                    x, y, width, height);
            if (area) {
                gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
            }
        } else {
            gtk_style_apply_default_background(style, window,
                    widget && !GTK_WIDGET_NO_WINDOW(widget),
                    state_type, area,
                    x, y, width, height);
        }
        /*
        gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
                detail, x, y, width, height);
         */
    } else if (DETAIL("menubar")) {
        if ((!style->bg_pixmap[state_type]) ||
                (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP)) {
            if (area) {
                gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
            }
            gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
                    x, y, width, height);
            if (area) {
                gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
            }
        } else {
            gtk_style_apply_default_background(style, window,
                    widget && !GTK_WIDGET_NO_WINDOW(widget),
                    state_type, area,
                    x, y, width, height);
        }
        /*
        gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
                detail, x, y, width, height);
         */
        /*
    } else if (DETAIL("notebook")) {
        thinice_notebook(style, window, state_type, shadow_type, area, widget,
                detail, x, y, width, height);
         */
    } else if (DETAIL("tab")) {
        thinice_tab(style, window, state_type, shadow_type, area, widget,
                detail, x, y, width, height);
    } else {
        if ((!style->bg_pixmap[state_type]) ||
                (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP)) {
            if (area) {
                gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
            }
            gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
                    x, y, width, height);
            if (area) {
                gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
            }
        } else {
            gtk_style_apply_default_background(style, window,
                    widget && !GTK_WIDGET_NO_WINDOW(widget),
                    state_type, area,
                    x, y, width, height);
        }
        gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
                detail, x, y, width, height);
    }
}



static void
draw_flat_box(GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail,
	      gint x,
	      gint y,
	      gint width,
	      gint height)
{
    GdkGC              *gc1;
    
    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);
    
    if ((width == -1) && (height == -1))
	gdk_window_get_size(window, &width, &height);
    else if (width == -1)
	gdk_window_get_size(window, &width, NULL);
    else if (height == -1)
	gdk_window_get_size(window, NULL, &height);
    
    gc1 = style->bg_gc[state_type];
    
    if (DETAIL("text") && (state_type == GTK_STATE_SELECTED))
	gc1 = style->bg_gc[GTK_STATE_SELECTED];
    else if (DETAIL("viewportbin"))
	gc1 = style->bg_gc[GTK_STATE_NORMAL];
    else if (DETAIL("entry_bg"))
	gc1 = style->white_gc;
    if ((!style->bg_pixmap[state_type]) || (gc1 != style->bg_gc[state_type]) ||
	(gdk_window_get_type(window) == GDK_WINDOW_PIXMAP))
    {
	if (area)
	    gdk_gc_set_clip_rectangle(gc1, area);

	gdk_draw_rectangle(window, gc1, TRUE,
			   x, y, width, height);
	if (DETAIL("tooltip"))
	    gdk_draw_rectangle(window, style->black_gc, FALSE,
			       x, y, width - 1, height - 1);
	if (area)
	    gdk_gc_set_clip_rectangle(gc1, NULL);
    }
    else
	gtk_style_apply_default_background(style, window,
                widget && !GTK_WIDGET_NO_WINDOW(widget),
                state_type, area, x, y, width, height);
}

static void
draw_check(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
    gtk_paint_box(style, window, state_type, shadow_type, area, widget, detail,
		  x, y, width, height);
}

/* Thanks to Evan Lawrence */
static void
draw_option(GtkStyle * style,
            GdkWindow * window,
            GtkStateType state_type,
            GtkShadowType shadow_type,
            GdkRectangle * area,
            GtkWidget * widget,
            gchar * detail,
            gint x,
            gint y,
            gint width,
            gint height)
{
  GdkGC              *gc1;
  GdkGC              *gc2;
  GdkGC              *gc3;

  if (shadow_type == GTK_SHADOW_IN)
  {
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      gc3 = style->bg_gc[state_type];
  }
  else
  {
      gc1 = style->light_gc[state_type]; 
      gc2 = style->dark_gc[state_type];
      gc3 = style->bg_gc[state_type];
  }
                          
  if (area)                           
  {
      gdk_gc_set_clip_rectangle(gc1, area);
      gdk_gc_set_clip_rectangle(gc2, area);
      gdk_gc_set_clip_rectangle(gc3, area);
  }
  
  gdk_draw_arc(window, gc3, TRUE, x, y, width, height, 0, 360 * 64);
  gdk_draw_arc(window, gc1, FALSE, x, y, width, height, 45 * 64, 225 * 64);
  gdk_draw_arc(window, gc2, FALSE, x, y, width, height, 225 * 64, 180 * 64);
  
  if (area)
  {
      gdk_gc_set_clip_rectangle(gc1, NULL);
      gdk_gc_set_clip_rectangle(gc2, NULL);
      gdk_gc_set_clip_rectangle(gc3, NULL);
  }
}


static void
draw_cross(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
}

static void
draw_ramp(GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail,
	  GtkArrowType arrow_type,
	  gint x,
	  gint y,
	  gint width,
	  gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
}

static void
draw_tab(GtkStyle * style,
	 GdkWindow * window,
	 GtkStateType state_type,
	 GtkShadowType shadow_type,
	 GdkRectangle * area,
	 GtkWidget * widget,
	 gchar * detail,
	 gint x,
	 gint y,
	 gint width,
	 gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

#ifdef DEBUG
  printf("draw_tab(%d,%d,%d,%d) (detail = '%s')\n", x, y, width, height, detail?detail:"<null>");
#endif
  gtk_paint_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y, width, height);
}

static void
draw_shadow_gap(GtkStyle * style,
		GdkWindow * window,
		GtkStateType state_type,
		GtkShadowType shadow_type,
		GdkRectangle * area,
		GtkWidget * widget,
		gchar * detail,
		gint x,
		gint y,
		gint width,
		gint height,
		GtkPositionType gap_side,
		gint gap_x,
		gint gap_width)
{
  GdkRectangle        rect;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  gtk_paint_shadow(style, window, state_type, shadow_type, area, widget, detail,
		   x, y, width, height);

  switch (gap_side)
    {
    case GTK_POS_TOP:
      rect.x = x + gap_x;
      rect.y = y;
      rect.width = gap_width;
      rect.height = 2;
      break;
    case GTK_POS_BOTTOM:
      rect.x = x + gap_x;
      rect.y = y + height - 2;
      rect.width = gap_width;
      rect.height = 2;
      break;
    case GTK_POS_LEFT:
      rect.x = x;
      rect.y = y + gap_x;
      rect.width = 2;
      rect.height = gap_width;
      break;
    case GTK_POS_RIGHT:
      rect.x = x + width - 2;
      rect.y = y + gap_x;
      rect.width = 2;
      rect.height = gap_width;
      break;
    }

  gtk_style_apply_default_background(style, window,
          widget && !GTK_WIDGET_NO_WINDOW(widget),
          state_type, area,
          rect.x, rect.y, rect.width, rect.height);
}

static void
draw_box_gap(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y,
	     gint width,
	     gint height,
	     GtkPositionType gap_side,
	     gint gap_x,
	     gint gap_width)
{
  GdkRectangle        rect;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  gtk_paint_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y, width, height);

  switch (gap_side)
    {
    case GTK_POS_TOP:
      rect.x = x + gap_x;
      rect.y = y;
      rect.width = gap_width;
      rect.height = 2;
      break;
    case GTK_POS_BOTTOM:
      rect.x = x + gap_x;
      rect.y = y + height - 2;
      rect.width = gap_width;
      rect.height = 2;
      break;
    case GTK_POS_LEFT:
      rect.x = x;
      rect.y = y + gap_x;
      rect.width = 2;
      rect.height = gap_width;
      break;
    case GTK_POS_RIGHT:
      rect.x = x + width - 2;
      rect.y = y + gap_x;
      rect.width = 2;
      rect.height = gap_width;
      break;
    }

  gtk_style_apply_default_background(style, window,
          widget && !GTK_WIDGET_NO_WINDOW(widget),
          state_type, area, rect.x, rect.y, rect.width, rect.height);
}

static void
draw_extension(GtkStyle * style,
	       GdkWindow * window,
	       GtkStateType state_type,
	       GtkShadowType shadow_type,
	       GdkRectangle * area,
	       GtkWidget * widget,
	       gchar * detail,
	       gint x,
	       gint y,
	       gint width,
	       gint height,
	       GtkPositionType gap_side)
{
  GdkRectangle        rect;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  gtk_paint_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y, width, height);

  switch (gap_side)
    {
    case GTK_POS_TOP:
      rect.x = x + style->klass->xthickness;
      rect.y = y;
      rect.width = width - style->klass->xthickness * 2;
      rect.height = style->klass->ythickness;
      break;
    case GTK_POS_BOTTOM:
      rect.x = x + style->klass->xthickness;
      rect.y = y + height - style->klass->ythickness;
      rect.width = width - style->klass->xthickness * 2;
      rect.height = style->klass->ythickness;
      break;
    case GTK_POS_LEFT:
      rect.x = x;
      rect.y = y + style->klass->ythickness;
      rect.width = style->klass->xthickness;
      rect.height = height - style->klass->ythickness * 2;
      break;
    case GTK_POS_RIGHT:
      rect.x = x + width - style->klass->xthickness;
      rect.y = y + style->klass->ythickness;
      rect.width = style->klass->xthickness;
      rect.height = height - style->klass->ythickness * 2;
      break;
    }

  gtk_style_apply_default_background(style, window,
          widget && !GTK_WIDGET_NO_WINDOW(widget),
          state_type, area, rect.x, rect.y, rect.width, rect.height);
}

static void
draw_focus(GtkStyle * style,
	   GdkWindow * window,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    {
      gdk_window_get_size(window, &width, &height);
      width -= 1;
      height -= 1;
    }
  else if (width == -1)
    {
      gdk_window_get_size(window, &width, NULL);
      width -= 1;
    }
  else if (height == -1)
    {
      gdk_window_get_size(window, NULL, &height);
      height -= 1;
    }
  if (area)
    {
      gdk_gc_set_clip_rectangle(style->black_gc, area);
    }
  gdk_draw_rectangle(window,
		     style->black_gc, FALSE,
		     x, y, width, height);
  if (area)
    {
      gdk_gc_set_clip_rectangle(style->black_gc, NULL);
    }
}

static void
draw_slider(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height,
	    GtkOrientation orientation)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

#ifdef DEBUG
  printf("draw_slider(%s, %d, %d, %d)\n", detail, x, y, orientation);
#endif
  gtk_draw_box(style, window, state_type, shadow_type, x, y,
	       width, height);
  if (orientation == GTK_ORIENTATION_HORIZONTAL)
    draw_vline(style, window, state_type, area, widget, detail,
	       style->klass->ythickness,
	       height - style->klass->ythickness - 1, width / 2);
  else
    draw_hline(style, window, state_type, area, widget, detail,
	       style->klass->xthickness,
	       width - style->klass->xthickness - 1, height / 2);
}

static void
draw_handle(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height,
	    GtkOrientation orientation)
{
    GdkGC              *light_gc, *dark_gc;
    GdkRectangle        dest;
    ThemeStyleData     *data = style->engine_data;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    if ((width == -1) && (height == -1))
        gdk_window_get_size(window, &width, &height);
    else if (width == -1)
        gdk_window_get_size(window, &width, NULL);
    else if (height == -1)
        gdk_window_get_size(window, NULL, &height);

    gtk_paint_box(style, window, state_type, shadow_type, area, widget,
            detail, x, y, width, height);

    if (height > width)
	draw_vline(style, window, state_type, area, widget, detail,
	    height/2-2, height/2+2, width/2-1);
    else
	draw_hline(style, window, state_type, area, widget, detail,
	    width/2-1, width/2+1, height/2-1);

}



static void
thinice_tab(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height)
{
    GtkNotebook *notebook;
    GdkGC *lightgc, *darkgc;
    int orientation;

    
    notebook = GTK_NOTEBOOK(widget);
    orientation = notebook->tab_pos;
    
    lightgc = style->light_gc[state_type];
    darkgc = style->dark_gc[state_type];

    if ((!style->bg_pixmap[state_type]) ||
            (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP)) {
        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
        }
        gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
                x, y, width, height);
        if (area) {
            gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
        }
    } else {
        gtk_style_apply_default_background(style, window,
                widget && !GTK_WIDGET_NO_WINDOW(widget),
                state_type, area, x, y, width, height);
    }
    if (area) {
        gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
        gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
    }
    switch(orientation) {
        case GTK_POS_TOP:
            gdk_draw_line(window, lightgc,
                    x, y + height - 1, x, y);
            gdk_draw_line(window, lightgc,
                    x, y, x + width - 1, y);
            gdk_draw_line(window, darkgc,
                    x + width - 1, y, x + width - 1, y + height - 1);
            break;
        case GTK_POS_BOTTOM:
            gdk_draw_line(window, lightgc,
                    x, y, x, y + height - 1);
            gdk_draw_line(window, darkgc,
                    x, y + height - 1, x + width - 1, y + height - 1);
            gdk_draw_line(window, darkgc,
                    x + width - 1, y + height - 1, x + width - 1, y);
            break;
        case GTK_POS_LEFT:
            gdk_draw_line(window, lightgc,
                    x, y + height - 1, x, y);
            gdk_draw_line(window, lightgc,
                    x, y, x + width - 1, y);
            gdk_draw_line(window, darkgc,
                    x, y + height - 1, x + width - 1, y + height - 1);
            break;
        case GTK_POS_RIGHT:
            gdk_draw_line(window, lightgc,
                    x, y, x + width - 1, y);
            gdk_draw_line(window, darkgc,
                    x + width - 1, y, x + width - 1, y + height - 1);
            gdk_draw_line(window, darkgc,
                    x, y + height - 1, x + width - 1, y + height - 1);
            break;
    }
    if (area) {
        gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
        gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
    }
        /*gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
                detail, x, y, width, height);
                */
}
