/*
 * $Id: utf8.c,v 1.7 2004/11/13 12:27:29 cbiere Exp $
 *
 * Copyright (c) 2002-2003, Raphael Manfredi
 *
 * Unicode Transformation Format 8 bits.
 *
 * This code has been heavily inspired by utf8.c/utf8.h from Perl 5.6.1,
 * written by Larry Wall et al.
 *
 *----------------------------------------------------------------------
 * This file is part of gtk-gnutella.
 *
 *  gtk-gnutella is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  gtk-gnutella is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with gtk-gnutella; if not, write to the Free Software
 *  Foundation, Inc.:
 *      59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *----------------------------------------------------------------------
 */

#include "common.h"

RCSID("$Id: utf8.c,v 1.7 2004/11/13 12:27:29 cbiere Exp $");

#ifdef ENABLE_NLS
#include <libintl.h>
#endif /* ENABLE_NLS */

#include <locale.h>

#if defined(I_LIBCHARSET)
#include <libcharset.h>
#else
#include <langinfo.h>
#endif /* I_LIBCHARSET */

#ifndef USE_GLIB2
#include <iconv.h>
#define GIConv iconv_t
#define g_iconv_open(t, f) iconv_open(t, f) 
#define g_iconv(c, i, n, o, m) iconv(c, i, n, o, m)
#endif /* !USE_GLIB2 */

#include "utf8.h"
#include "misc.h"
#include "glib-missing.h"
#include "override.h"		/* Must be the last header included */

static guint32 common_dbg = 0;	/* XXX -- need to init lib's props --RAM */

static void unicode_decompose_init(void);
size_t utf8_decompose_nfd(const gchar *in, gchar *out, size_t size);
static inline const guint32 *utf32_decompose_nfd_char(guint32 uc, size_t *len);
size_t utf32_strmaxlen(const guint32 *s, size_t maxlen);
size_t utf32_to_utf8(const guint32 *in, gchar *out, size_t size);

#ifdef USE_ICU
static  UConverter *conv_icu_locale = NULL;
static  UConverter *conv_icu_utf8 = NULL;
#endif /* USE_ICU */
 
static const gchar *charset = NULL;

static GIConv cd_locale_to_utf8	= (GIConv) -1;
static GIConv cd_utf8_to_locale	= (GIConv) -1;
static GIConv cd_latin_to_utf8	= (GIConv) -1;

/*
 * How wide is an UTF-8 encoded char, depending on its first byte?
 */
static guint8 utf8len[256] = {
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 000-015: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 016-031: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 032-047: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 048-063: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 064-079: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 080-095: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 096-111: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 112-127: ASCII */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 128-143: invalid! */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 128-159: invalid! */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 160-175: invalid! */
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,		/* 176-191: invalid! */
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,		/* 192-207 */
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,		/* 208-223 */
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,		/* 224-239 */
	4,4,4,4,4,4,4,4,5,5,5,5,6,6,			/* 240-253 */
	7,7										/* 254-255: special */
};

#define UTF8_SKIP(s)	utf8len[*((const guchar *) s)]

static const guint8 utf8len_mark[] = {
	0x00, 0x00, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC 
};

#define UTF8_LENGTH_MARK(len)	utf8len_mark[len]
	
/*
 * The following table is from Unicode 3.1.
 *
 * Code Points           1st Byte    2nd Byte    3rd Byte    4th Byte
 *
 *    U+0000..U+007F      00..7F
 *    U+0080..U+07FF      C2..DF      80..BF
 *    U+0800..U+0FFF      E0          A0..BF      80..BF
 *    U+1000..U+FFFF      E1..EF      80..BF      80..BF
 *   U+10000..U+3FFFF     F0          90..BF      80..BF      80..BF
 *   U+40000..U+FFFFF     F1..F3      80..BF      80..BF      80..BF
 *  U+100000..U+10FFFF    F4          80..8F      80..BF      80..BF

 */

#define CHAR(x)					((guchar) (x))
#define UTF8_BYTE_MARK			0x80
#define UTF8_BYTE_MASK			0xbf
#define UTF8_IS_ASCII(x)		(CHAR(x) < UTF8_BYTE_MARK)
#define UTF8_IS_START(x)		(CHAR(x) >= 0xc0 && CHAR(x) <= 0xfd)
#define UTF8_IS_CONTINUATION(x)	\
	(CHAR(x) >= UTF8_BYTE_MARK && CHAR(x) <= UTF8_BYTE_MASK)
#define UTF8_IS_CONTINUED(x)	(CHAR(x) & UTF8_BYTE_MARK)

#define UTF8_CONT_MASK			(CHAR(0x3f))
#define UTF8_ACCU_SHIFT			6
#define UTF8_ACCUMULATE(o,n)	\
	(((o) << UTF8_ACCU_SHIFT) | (CHAR(n) & UTF8_CONT_MASK))

#define UNISKIP(v) (			\
	(v) <  0x80 		? 1 :	\
	(v) <  0x800 		? 2 :	\
	(v) <  0x10000 		? 3 :	\
	(v) <  0x200000		? 4 :	\
	(v) <  0x4000000	? 5 :	\
	(v) <  0x80000000	? 6 : 7)

#define UNI_SURROGATE_FIRST		0xd800
#define UNI_SURROGATE_LAST		0xdfff
#define UNI_HANGUL_FIRST		0xac00
#define UNI_HANGUL_LAST			0Xd7a3
#define UNI_REPLACEMENT			0xfffd
#define UNI_BYTE_ORDER_MARK		0xfffe
#define UNI_ILLEGAL				0xffff

#define UNICODE_IS_SURROGATE(x)	\
	((x) >= UNI_SURROGATE_FIRST && (x) <= UNI_SURROGATE_LAST)

#define UNICODE_IS_HANGUL(x)	\
	((x) >= UNI_HANGUL_FIRST && (x) <= UNI_HANGUL_LAST)
	
#define UNICODE_IS_ASCII(x)				((x) < 0x0080U)
#define UNICODE_IS_REPLACEMENT(x)		((x) == UNI_REPLACEMENT)
#define UNICODE_IS_BYTE_ORDER_MARK(x)	((x) == UNI_BYTE_ORDER_MARK)
#define UNICODE_IS_ILLEGAL(x)			((x) == UNI_ILLEGAL)

#define UTF32_NFD_REPLACE_MAXLEN 4
/**
 * This is the lookup table for NFD normalization/decomposition.
 *
 * The table is extracted from UnicodeData-3.2.0.txt - the
 * sixth column is relevant.
 * 
 * The mappings for U+2F868, U+2F874, U+2F91F, U+2F95F, and U+2F9BF
 * have been corrected in accordance with
 *
 * 		http://www.unicode.org/versions/corrigendum4.html
 *
 * The complete specifications regarding Unicode normalization and
 * decomposition are available here:
 *
 * 		http://www.unicode.org/reports/tr15/
 *
 * NB:	The entries are only zero-terminated if the replacement string is
 * 		shorter than UTF32_NFD_REPLACE_MAXLEN characters! The first
 *		element of each entry is the decomposable UTF-32 character, the
 *		next elements form the UTF-32 string.
 */ 
static const guint32 utf32_nfd_lut[][UTF32_NFD_REPLACE_MAXLEN + 1] = {
	{ 0x000C0, 0x00041, 0x00300 },
	{ 0x000C1, 0x00041, 0x00301 },
	{ 0x000C2, 0x00041, 0x00302 },
	{ 0x000C3, 0x00041, 0x00303 },
	{ 0x000C4, 0x00041, 0x00308 },
	{ 0x000C5, 0x00041, 0x0030A },
	{ 0x000C7, 0x00043, 0x00327 },
	{ 0x000C8, 0x00045, 0x00300 },
	{ 0x000C9, 0x00045, 0x00301 },
	{ 0x000CA, 0x00045, 0x00302 },
	{ 0x000CB, 0x00045, 0x00308 },
	{ 0x000CC, 0x00049, 0x00300 },
	{ 0x000CD, 0x00049, 0x00301 },
	{ 0x000CE, 0x00049, 0x00302 },
	{ 0x000CF, 0x00049, 0x00308 },
	{ 0x000D1, 0x0004E, 0x00303 },
	{ 0x000D2, 0x0004F, 0x00300 },
	{ 0x000D3, 0x0004F, 0x00301 },
	{ 0x000D4, 0x0004F, 0x00302 },
	{ 0x000D5, 0x0004F, 0x00303 },
	{ 0x000D6, 0x0004F, 0x00308 },
	{ 0x000D9, 0x00055, 0x00300 },
	{ 0x000DA, 0x00055, 0x00301 },
	{ 0x000DB, 0x00055, 0x00302 },
	{ 0x000DC, 0x00055, 0x00308 },
	{ 0x000DD, 0x00059, 0x00301 },
	{ 0x000E0, 0x00061, 0x00300 },
	{ 0x000E1, 0x00061, 0x00301 },
	{ 0x000E2, 0x00061, 0x00302 },
	{ 0x000E3, 0x00061, 0x00303 },
	{ 0x000E4, 0x00061, 0x00308 },
	{ 0x000E5, 0x00061, 0x0030A },
	{ 0x000E7, 0x00063, 0x00327 },
	{ 0x000E8, 0x00065, 0x00300 },
	{ 0x000E9, 0x00065, 0x00301 },
	{ 0x000EA, 0x00065, 0x00302 },
	{ 0x000EB, 0x00065, 0x00308 },
	{ 0x000EC, 0x00069, 0x00300 },
	{ 0x000ED, 0x00069, 0x00301 },
	{ 0x000EE, 0x00069, 0x00302 },
	{ 0x000EF, 0x00069, 0x00308 },
	{ 0x000F1, 0x0006E, 0x00303 },
	{ 0x000F2, 0x0006F, 0x00300 },
	{ 0x000F3, 0x0006F, 0x00301 },
	{ 0x000F4, 0x0006F, 0x00302 },
	{ 0x000F5, 0x0006F, 0x00303 },
	{ 0x000F6, 0x0006F, 0x00308 },
	{ 0x000F9, 0x00075, 0x00300 },
	{ 0x000FA, 0x00075, 0x00301 },
	{ 0x000FB, 0x00075, 0x00302 },
	{ 0x000FC, 0x00075, 0x00308 },
	{ 0x000FD, 0x00079, 0x00301 },
	{ 0x000FF, 0x00079, 0x00308 },
	{ 0x00100, 0x00041, 0x00304 },
	{ 0x00101, 0x00061, 0x00304 },
	{ 0x00102, 0x00041, 0x00306 },
	{ 0x00103, 0x00061, 0x00306 },
	{ 0x00104, 0x00041, 0x00328 },
	{ 0x00105, 0x00061, 0x00328 },
	{ 0x00106, 0x00043, 0x00301 },
	{ 0x00107, 0x00063, 0x00301 },
	{ 0x00108, 0x00043, 0x00302 },
	{ 0x00109, 0x00063, 0x00302 },
	{ 0x0010A, 0x00043, 0x00307 },
	{ 0x0010B, 0x00063, 0x00307 },
	{ 0x0010C, 0x00043, 0x0030C },
	{ 0x0010D, 0x00063, 0x0030C },
	{ 0x0010E, 0x00044, 0x0030C },
	{ 0x0010F, 0x00064, 0x0030C },
	{ 0x00112, 0x00045, 0x00304 },
	{ 0x00113, 0x00065, 0x00304 },
	{ 0x00114, 0x00045, 0x00306 },
	{ 0x00115, 0x00065, 0x00306 },
	{ 0x00116, 0x00045, 0x00307 },
	{ 0x00117, 0x00065, 0x00307 },
	{ 0x00118, 0x00045, 0x00328 },
	{ 0x00119, 0x00065, 0x00328 },
	{ 0x0011A, 0x00045, 0x0030C },
	{ 0x0011B, 0x00065, 0x0030C },
	{ 0x0011C, 0x00047, 0x00302 },
	{ 0x0011D, 0x00067, 0x00302 },
	{ 0x0011E, 0x00047, 0x00306 },
	{ 0x0011F, 0x00067, 0x00306 },
	{ 0x00120, 0x00047, 0x00307 },
	{ 0x00121, 0x00067, 0x00307 },
	{ 0x00122, 0x00047, 0x00327 },
	{ 0x00123, 0x00067, 0x00327 },
	{ 0x00124, 0x00048, 0x00302 },
	{ 0x00125, 0x00068, 0x00302 },
	{ 0x00128, 0x00049, 0x00303 },
	{ 0x00129, 0x00069, 0x00303 },
	{ 0x0012A, 0x00049, 0x00304 },
	{ 0x0012B, 0x00069, 0x00304 },
	{ 0x0012C, 0x00049, 0x00306 },
	{ 0x0012D, 0x00069, 0x00306 },
	{ 0x0012E, 0x00049, 0x00328 },
	{ 0x0012F, 0x00069, 0x00328 },
	{ 0x00130, 0x00049, 0x00307 },
	{ 0x00134, 0x0004A, 0x00302 },
	{ 0x00135, 0x0006A, 0x00302 },
	{ 0x00136, 0x0004B, 0x00327 },
	{ 0x00137, 0x0006B, 0x00327 },
	{ 0x00139, 0x0004C, 0x00301 },
	{ 0x0013A, 0x0006C, 0x00301 },
	{ 0x0013B, 0x0004C, 0x00327 },
	{ 0x0013C, 0x0006C, 0x00327 },
	{ 0x0013D, 0x0004C, 0x0030C },
	{ 0x0013E, 0x0006C, 0x0030C },
	{ 0x00143, 0x0004E, 0x00301 },
	{ 0x00144, 0x0006E, 0x00301 },
	{ 0x00145, 0x0004E, 0x00327 },
	{ 0x00146, 0x0006E, 0x00327 },
	{ 0x00147, 0x0004E, 0x0030C },
	{ 0x00148, 0x0006E, 0x0030C },
	{ 0x0014C, 0x0004F, 0x00304 },
	{ 0x0014D, 0x0006F, 0x00304 },
	{ 0x0014E, 0x0004F, 0x00306 },
	{ 0x0014F, 0x0006F, 0x00306 },
	{ 0x00150, 0x0004F, 0x0030B },
	{ 0x00151, 0x0006F, 0x0030B },
	{ 0x00154, 0x00052, 0x00301 },
	{ 0x00155, 0x00072, 0x00301 },
	{ 0x00156, 0x00052, 0x00327 },
	{ 0x00157, 0x00072, 0x00327 },
	{ 0x00158, 0x00052, 0x0030C },
	{ 0x00159, 0x00072, 0x0030C },
	{ 0x0015A, 0x00053, 0x00301 },
	{ 0x0015B, 0x00073, 0x00301 },
	{ 0x0015C, 0x00053, 0x00302 },
	{ 0x0015D, 0x00073, 0x00302 },
	{ 0x0015E, 0x00053, 0x00327 },
	{ 0x0015F, 0x00073, 0x00327 },
	{ 0x00160, 0x00053, 0x0030C },
	{ 0x00161, 0x00073, 0x0030C },
	{ 0x00162, 0x00054, 0x00327 },
	{ 0x00163, 0x00074, 0x00327 },
	{ 0x00164, 0x00054, 0x0030C },
	{ 0x00165, 0x00074, 0x0030C },
	{ 0x00168, 0x00055, 0x00303 },
	{ 0x00169, 0x00075, 0x00303 },
	{ 0x0016A, 0x00055, 0x00304 },
	{ 0x0016B, 0x00075, 0x00304 },
	{ 0x0016C, 0x00055, 0x00306 },
	{ 0x0016D, 0x00075, 0x00306 },
	{ 0x0016E, 0x00055, 0x0030A },
	{ 0x0016F, 0x00075, 0x0030A },
	{ 0x00170, 0x00055, 0x0030B },
	{ 0x00171, 0x00075, 0x0030B },
	{ 0x00172, 0x00055, 0x00328 },
	{ 0x00173, 0x00075, 0x00328 },
	{ 0x00174, 0x00057, 0x00302 },
	{ 0x00175, 0x00077, 0x00302 },
	{ 0x00176, 0x00059, 0x00302 },
	{ 0x00177, 0x00079, 0x00302 },
	{ 0x00178, 0x00059, 0x00308 },
	{ 0x00179, 0x0005A, 0x00301 },
	{ 0x0017A, 0x0007A, 0x00301 },
	{ 0x0017B, 0x0005A, 0x00307 },
	{ 0x0017C, 0x0007A, 0x00307 },
	{ 0x0017D, 0x0005A, 0x0030C },
	{ 0x0017E, 0x0007A, 0x0030C },
	{ 0x001A0, 0x0004F, 0x0031B },
	{ 0x001A1, 0x0006F, 0x0031B },
	{ 0x001AF, 0x00055, 0x0031B },
	{ 0x001B0, 0x00075, 0x0031B },
	{ 0x001CD, 0x00041, 0x0030C },
	{ 0x001CE, 0x00061, 0x0030C },
	{ 0x001CF, 0x00049, 0x0030C },
	{ 0x001D0, 0x00069, 0x0030C },
	{ 0x001D1, 0x0004F, 0x0030C },
	{ 0x001D2, 0x0006F, 0x0030C },
	{ 0x001D3, 0x00055, 0x0030C },
	{ 0x001D4, 0x00075, 0x0030C },
	{ 0x001D5, 0x00055, 0x00308, 0x00304 },
	{ 0x001D6, 0x00075, 0x00308, 0x00304 },
	{ 0x001D7, 0x00055, 0x00308, 0x00301 },
	{ 0x001D8, 0x00075, 0x00308, 0x00301 },
	{ 0x001D9, 0x00055, 0x00308, 0x0030C },
	{ 0x001DA, 0x00075, 0x00308, 0x0030C },
	{ 0x001DB, 0x00055, 0x00308, 0x00300 },
	{ 0x001DC, 0x00075, 0x00308, 0x00300 },
	{ 0x001DE, 0x00041, 0x00308, 0x00304 },
	{ 0x001DF, 0x00061, 0x00308, 0x00304 },
	{ 0x001E0, 0x00041, 0x00307, 0x00304 },
	{ 0x001E1, 0x00061, 0x00307, 0x00304 },
	{ 0x001E2, 0x000C6, 0x00304 },
	{ 0x001E3, 0x000E6, 0x00304 },
	{ 0x001E6, 0x00047, 0x0030C },
	{ 0x001E7, 0x00067, 0x0030C },
	{ 0x001E8, 0x0004B, 0x0030C },
	{ 0x001E9, 0x0006B, 0x0030C },
	{ 0x001EA, 0x0004F, 0x00328 },
	{ 0x001EB, 0x0006F, 0x00328 },
	{ 0x001EC, 0x0004F, 0x00328, 0x00304 },
	{ 0x001ED, 0x0006F, 0x00328, 0x00304 },
	{ 0x001EE, 0x001B7, 0x0030C },
	{ 0x001EF, 0x00292, 0x0030C },
	{ 0x001F0, 0x0006A, 0x0030C },
	{ 0x001F4, 0x00047, 0x00301 },
	{ 0x001F5, 0x00067, 0x00301 },
	{ 0x001F8, 0x0004E, 0x00300 },
	{ 0x001F9, 0x0006E, 0x00300 },
	{ 0x001FA, 0x00041, 0x0030A, 0x00301 },
	{ 0x001FB, 0x00061, 0x0030A, 0x00301 },
	{ 0x001FC, 0x000C6, 0x00301 },
	{ 0x001FD, 0x000E6, 0x00301 },
	{ 0x001FE, 0x000D8, 0x00301 },
	{ 0x001FF, 0x000F8, 0x00301 },
	{ 0x00200, 0x00041, 0x0030F },
	{ 0x00201, 0x00061, 0x0030F },
	{ 0x00202, 0x00041, 0x00311 },
	{ 0x00203, 0x00061, 0x00311 },
	{ 0x00204, 0x00045, 0x0030F },
	{ 0x00205, 0x00065, 0x0030F },
	{ 0x00206, 0x00045, 0x00311 },
	{ 0x00207, 0x00065, 0x00311 },
	{ 0x00208, 0x00049, 0x0030F },
	{ 0x00209, 0x00069, 0x0030F },
	{ 0x0020A, 0x00049, 0x00311 },
	{ 0x0020B, 0x00069, 0x00311 },
	{ 0x0020C, 0x0004F, 0x0030F },
	{ 0x0020D, 0x0006F, 0x0030F },
	{ 0x0020E, 0x0004F, 0x00311 },
	{ 0x0020F, 0x0006F, 0x00311 },
	{ 0x00210, 0x00052, 0x0030F },
	{ 0x00211, 0x00072, 0x0030F },
	{ 0x00212, 0x00052, 0x00311 },
	{ 0x00213, 0x00072, 0x00311 },
	{ 0x00214, 0x00055, 0x0030F },
	{ 0x00215, 0x00075, 0x0030F },
	{ 0x00216, 0x00055, 0x00311 },
	{ 0x00217, 0x00075, 0x00311 },
	{ 0x00218, 0x00053, 0x00326 },
	{ 0x00219, 0x00073, 0x00326 },
	{ 0x0021A, 0x00054, 0x00326 },
	{ 0x0021B, 0x00074, 0x00326 },
	{ 0x0021E, 0x00048, 0x0030C },
	{ 0x0021F, 0x00068, 0x0030C },
	{ 0x00226, 0x00041, 0x00307 },
	{ 0x00227, 0x00061, 0x00307 },
	{ 0x00228, 0x00045, 0x00327 },
	{ 0x00229, 0x00065, 0x00327 },
	{ 0x0022A, 0x0004F, 0x00308, 0x00304 },
	{ 0x0022B, 0x0006F, 0x00308, 0x00304 },
	{ 0x0022C, 0x0004F, 0x00303, 0x00304 },
	{ 0x0022D, 0x0006F, 0x00303, 0x00304 },
	{ 0x0022E, 0x0004F, 0x00307 },
	{ 0x0022F, 0x0006F, 0x00307 },
	{ 0x00230, 0x0004F, 0x00307, 0x00304 },
	{ 0x00231, 0x0006F, 0x00307, 0x00304 },
	{ 0x00232, 0x00059, 0x00304 },
	{ 0x00233, 0x00079, 0x00304 },
	{ 0x00340, 0x00300 },
	{ 0x00341, 0x00301 },
	{ 0x00343, 0x00313 },
	{ 0x00344, 0x00308, 0x00301 },
	{ 0x00374, 0x002B9 },
	{ 0x0037E, 0x0003B },
	{ 0x00385, 0x000A8, 0x00301 },
	{ 0x00386, 0x00391, 0x00301 },
	{ 0x00387, 0x000B7 },
	{ 0x00388, 0x00395, 0x00301 },
	{ 0x00389, 0x00397, 0x00301 },
	{ 0x0038A, 0x00399, 0x00301 },
	{ 0x0038C, 0x0039F, 0x00301 },
	{ 0x0038E, 0x003A5, 0x00301 },
	{ 0x0038F, 0x003A9, 0x00301 },
	{ 0x00390, 0x003B9, 0x00308, 0x00301 },
	{ 0x003AA, 0x00399, 0x00308 },
	{ 0x003AB, 0x003A5, 0x00308 },
	{ 0x003AC, 0x003B1, 0x00301 },
	{ 0x003AD, 0x003B5, 0x00301 },
	{ 0x003AE, 0x003B7, 0x00301 },
	{ 0x003AF, 0x003B9, 0x00301 },
	{ 0x003B0, 0x003C5, 0x00308, 0x00301 },
	{ 0x003CA, 0x003B9, 0x00308 },
	{ 0x003CB, 0x003C5, 0x00308 },
	{ 0x003CC, 0x003BF, 0x00301 },
	{ 0x003CD, 0x003C5, 0x00301 },
	{ 0x003CE, 0x003C9, 0x00301 },
	{ 0x003D3, 0x003D2, 0x00301 },
	{ 0x003D4, 0x003D2, 0x00308 },
	{ 0x00400, 0x00415, 0x00300 },
	{ 0x00401, 0x00415, 0x00308 },
	{ 0x00403, 0x00413, 0x00301 },
	{ 0x00407, 0x00406, 0x00308 },
	{ 0x0040C, 0x0041A, 0x00301 },
	{ 0x0040D, 0x00418, 0x00300 },
	{ 0x0040E, 0x00423, 0x00306 },
	{ 0x00419, 0x00418, 0x00306 },
	{ 0x00439, 0x00438, 0x00306 },
	{ 0x00450, 0x00435, 0x00300 },
	{ 0x00451, 0x00435, 0x00308 },
	{ 0x00453, 0x00433, 0x00301 },
	{ 0x00457, 0x00456, 0x00308 },
	{ 0x0045C, 0x0043A, 0x00301 },
	{ 0x0045D, 0x00438, 0x00300 },
	{ 0x0045E, 0x00443, 0x00306 },
	{ 0x00476, 0x00474, 0x0030F },
	{ 0x00477, 0x00475, 0x0030F },
	{ 0x004C1, 0x00416, 0x00306 },
	{ 0x004C2, 0x00436, 0x00306 },
	{ 0x004D0, 0x00410, 0x00306 },
	{ 0x004D1, 0x00430, 0x00306 },
	{ 0x004D2, 0x00410, 0x00308 },
	{ 0x004D3, 0x00430, 0x00308 },
	{ 0x004D6, 0x00415, 0x00306 },
	{ 0x004D7, 0x00435, 0x00306 },
	{ 0x004DA, 0x004D8, 0x00308 },
	{ 0x004DB, 0x004D9, 0x00308 },
	{ 0x004DC, 0x00416, 0x00308 },
	{ 0x004DD, 0x00436, 0x00308 },
	{ 0x004DE, 0x00417, 0x00308 },
	{ 0x004DF, 0x00437, 0x00308 },
	{ 0x004E2, 0x00418, 0x00304 },
	{ 0x004E3, 0x00438, 0x00304 },
	{ 0x004E4, 0x00418, 0x00308 },
	{ 0x004E5, 0x00438, 0x00308 },
	{ 0x004E6, 0x0041E, 0x00308 },
	{ 0x004E7, 0x0043E, 0x00308 },
	{ 0x004EA, 0x004E8, 0x00308 },
	{ 0x004EB, 0x004E9, 0x00308 },
	{ 0x004EC, 0x0042D, 0x00308 },
	{ 0x004ED, 0x0044D, 0x00308 },
	{ 0x004EE, 0x00423, 0x00304 },
	{ 0x004EF, 0x00443, 0x00304 },
	{ 0x004F0, 0x00423, 0x00308 },
	{ 0x004F1, 0x00443, 0x00308 },
	{ 0x004F2, 0x00423, 0x0030B },
	{ 0x004F3, 0x00443, 0x0030B },
	{ 0x004F4, 0x00427, 0x00308 },
	{ 0x004F5, 0x00447, 0x00308 },
	{ 0x004F8, 0x0042B, 0x00308 },
	{ 0x004F9, 0x0044B, 0x00308 },
	{ 0x00622, 0x00627, 0x00653 },
	{ 0x00623, 0x00627, 0x00654 },
	{ 0x00624, 0x00648, 0x00654 },
	{ 0x00625, 0x00627, 0x00655 },
	{ 0x00626, 0x0064A, 0x00654 },
	{ 0x006C0, 0x006D5, 0x00654 },
	{ 0x006C2, 0x006C1, 0x00654 },
	{ 0x006D3, 0x006D2, 0x00654 },
	{ 0x00929, 0x00928, 0x0093C },
	{ 0x00931, 0x00930, 0x0093C },
	{ 0x00934, 0x00933, 0x0093C },
	{ 0x00958, 0x00915, 0x0093C },
	{ 0x00959, 0x00916, 0x0093C },
	{ 0x0095A, 0x00917, 0x0093C },
	{ 0x0095B, 0x0091C, 0x0093C },
	{ 0x0095C, 0x00921, 0x0093C },
	{ 0x0095D, 0x00922, 0x0093C },
	{ 0x0095E, 0x0092B, 0x0093C },
	{ 0x0095F, 0x0092F, 0x0093C },
	{ 0x009CB, 0x009C7, 0x009BE },
	{ 0x009CC, 0x009C7, 0x009D7 },
	{ 0x009DC, 0x009A1, 0x009BC },
	{ 0x009DD, 0x009A2, 0x009BC },
	{ 0x009DF, 0x009AF, 0x009BC },
	{ 0x00A33, 0x00A32, 0x00A3C },
	{ 0x00A36, 0x00A38, 0x00A3C },
	{ 0x00A59, 0x00A16, 0x00A3C },
	{ 0x00A5A, 0x00A17, 0x00A3C },
	{ 0x00A5B, 0x00A1C, 0x00A3C },
	{ 0x00A5E, 0x00A2B, 0x00A3C },
	{ 0x00B48, 0x00B47, 0x00B56 },
	{ 0x00B4B, 0x00B47, 0x00B3E },
	{ 0x00B4C, 0x00B47, 0x00B57 },
	{ 0x00B5C, 0x00B21, 0x00B3C },
	{ 0x00B5D, 0x00B22, 0x00B3C },
	{ 0x00B94, 0x00B92, 0x00BD7 },
	{ 0x00BCA, 0x00BC6, 0x00BBE },
	{ 0x00BCB, 0x00BC7, 0x00BBE },
	{ 0x00BCC, 0x00BC6, 0x00BD7 },
	{ 0x00C48, 0x00C46, 0x00C56 },
	{ 0x00CC0, 0x00CBF, 0x00CD5 },
	{ 0x00CC7, 0x00CC6, 0x00CD5 },
	{ 0x00CC8, 0x00CC6, 0x00CD6 },
	{ 0x00CCA, 0x00CC6, 0x00CC2 },
	{ 0x00CCB, 0x00CC6, 0x00CC2, 0x00CD5 },
	{ 0x00D4A, 0x00D46, 0x00D3E },
	{ 0x00D4B, 0x00D47, 0x00D3E },
	{ 0x00D4C, 0x00D46, 0x00D57 },
	{ 0x00DDA, 0x00DD9, 0x00DCA },
	{ 0x00DDC, 0x00DD9, 0x00DCF },
	{ 0x00DDD, 0x00DD9, 0x00DCF, 0x00DCA },
	{ 0x00DDE, 0x00DD9, 0x00DDF },
	{ 0x00F43, 0x00F42, 0x00FB7 },
	{ 0x00F4D, 0x00F4C, 0x00FB7 },
	{ 0x00F52, 0x00F51, 0x00FB7 },
	{ 0x00F57, 0x00F56, 0x00FB7 },
	{ 0x00F5C, 0x00F5B, 0x00FB7 },
	{ 0x00F69, 0x00F40, 0x00FB5 },
	{ 0x00F73, 0x00F71, 0x00F72 },
	{ 0x00F75, 0x00F71, 0x00F74 },
	{ 0x00F76, 0x00FB2, 0x00F80 },
	{ 0x00F78, 0x00FB3, 0x00F80 },
	{ 0x00F81, 0x00F71, 0x00F80 },
	{ 0x00F93, 0x00F92, 0x00FB7 },
	{ 0x00F9D, 0x00F9C, 0x00FB7 },
	{ 0x00FA2, 0x00FA1, 0x00FB7 },
	{ 0x00FA7, 0x00FA6, 0x00FB7 },
	{ 0x00FAC, 0x00FAB, 0x00FB7 },
	{ 0x00FB9, 0x00F90, 0x00FB5 },
	{ 0x01026, 0x01025, 0x0102E },
	{ 0x01E00, 0x00041, 0x00325 },
	{ 0x01E01, 0x00061, 0x00325 },
	{ 0x01E02, 0x00042, 0x00307 },
	{ 0x01E03, 0x00062, 0x00307 },
	{ 0x01E04, 0x00042, 0x00323 },
	{ 0x01E05, 0x00062, 0x00323 },
	{ 0x01E06, 0x00042, 0x00331 },
	{ 0x01E07, 0x00062, 0x00331 },
	{ 0x01E08, 0x00043, 0x00327, 0x00301 },
	{ 0x01E09, 0x00063, 0x00327, 0x00301 },
	{ 0x01E0A, 0x00044, 0x00307 },
	{ 0x01E0B, 0x00064, 0x00307 },
	{ 0x01E0C, 0x00044, 0x00323 },
	{ 0x01E0D, 0x00064, 0x00323 },
	{ 0x01E0E, 0x00044, 0x00331 },
	{ 0x01E0F, 0x00064, 0x00331 },
	{ 0x01E10, 0x00044, 0x00327 },
	{ 0x01E11, 0x00064, 0x00327 },
	{ 0x01E12, 0x00044, 0x0032D },
	{ 0x01E13, 0x00064, 0x0032D },
	{ 0x01E14, 0x00045, 0x00304, 0x00300 },
	{ 0x01E15, 0x00065, 0x00304, 0x00300 },
	{ 0x01E16, 0x00045, 0x00304, 0x00301 },
	{ 0x01E17, 0x00065, 0x00304, 0x00301 },
	{ 0x01E18, 0x00045, 0x0032D },
	{ 0x01E19, 0x00065, 0x0032D },
	{ 0x01E1A, 0x00045, 0x00330 },
	{ 0x01E1B, 0x00065, 0x00330 },
	{ 0x01E1C, 0x00045, 0x00327, 0x00306 },
	{ 0x01E1D, 0x00065, 0x00327, 0x00306 },
	{ 0x01E1E, 0x00046, 0x00307 },
	{ 0x01E1F, 0x00066, 0x00307 },
	{ 0x01E20, 0x00047, 0x00304 },
	{ 0x01E21, 0x00067, 0x00304 },
	{ 0x01E22, 0x00048, 0x00307 },
	{ 0x01E23, 0x00068, 0x00307 },
	{ 0x01E24, 0x00048, 0x00323 },
	{ 0x01E25, 0x00068, 0x00323 },
	{ 0x01E26, 0x00048, 0x00308 },
	{ 0x01E27, 0x00068, 0x00308 },
	{ 0x01E28, 0x00048, 0x00327 },
	{ 0x01E29, 0x00068, 0x00327 },
	{ 0x01E2A, 0x00048, 0x0032E },
	{ 0x01E2B, 0x00068, 0x0032E },
	{ 0x01E2C, 0x00049, 0x00330 },
	{ 0x01E2D, 0x00069, 0x00330 },
	{ 0x01E2E, 0x00049, 0x00308, 0x00301 },
	{ 0x01E2F, 0x00069, 0x00308, 0x00301 },
	{ 0x01E30, 0x0004B, 0x00301 },
	{ 0x01E31, 0x0006B, 0x00301 },
	{ 0x01E32, 0x0004B, 0x00323 },
	{ 0x01E33, 0x0006B, 0x00323 },
	{ 0x01E34, 0x0004B, 0x00331 },
	{ 0x01E35, 0x0006B, 0x00331 },
	{ 0x01E36, 0x0004C, 0x00323 },
	{ 0x01E37, 0x0006C, 0x00323 },
	{ 0x01E38, 0x0004C, 0x00323, 0x00304 },
	{ 0x01E39, 0x0006C, 0x00323, 0x00304 },
	{ 0x01E3A, 0x0004C, 0x00331 },
	{ 0x01E3B, 0x0006C, 0x00331 },
	{ 0x01E3C, 0x0004C, 0x0032D },
	{ 0x01E3D, 0x0006C, 0x0032D },
	{ 0x01E3E, 0x0004D, 0x00301 },
	{ 0x01E3F, 0x0006D, 0x00301 },
	{ 0x01E40, 0x0004D, 0x00307 },
	{ 0x01E41, 0x0006D, 0x00307 },
	{ 0x01E42, 0x0004D, 0x00323 },
	{ 0x01E43, 0x0006D, 0x00323 },
	{ 0x01E44, 0x0004E, 0x00307 },
	{ 0x01E45, 0x0006E, 0x00307 },
	{ 0x01E46, 0x0004E, 0x00323 },
	{ 0x01E47, 0x0006E, 0x00323 },
	{ 0x01E48, 0x0004E, 0x00331 },
	{ 0x01E49, 0x0006E, 0x00331 },
	{ 0x01E4A, 0x0004E, 0x0032D },
	{ 0x01E4B, 0x0006E, 0x0032D },
	{ 0x01E4C, 0x0004F, 0x00303, 0x00301 },
	{ 0x01E4D, 0x0006F, 0x00303, 0x00301 },
	{ 0x01E4E, 0x0004F, 0x00303, 0x00308 },
	{ 0x01E4F, 0x0006F, 0x00303, 0x00308 },
	{ 0x01E50, 0x0004F, 0x00304, 0x00300 },
	{ 0x01E51, 0x0006F, 0x00304, 0x00300 },
	{ 0x01E52, 0x0004F, 0x00304, 0x00301 },
	{ 0x01E53, 0x0006F, 0x00304, 0x00301 },
	{ 0x01E54, 0x00050, 0x00301 },
	{ 0x01E55, 0x00070, 0x00301 },
	{ 0x01E56, 0x00050, 0x00307 },
	{ 0x01E57, 0x00070, 0x00307 },
	{ 0x01E58, 0x00052, 0x00307 },
	{ 0x01E59, 0x00072, 0x00307 },
	{ 0x01E5A, 0x00052, 0x00323 },
	{ 0x01E5B, 0x00072, 0x00323 },
	{ 0x01E5C, 0x00052, 0x00323, 0x00304 },
	{ 0x01E5D, 0x00072, 0x00323, 0x00304 },
	{ 0x01E5E, 0x00052, 0x00331 },
	{ 0x01E5F, 0x00072, 0x00331 },
	{ 0x01E60, 0x00053, 0x00307 },
	{ 0x01E61, 0x00073, 0x00307 },
	{ 0x01E62, 0x00053, 0x00323 },
	{ 0x01E63, 0x00073, 0x00323 },
	{ 0x01E64, 0x00053, 0x00301, 0x00307 },
	{ 0x01E65, 0x00073, 0x00301, 0x00307 },
	{ 0x01E66, 0x00053, 0x0030C, 0x00307 },
	{ 0x01E67, 0x00073, 0x0030C, 0x00307 },
	{ 0x01E68, 0x00053, 0x00323, 0x00307 },
	{ 0x01E69, 0x00073, 0x00323, 0x00307 },
	{ 0x01E6A, 0x00054, 0x00307 },
	{ 0x01E6B, 0x00074, 0x00307 },
	{ 0x01E6C, 0x00054, 0x00323 },
	{ 0x01E6D, 0x00074, 0x00323 },
	{ 0x01E6E, 0x00054, 0x00331 },
	{ 0x01E6F, 0x00074, 0x00331 },
	{ 0x01E70, 0x00054, 0x0032D },
	{ 0x01E71, 0x00074, 0x0032D },
	{ 0x01E72, 0x00055, 0x00324 },
	{ 0x01E73, 0x00075, 0x00324 },
	{ 0x01E74, 0x00055, 0x00330 },
	{ 0x01E75, 0x00075, 0x00330 },
	{ 0x01E76, 0x00055, 0x0032D },
	{ 0x01E77, 0x00075, 0x0032D },
	{ 0x01E78, 0x00055, 0x00303, 0x00301 },
	{ 0x01E79, 0x00075, 0x00303, 0x00301 },
	{ 0x01E7A, 0x00055, 0x00304, 0x00308 },
	{ 0x01E7B, 0x00075, 0x00304, 0x00308 },
	{ 0x01E7C, 0x00056, 0x00303 },
	{ 0x01E7D, 0x00076, 0x00303 },
	{ 0x01E7E, 0x00056, 0x00323 },
	{ 0x01E7F, 0x00076, 0x00323 },
	{ 0x01E80, 0x00057, 0x00300 },
	{ 0x01E81, 0x00077, 0x00300 },
	{ 0x01E82, 0x00057, 0x00301 },
	{ 0x01E83, 0x00077, 0x00301 },
	{ 0x01E84, 0x00057, 0x00308 },
	{ 0x01E85, 0x00077, 0x00308 },
	{ 0x01E86, 0x00057, 0x00307 },
	{ 0x01E87, 0x00077, 0x00307 },
	{ 0x01E88, 0x00057, 0x00323 },
	{ 0x01E89, 0x00077, 0x00323 },
	{ 0x01E8A, 0x00058, 0x00307 },
	{ 0x01E8B, 0x00078, 0x00307 },
	{ 0x01E8C, 0x00058, 0x00308 },
	{ 0x01E8D, 0x00078, 0x00308 },
	{ 0x01E8E, 0x00059, 0x00307 },
	{ 0x01E8F, 0x00079, 0x00307 },
	{ 0x01E90, 0x0005A, 0x00302 },
	{ 0x01E91, 0x0007A, 0x00302 },
	{ 0x01E92, 0x0005A, 0x00323 },
	{ 0x01E93, 0x0007A, 0x00323 },
	{ 0x01E94, 0x0005A, 0x00331 },
	{ 0x01E95, 0x0007A, 0x00331 },
	{ 0x01E96, 0x00068, 0x00331 },
	{ 0x01E97, 0x00074, 0x00308 },
	{ 0x01E98, 0x00077, 0x0030A },
	{ 0x01E99, 0x00079, 0x0030A },
	{ 0x01E9B, 0x0017F, 0x00307 },
	{ 0x01EA0, 0x00041, 0x00323 },
	{ 0x01EA1, 0x00061, 0x00323 },
	{ 0x01EA2, 0x00041, 0x00309 },
	{ 0x01EA3, 0x00061, 0x00309 },
	{ 0x01EA4, 0x00041, 0x00302, 0x00301 },
	{ 0x01EA5, 0x00061, 0x00302, 0x00301 },
	{ 0x01EA6, 0x00041, 0x00302, 0x00300 },
	{ 0x01EA7, 0x00061, 0x00302, 0x00300 },
	{ 0x01EA8, 0x00041, 0x00302, 0x00309 },
	{ 0x01EA9, 0x00061, 0x00302, 0x00309 },
	{ 0x01EAA, 0x00041, 0x00302, 0x00303 },
	{ 0x01EAB, 0x00061, 0x00302, 0x00303 },
	{ 0x01EAC, 0x00041, 0x00323, 0x00302 },
	{ 0x01EAD, 0x00061, 0x00323, 0x00302 },
	{ 0x01EAE, 0x00041, 0x00306, 0x00301 },
	{ 0x01EAF, 0x00061, 0x00306, 0x00301 },
	{ 0x01EB0, 0x00041, 0x00306, 0x00300 },
	{ 0x01EB1, 0x00061, 0x00306, 0x00300 },
	{ 0x01EB2, 0x00041, 0x00306, 0x00309 },
	{ 0x01EB3, 0x00061, 0x00306, 0x00309 },
	{ 0x01EB4, 0x00041, 0x00306, 0x00303 },
	{ 0x01EB5, 0x00061, 0x00306, 0x00303 },
	{ 0x01EB6, 0x00041, 0x00323, 0x00306 },
	{ 0x01EB7, 0x00061, 0x00323, 0x00306 },
	{ 0x01EB8, 0x00045, 0x00323 },
	{ 0x01EB9, 0x00065, 0x00323 },
	{ 0x01EBA, 0x00045, 0x00309 },
	{ 0x01EBB, 0x00065, 0x00309 },
	{ 0x01EBC, 0x00045, 0x00303 },
	{ 0x01EBD, 0x00065, 0x00303 },
	{ 0x01EBE, 0x00045, 0x00302, 0x00301 },
	{ 0x01EBF, 0x00065, 0x00302, 0x00301 },
	{ 0x01EC0, 0x00045, 0x00302, 0x00300 },
	{ 0x01EC1, 0x00065, 0x00302, 0x00300 },
	{ 0x01EC2, 0x00045, 0x00302, 0x00309 },
	{ 0x01EC3, 0x00065, 0x00302, 0x00309 },
	{ 0x01EC4, 0x00045, 0x00302, 0x00303 },
	{ 0x01EC5, 0x00065, 0x00302, 0x00303 },
	{ 0x01EC6, 0x00045, 0x00323, 0x00302 },
	{ 0x01EC7, 0x00065, 0x00323, 0x00302 },
	{ 0x01EC8, 0x00049, 0x00309 },
	{ 0x01EC9, 0x00069, 0x00309 },
	{ 0x01ECA, 0x00049, 0x00323 },
	{ 0x01ECB, 0x00069, 0x00323 },
	{ 0x01ECC, 0x0004F, 0x00323 },
	{ 0x01ECD, 0x0006F, 0x00323 },
	{ 0x01ECE, 0x0004F, 0x00309 },
	{ 0x01ECF, 0x0006F, 0x00309 },
	{ 0x01ED0, 0x0004F, 0x00302, 0x00301 },
	{ 0x01ED1, 0x0006F, 0x00302, 0x00301 },
	{ 0x01ED2, 0x0004F, 0x00302, 0x00300 },
	{ 0x01ED3, 0x0006F, 0x00302, 0x00300 },
	{ 0x01ED4, 0x0004F, 0x00302, 0x00309 },
	{ 0x01ED5, 0x0006F, 0x00302, 0x00309 },
	{ 0x01ED6, 0x0004F, 0x00302, 0x00303 },
	{ 0x01ED7, 0x0006F, 0x00302, 0x00303 },
	{ 0x01ED8, 0x0004F, 0x00323, 0x00302 },
	{ 0x01ED9, 0x0006F, 0x00323, 0x00302 },
	{ 0x01EDA, 0x0004F, 0x0031B, 0x00301 },
	{ 0x01EDB, 0x0006F, 0x0031B, 0x00301 },
	{ 0x01EDC, 0x0004F, 0x0031B, 0x00300 },
	{ 0x01EDD, 0x0006F, 0x0031B, 0x00300 },
	{ 0x01EDE, 0x0004F, 0x0031B, 0x00309 },
	{ 0x01EDF, 0x0006F, 0x0031B, 0x00309 },
	{ 0x01EE0, 0x0004F, 0x0031B, 0x00303 },
	{ 0x01EE1, 0x0006F, 0x0031B, 0x00303 },
	{ 0x01EE2, 0x0004F, 0x0031B, 0x00323 },
	{ 0x01EE3, 0x0006F, 0x0031B, 0x00323 },
	{ 0x01EE4, 0x00055, 0x00323 },
	{ 0x01EE5, 0x00075, 0x00323 },
	{ 0x01EE6, 0x00055, 0x00309 },
	{ 0x01EE7, 0x00075, 0x00309 },
	{ 0x01EE8, 0x00055, 0x0031B, 0x00301 },
	{ 0x01EE9, 0x00075, 0x0031B, 0x00301 },
	{ 0x01EEA, 0x00055, 0x0031B, 0x00300 },
	{ 0x01EEB, 0x00075, 0x0031B, 0x00300 },
	{ 0x01EEC, 0x00055, 0x0031B, 0x00309 },
	{ 0x01EED, 0x00075, 0x0031B, 0x00309 },
	{ 0x01EEE, 0x00055, 0x0031B, 0x00303 },
	{ 0x01EEF, 0x00075, 0x0031B, 0x00303 },
	{ 0x01EF0, 0x00055, 0x0031B, 0x00323 },
	{ 0x01EF1, 0x00075, 0x0031B, 0x00323 },
	{ 0x01EF2, 0x00059, 0x00300 },
	{ 0x01EF3, 0x00079, 0x00300 },
	{ 0x01EF4, 0x00059, 0x00323 },
	{ 0x01EF5, 0x00079, 0x00323 },
	{ 0x01EF6, 0x00059, 0x00309 },
	{ 0x01EF7, 0x00079, 0x00309 },
	{ 0x01EF8, 0x00059, 0x00303 },
	{ 0x01EF9, 0x00079, 0x00303 },
	{ 0x01F00, 0x003B1, 0x00313 },
	{ 0x01F01, 0x003B1, 0x00314 },
	{ 0x01F02, 0x003B1, 0x00313, 0x00300 },
	{ 0x01F03, 0x003B1, 0x00314, 0x00300 },
	{ 0x01F04, 0x003B1, 0x00313, 0x00301 },
	{ 0x01F05, 0x003B1, 0x00314, 0x00301 },
	{ 0x01F06, 0x003B1, 0x00313, 0x00342 },
	{ 0x01F07, 0x003B1, 0x00314, 0x00342 },
	{ 0x01F08, 0x00391, 0x00313 },
	{ 0x01F09, 0x00391, 0x00314 },
	{ 0x01F0A, 0x00391, 0x00313, 0x00300 },
	{ 0x01F0B, 0x00391, 0x00314, 0x00300 },
	{ 0x01F0C, 0x00391, 0x00313, 0x00301 },
	{ 0x01F0D, 0x00391, 0x00314, 0x00301 },
	{ 0x01F0E, 0x00391, 0x00313, 0x00342 },
	{ 0x01F0F, 0x00391, 0x00314, 0x00342 },
	{ 0x01F10, 0x003B5, 0x00313 },
	{ 0x01F11, 0x003B5, 0x00314 },
	{ 0x01F12, 0x003B5, 0x00313, 0x00300 },
	{ 0x01F13, 0x003B5, 0x00314, 0x00300 },
	{ 0x01F14, 0x003B5, 0x00313, 0x00301 },
	{ 0x01F15, 0x003B5, 0x00314, 0x00301 },
	{ 0x01F18, 0x00395, 0x00313 },
	{ 0x01F19, 0x00395, 0x00314 },
	{ 0x01F1A, 0x00395, 0x00313, 0x00300 },
	{ 0x01F1B, 0x00395, 0x00314, 0x00300 },
	{ 0x01F1C, 0x00395, 0x00313, 0x00301 },
	{ 0x01F1D, 0x00395, 0x00314, 0x00301 },
	{ 0x01F20, 0x003B7, 0x00313 },
	{ 0x01F21, 0x003B7, 0x00314 },
	{ 0x01F22, 0x003B7, 0x00313, 0x00300 },
	{ 0x01F23, 0x003B7, 0x00314, 0x00300 },
	{ 0x01F24, 0x003B7, 0x00313, 0x00301 },
	{ 0x01F25, 0x003B7, 0x00314, 0x00301 },
	{ 0x01F26, 0x003B7, 0x00313, 0x00342 },
	{ 0x01F27, 0x003B7, 0x00314, 0x00342 },
	{ 0x01F28, 0x00397, 0x00313 },
	{ 0x01F29, 0x00397, 0x00314 },
	{ 0x01F2A, 0x00397, 0x00313, 0x00300 },
	{ 0x01F2B, 0x00397, 0x00314, 0x00300 },
	{ 0x01F2C, 0x00397, 0x00313, 0x00301 },
	{ 0x01F2D, 0x00397, 0x00314, 0x00301 },
	{ 0x01F2E, 0x00397, 0x00313, 0x00342 },
	{ 0x01F2F, 0x00397, 0x00314, 0x00342 },
	{ 0x01F30, 0x003B9, 0x00313 },
	{ 0x01F31, 0x003B9, 0x00314 },
	{ 0x01F32, 0x003B9, 0x00313, 0x00300 },
	{ 0x01F33, 0x003B9, 0x00314, 0x00300 },
	{ 0x01F34, 0x003B9, 0x00313, 0x00301 },
	{ 0x01F35, 0x003B9, 0x00314, 0x00301 },
	{ 0x01F36, 0x003B9, 0x00313, 0x00342 },
	{ 0x01F37, 0x003B9, 0x00314, 0x00342 },
	{ 0x01F38, 0x00399, 0x00313 },
	{ 0x01F39, 0x00399, 0x00314 },
	{ 0x01F3A, 0x00399, 0x00313, 0x00300 },
	{ 0x01F3B, 0x00399, 0x00314, 0x00300 },
	{ 0x01F3C, 0x00399, 0x00313, 0x00301 },
	{ 0x01F3D, 0x00399, 0x00314, 0x00301 },
	{ 0x01F3E, 0x00399, 0x00313, 0x00342 },
	{ 0x01F3F, 0x00399, 0x00314, 0x00342 },
	{ 0x01F40, 0x003BF, 0x00313 },
	{ 0x01F41, 0x003BF, 0x00314 },
	{ 0x01F42, 0x003BF, 0x00313, 0x00300 },
	{ 0x01F43, 0x003BF, 0x00314, 0x00300 },
	{ 0x01F44, 0x003BF, 0x00313, 0x00301 },
	{ 0x01F45, 0x003BF, 0x00314, 0x00301 },
	{ 0x01F48, 0x0039F, 0x00313 },
	{ 0x01F49, 0x0039F, 0x00314 },
	{ 0x01F4A, 0x0039F, 0x00313, 0x00300 },
	{ 0x01F4B, 0x0039F, 0x00314, 0x00300 },
	{ 0x01F4C, 0x0039F, 0x00313, 0x00301 },
	{ 0x01F4D, 0x0039F, 0x00314, 0x00301 },
	{ 0x01F50, 0x003C5, 0x00313 },
	{ 0x01F51, 0x003C5, 0x00314 },
	{ 0x01F52, 0x003C5, 0x00313, 0x00300 },
	{ 0x01F53, 0x003C5, 0x00314, 0x00300 },
	{ 0x01F54, 0x003C5, 0x00313, 0x00301 },
	{ 0x01F55, 0x003C5, 0x00314, 0x00301 },
	{ 0x01F56, 0x003C5, 0x00313, 0x00342 },
	{ 0x01F57, 0x003C5, 0x00314, 0x00342 },
	{ 0x01F59, 0x003A5, 0x00314 },
	{ 0x01F5B, 0x003A5, 0x00314, 0x00300 },
	{ 0x01F5D, 0x003A5, 0x00314, 0x00301 },
	{ 0x01F5F, 0x003A5, 0x00314, 0x00342 },
	{ 0x01F60, 0x003C9, 0x00313 },
	{ 0x01F61, 0x003C9, 0x00314 },
	{ 0x01F62, 0x003C9, 0x00313, 0x00300 },
	{ 0x01F63, 0x003C9, 0x00314, 0x00300 },
	{ 0x01F64, 0x003C9, 0x00313, 0x00301 },
	{ 0x01F65, 0x003C9, 0x00314, 0x00301 },
	{ 0x01F66, 0x003C9, 0x00313, 0x00342 },
	{ 0x01F67, 0x003C9, 0x00314, 0x00342 },
	{ 0x01F68, 0x003A9, 0x00313 },
	{ 0x01F69, 0x003A9, 0x00314 },
	{ 0x01F6A, 0x003A9, 0x00313, 0x00300 },
	{ 0x01F6B, 0x003A9, 0x00314, 0x00300 },
	{ 0x01F6C, 0x003A9, 0x00313, 0x00301 },
	{ 0x01F6D, 0x003A9, 0x00314, 0x00301 },
	{ 0x01F6E, 0x003A9, 0x00313, 0x00342 },
	{ 0x01F6F, 0x003A9, 0x00314, 0x00342 },
	{ 0x01F70, 0x003B1, 0x00300 },
	{ 0x01F71, 0x003B1, 0x00301 },
	{ 0x01F72, 0x003B5, 0x00300 },
	{ 0x01F73, 0x003B5, 0x00301 },
	{ 0x01F74, 0x003B7, 0x00300 },
	{ 0x01F75, 0x003B7, 0x00301 },
	{ 0x01F76, 0x003B9, 0x00300 },
	{ 0x01F77, 0x003B9, 0x00301 },
	{ 0x01F78, 0x003BF, 0x00300 },
	{ 0x01F79, 0x003BF, 0x00301 },
	{ 0x01F7A, 0x003C5, 0x00300 },
	{ 0x01F7B, 0x003C5, 0x00301 },
	{ 0x01F7C, 0x003C9, 0x00300 },
	{ 0x01F7D, 0x003C9, 0x00301 },
	{ 0x01F80, 0x003B1, 0x00313, 0x00345 },
	{ 0x01F81, 0x003B1, 0x00314, 0x00345 },
	{ 0x01F82, 0x003B1, 0x00313, 0x00300, 0x00345 },
	{ 0x01F83, 0x003B1, 0x00314, 0x00300, 0x00345 },
	{ 0x01F84, 0x003B1, 0x00313, 0x00301, 0x00345 },
	{ 0x01F85, 0x003B1, 0x00314, 0x00301, 0x00345 },
	{ 0x01F86, 0x003B1, 0x00313, 0x00342, 0x00345 },
	{ 0x01F87, 0x003B1, 0x00314, 0x00342, 0x00345 },
	{ 0x01F88, 0x00391, 0x00313, 0x00345 },
	{ 0x01F89, 0x00391, 0x00314, 0x00345 },
	{ 0x01F8A, 0x00391, 0x00313, 0x00300, 0x00345 },
	{ 0x01F8B, 0x00391, 0x00314, 0x00300, 0x00345 },
	{ 0x01F8C, 0x00391, 0x00313, 0x00301, 0x00345 },
	{ 0x01F8D, 0x00391, 0x00314, 0x00301, 0x00345 },
	{ 0x01F8E, 0x00391, 0x00313, 0x00342, 0x00345 },
	{ 0x01F8F, 0x00391, 0x00314, 0x00342, 0x00345 },
	{ 0x01F90, 0x003B7, 0x00313, 0x00345 },
	{ 0x01F91, 0x003B7, 0x00314, 0x00345 },
	{ 0x01F92, 0x003B7, 0x00313, 0x00300, 0x00345 },
	{ 0x01F93, 0x003B7, 0x00314, 0x00300, 0x00345 },
	{ 0x01F94, 0x003B7, 0x00313, 0x00301, 0x00345 },
	{ 0x01F95, 0x003B7, 0x00314, 0x00301, 0x00345 },
	{ 0x01F96, 0x003B7, 0x00313, 0x00342, 0x00345 },
	{ 0x01F97, 0x003B7, 0x00314, 0x00342, 0x00345 },
	{ 0x01F98, 0x00397, 0x00313, 0x00345 },
	{ 0x01F99, 0x00397, 0x00314, 0x00345 },
	{ 0x01F9A, 0x00397, 0x00313, 0x00300, 0x00345 },
	{ 0x01F9B, 0x00397, 0x00314, 0x00300, 0x00345 },
	{ 0x01F9C, 0x00397, 0x00313, 0x00301, 0x00345 },
	{ 0x01F9D, 0x00397, 0x00314, 0x00301, 0x00345 },
	{ 0x01F9E, 0x00397, 0x00313, 0x00342, 0x00345 },
	{ 0x01F9F, 0x00397, 0x00314, 0x00342, 0x00345 },
	{ 0x01FA0, 0x003C9, 0x00313, 0x00345 },
	{ 0x01FA1, 0x003C9, 0x00314, 0x00345 },
	{ 0x01FA2, 0x003C9, 0x00313, 0x00300, 0x00345 },
	{ 0x01FA3, 0x003C9, 0x00314, 0x00300, 0x00345 },
	{ 0x01FA4, 0x003C9, 0x00313, 0x00301, 0x00345 },
	{ 0x01FA5, 0x003C9, 0x00314, 0x00301, 0x00345 },
	{ 0x01FA6, 0x003C9, 0x00313, 0x00342, 0x00345 },
	{ 0x01FA7, 0x003C9, 0x00314, 0x00342, 0x00345 },
	{ 0x01FA8, 0x003A9, 0x00313, 0x00345 },
	{ 0x01FA9, 0x003A9, 0x00314, 0x00345 },
	{ 0x01FAA, 0x003A9, 0x00313, 0x00300, 0x00345 },
	{ 0x01FAB, 0x003A9, 0x00314, 0x00300, 0x00345 },
	{ 0x01FAC, 0x003A9, 0x00313, 0x00301, 0x00345 },
	{ 0x01FAD, 0x003A9, 0x00314, 0x00301, 0x00345 },
	{ 0x01FAE, 0x003A9, 0x00313, 0x00342, 0x00345 },
	{ 0x01FAF, 0x003A9, 0x00314, 0x00342, 0x00345 },
	{ 0x01FB0, 0x003B1, 0x00306 },
	{ 0x01FB1, 0x003B1, 0x00304 },
	{ 0x01FB2, 0x003B1, 0x00300, 0x00345 },
	{ 0x01FB3, 0x003B1, 0x00345 },
	{ 0x01FB4, 0x003B1, 0x00301, 0x00345 },
	{ 0x01FB6, 0x003B1, 0x00342 },
	{ 0x01FB7, 0x003B1, 0x00342, 0x00345 },
	{ 0x01FB8, 0x00391, 0x00306 },
	{ 0x01FB9, 0x00391, 0x00304 },
	{ 0x01FBA, 0x00391, 0x00300 },
	{ 0x01FBB, 0x00391, 0x00301 },
	{ 0x01FBC, 0x00391, 0x00345 },
	{ 0x01FBE, 0x003B9 },
	{ 0x01FC1, 0x000A8, 0x00342 },
	{ 0x01FC2, 0x003B7, 0x00300, 0x00345 },
	{ 0x01FC3, 0x003B7, 0x00345 },
	{ 0x01FC4, 0x003B7, 0x00301, 0x00345 },
	{ 0x01FC6, 0x003B7, 0x00342 },
	{ 0x01FC7, 0x003B7, 0x00342, 0x00345 },
	{ 0x01FC8, 0x00395, 0x00300 },
	{ 0x01FC9, 0x00395, 0x00301 },
	{ 0x01FCA, 0x00397, 0x00300 },
	{ 0x01FCB, 0x00397, 0x00301 },
	{ 0x01FCC, 0x00397, 0x00345 },
	{ 0x01FCD, 0x01FBF, 0x00300 },
	{ 0x01FCE, 0x01FBF, 0x00301 },
	{ 0x01FCF, 0x01FBF, 0x00342 },
	{ 0x01FD0, 0x003B9, 0x00306 },
	{ 0x01FD1, 0x003B9, 0x00304 },
	{ 0x01FD2, 0x003B9, 0x00308, 0x00300 },
	{ 0x01FD3, 0x003B9, 0x00308, 0x00301 },
	{ 0x01FD6, 0x003B9, 0x00342 },
	{ 0x01FD7, 0x003B9, 0x00308, 0x00342 },
	{ 0x01FD8, 0x00399, 0x00306 },
	{ 0x01FD9, 0x00399, 0x00304 },
	{ 0x01FDA, 0x00399, 0x00300 },
	{ 0x01FDB, 0x00399, 0x00301 },
	{ 0x01FDD, 0x01FFE, 0x00300 },
	{ 0x01FDE, 0x01FFE, 0x00301 },
	{ 0x01FDF, 0x01FFE, 0x00342 },
	{ 0x01FE0, 0x003C5, 0x00306 },
	{ 0x01FE1, 0x003C5, 0x00304 },
	{ 0x01FE2, 0x003C5, 0x00308, 0x00300 },
	{ 0x01FE3, 0x003C5, 0x00308, 0x00301 },
	{ 0x01FE4, 0x003C1, 0x00313 },
	{ 0x01FE5, 0x003C1, 0x00314 },
	{ 0x01FE6, 0x003C5, 0x00342 },
	{ 0x01FE7, 0x003C5, 0x00308, 0x00342 },
	{ 0x01FE8, 0x003A5, 0x00306 },
	{ 0x01FE9, 0x003A5, 0x00304 },
	{ 0x01FEA, 0x003A5, 0x00300 },
	{ 0x01FEB, 0x003A5, 0x00301 },
	{ 0x01FEC, 0x003A1, 0x00314 },
	{ 0x01FED, 0x000A8, 0x00300 },
	{ 0x01FEE, 0x000A8, 0x00301 },
	{ 0x01FEF, 0x00060 },
	{ 0x01FF2, 0x003C9, 0x00300, 0x00345 },
	{ 0x01FF3, 0x003C9, 0x00345 },
	{ 0x01FF4, 0x003C9, 0x00301, 0x00345 },
	{ 0x01FF6, 0x003C9, 0x00342 },
	{ 0x01FF7, 0x003C9, 0x00342, 0x00345 },
	{ 0x01FF8, 0x0039F, 0x00300 },
	{ 0x01FF9, 0x0039F, 0x00301 },
	{ 0x01FFA, 0x003A9, 0x00300 },
	{ 0x01FFB, 0x003A9, 0x00301 },
	{ 0x01FFC, 0x003A9, 0x00345 },
	{ 0x01FFD, 0x000B4 },
	{ 0x02000, 0x02002 },
	{ 0x02001, 0x02003 },
	{ 0x02126, 0x003A9 },
	{ 0x0212A, 0x0004B },
	{ 0x0212B, 0x00041, 0x0030A },
	{ 0x0219A, 0x02190, 0x00338 },
	{ 0x0219B, 0x02192, 0x00338 },
	{ 0x021AE, 0x02194, 0x00338 },
	{ 0x021CD, 0x021D0, 0x00338 },
	{ 0x021CE, 0x021D4, 0x00338 },
	{ 0x021CF, 0x021D2, 0x00338 },
	{ 0x02204, 0x02203, 0x00338 },
	{ 0x02209, 0x02208, 0x00338 },
	{ 0x0220C, 0x0220B, 0x00338 },
	{ 0x02224, 0x02223, 0x00338 },
	{ 0x02226, 0x02225, 0x00338 },
	{ 0x02241, 0x0223C, 0x00338 },
	{ 0x02244, 0x02243, 0x00338 },
	{ 0x02247, 0x02245, 0x00338 },
	{ 0x02249, 0x02248, 0x00338 },
	{ 0x02260, 0x0003D, 0x00338 },
	{ 0x02262, 0x02261, 0x00338 },
	{ 0x0226D, 0x0224D, 0x00338 },
	{ 0x0226E, 0x0003C, 0x00338 },
	{ 0x0226F, 0x0003E, 0x00338 },
	{ 0x02270, 0x02264, 0x00338 },
	{ 0x02271, 0x02265, 0x00338 },
	{ 0x02274, 0x02272, 0x00338 },
	{ 0x02275, 0x02273, 0x00338 },
	{ 0x02278, 0x02276, 0x00338 },
	{ 0x02279, 0x02277, 0x00338 },
	{ 0x02280, 0x0227A, 0x00338 },
	{ 0x02281, 0x0227B, 0x00338 },
	{ 0x02284, 0x02282, 0x00338 },
	{ 0x02285, 0x02283, 0x00338 },
	{ 0x02288, 0x02286, 0x00338 },
	{ 0x02289, 0x02287, 0x00338 },
	{ 0x022AC, 0x022A2, 0x00338 },
	{ 0x022AD, 0x022A8, 0x00338 },
	{ 0x022AE, 0x022A9, 0x00338 },
	{ 0x022AF, 0x022AB, 0x00338 },
	{ 0x022E0, 0x0227C, 0x00338 },
	{ 0x022E1, 0x0227D, 0x00338 },
	{ 0x022E2, 0x02291, 0x00338 },
	{ 0x022E3, 0x02292, 0x00338 },
	{ 0x022EA, 0x022B2, 0x00338 },
	{ 0x022EB, 0x022B3, 0x00338 },
	{ 0x022EC, 0x022B4, 0x00338 },
	{ 0x022ED, 0x022B5, 0x00338 },
	{ 0x02329, 0x03008 },
	{ 0x0232A, 0x03009 },
	{ 0x02ADC, 0x02ADD, 0x00338 },
	{ 0x0304C, 0x0304B, 0x03099 },
	{ 0x0304E, 0x0304D, 0x03099 },
	{ 0x03050, 0x0304F, 0x03099 },
	{ 0x03052, 0x03051, 0x03099 },
	{ 0x03054, 0x03053, 0x03099 },
	{ 0x03056, 0x03055, 0x03099 },
	{ 0x03058, 0x03057, 0x03099 },
	{ 0x0305A, 0x03059, 0x03099 },
	{ 0x0305C, 0x0305B, 0x03099 },
	{ 0x0305E, 0x0305D, 0x03099 },
	{ 0x03060, 0x0305F, 0x03099 },
	{ 0x03062, 0x03061, 0x03099 },
	{ 0x03065, 0x03064, 0x03099 },
	{ 0x03067, 0x03066, 0x03099 },
	{ 0x03069, 0x03068, 0x03099 },
	{ 0x03070, 0x0306F, 0x03099 },
	{ 0x03071, 0x0306F, 0x0309A },
	{ 0x03073, 0x03072, 0x03099 },
	{ 0x03074, 0x03072, 0x0309A },
	{ 0x03076, 0x03075, 0x03099 },
	{ 0x03077, 0x03075, 0x0309A },
	{ 0x03079, 0x03078, 0x03099 },
	{ 0x0307A, 0x03078, 0x0309A },
	{ 0x0307C, 0x0307B, 0x03099 },
	{ 0x0307D, 0x0307B, 0x0309A },
	{ 0x03094, 0x03046, 0x03099 },
	{ 0x0309E, 0x0309D, 0x03099 },
	{ 0x030AC, 0x030AB, 0x03099 },
	{ 0x030AE, 0x030AD, 0x03099 },
	{ 0x030B0, 0x030AF, 0x03099 },
	{ 0x030B2, 0x030B1, 0x03099 },
	{ 0x030B4, 0x030B3, 0x03099 },
	{ 0x030B6, 0x030B5, 0x03099 },
	{ 0x030B8, 0x030B7, 0x03099 },
	{ 0x030BA, 0x030B9, 0x03099 },
	{ 0x030BC, 0x030BB, 0x03099 },
	{ 0x030BE, 0x030BD, 0x03099 },
	{ 0x030C0, 0x030BF, 0x03099 },
	{ 0x030C2, 0x030C1, 0x03099 },
	{ 0x030C5, 0x030C4, 0x03099 },
	{ 0x030C7, 0x030C6, 0x03099 },
	{ 0x030C9, 0x030C8, 0x03099 },
	{ 0x030D0, 0x030CF, 0x03099 },
	{ 0x030D1, 0x030CF, 0x0309A },
	{ 0x030D3, 0x030D2, 0x03099 },
	{ 0x030D4, 0x030D2, 0x0309A },
	{ 0x030D6, 0x030D5, 0x03099 },
	{ 0x030D7, 0x030D5, 0x0309A },
	{ 0x030D9, 0x030D8, 0x03099 },
	{ 0x030DA, 0x030D8, 0x0309A },
	{ 0x030DC, 0x030DB, 0x03099 },
	{ 0x030DD, 0x030DB, 0x0309A },
	{ 0x030F4, 0x030A6, 0x03099 },
	{ 0x030F7, 0x030EF, 0x03099 },
	{ 0x030F8, 0x030F0, 0x03099 },
	{ 0x030F9, 0x030F1, 0x03099 },
	{ 0x030FA, 0x030F2, 0x03099 },
	{ 0x030FE, 0x030FD, 0x03099 },
	{ 0x0F900, 0x08C48 },
	{ 0x0F901, 0x066F4 },
	{ 0x0F902, 0x08ECA },
	{ 0x0F903, 0x08CC8 },
	{ 0x0F904, 0x06ED1 },
	{ 0x0F905, 0x04E32 },
	{ 0x0F906, 0x053E5 },
	{ 0x0F907, 0x09F9C },
	{ 0x0F908, 0x09F9C },
	{ 0x0F909, 0x05951 },
	{ 0x0F90A, 0x091D1 },
	{ 0x0F90B, 0x05587 },
	{ 0x0F90C, 0x05948 },
	{ 0x0F90D, 0x061F6 },
	{ 0x0F90E, 0x07669 },
	{ 0x0F90F, 0x07F85 },
	{ 0x0F910, 0x0863F },
	{ 0x0F911, 0x087BA },
	{ 0x0F912, 0x088F8 },
	{ 0x0F913, 0x0908F },
	{ 0x0F914, 0x06A02 },
	{ 0x0F915, 0x06D1B },
	{ 0x0F916, 0x070D9 },
	{ 0x0F917, 0x073DE },
	{ 0x0F918, 0x0843D },
	{ 0x0F919, 0x0916A },
	{ 0x0F91A, 0x099F1 },
	{ 0x0F91B, 0x04E82 },
	{ 0x0F91C, 0x05375 },
	{ 0x0F91D, 0x06B04 },
	{ 0x0F91E, 0x0721B },
	{ 0x0F91F, 0x0862D },
	{ 0x0F920, 0x09E1E },
	{ 0x0F921, 0x05D50 },
	{ 0x0F922, 0x06FEB },
	{ 0x0F923, 0x085CD },
	{ 0x0F924, 0x08964 },
	{ 0x0F925, 0x062C9 },
	{ 0x0F926, 0x081D8 },
	{ 0x0F927, 0x0881F },
	{ 0x0F928, 0x05ECA },
	{ 0x0F929, 0x06717 },
	{ 0x0F92A, 0x06D6A },
	{ 0x0F92B, 0x072FC },
	{ 0x0F92C, 0x090CE },
	{ 0x0F92D, 0x04F86 },
	{ 0x0F92E, 0x051B7 },
	{ 0x0F92F, 0x052DE },
	{ 0x0F930, 0x064C4 },
	{ 0x0F931, 0x06AD3 },
	{ 0x0F932, 0x07210 },
	{ 0x0F933, 0x076E7 },
	{ 0x0F934, 0x08001 },
	{ 0x0F935, 0x08606 },
	{ 0x0F936, 0x0865C },
	{ 0x0F937, 0x08DEF },
	{ 0x0F938, 0x09732 },
	{ 0x0F939, 0x09B6F },
	{ 0x0F93A, 0x09DFA },
	{ 0x0F93B, 0x0788C },
	{ 0x0F93C, 0x0797F },
	{ 0x0F93D, 0x07DA0 },
	{ 0x0F93E, 0x083C9 },
	{ 0x0F93F, 0x09304 },
	{ 0x0F940, 0x09E7F },
	{ 0x0F941, 0x08AD6 },
	{ 0x0F942, 0x058DF },
	{ 0x0F943, 0x05F04 },
	{ 0x0F944, 0x07C60 },
	{ 0x0F945, 0x0807E },
	{ 0x0F946, 0x07262 },
	{ 0x0F947, 0x078CA },
	{ 0x0F948, 0x08CC2 },
	{ 0x0F949, 0x096F7 },
	{ 0x0F94A, 0x058D8 },
	{ 0x0F94B, 0x05C62 },
	{ 0x0F94C, 0x06A13 },
	{ 0x0F94D, 0x06DDA },
	{ 0x0F94E, 0x06F0F },
	{ 0x0F94F, 0x07D2F },
	{ 0x0F950, 0x07E37 },
	{ 0x0F951, 0x0964B },
	{ 0x0F952, 0x052D2 },
	{ 0x0F953, 0x0808B },
	{ 0x0F954, 0x051DC },
	{ 0x0F955, 0x051CC },
	{ 0x0F956, 0x07A1C },
	{ 0x0F957, 0x07DBE },
	{ 0x0F958, 0x083F1 },
	{ 0x0F959, 0x09675 },
	{ 0x0F95A, 0x08B80 },
	{ 0x0F95B, 0x062CF },
	{ 0x0F95C, 0x06A02 },
	{ 0x0F95D, 0x08AFE },
	{ 0x0F95E, 0x04E39 },
	{ 0x0F95F, 0x05BE7 },
	{ 0x0F960, 0x06012 },
	{ 0x0F961, 0x07387 },
	{ 0x0F962, 0x07570 },
	{ 0x0F963, 0x05317 },
	{ 0x0F964, 0x078FB },
	{ 0x0F965, 0x04FBF },
	{ 0x0F966, 0x05FA9 },
	{ 0x0F967, 0x04E0D },
	{ 0x0F968, 0x06CCC },
	{ 0x0F969, 0x06578 },
	{ 0x0F96A, 0x07D22 },
	{ 0x0F96B, 0x053C3 },
	{ 0x0F96C, 0x0585E },
	{ 0x0F96D, 0x07701 },
	{ 0x0F96E, 0x08449 },
	{ 0x0F96F, 0x08AAA },
	{ 0x0F970, 0x06BBA },
	{ 0x0F971, 0x08FB0 },
	{ 0x0F972, 0x06C88 },
	{ 0x0F973, 0x062FE },
	{ 0x0F974, 0x082E5 },
	{ 0x0F975, 0x063A0 },
	{ 0x0F976, 0x07565 },
	{ 0x0F977, 0x04EAE },
	{ 0x0F978, 0x05169 },
	{ 0x0F979, 0x051C9 },
	{ 0x0F97A, 0x06881 },
	{ 0x0F97B, 0x07CE7 },
	{ 0x0F97C, 0x0826F },
	{ 0x0F97D, 0x08AD2 },
	{ 0x0F97E, 0x091CF },
	{ 0x0F97F, 0x052F5 },
	{ 0x0F980, 0x05442 },
	{ 0x0F981, 0x05973 },
	{ 0x0F982, 0x05EEC },
	{ 0x0F983, 0x065C5 },
	{ 0x0F984, 0x06FFE },
	{ 0x0F985, 0x0792A },
	{ 0x0F986, 0x095AD },
	{ 0x0F987, 0x09A6A },
	{ 0x0F988, 0x09E97 },
	{ 0x0F989, 0x09ECE },
	{ 0x0F98A, 0x0529B },
	{ 0x0F98B, 0x066C6 },
	{ 0x0F98C, 0x06B77 },
	{ 0x0F98D, 0x08F62 },
	{ 0x0F98E, 0x05E74 },
	{ 0x0F98F, 0x06190 },
	{ 0x0F990, 0x06200 },
	{ 0x0F991, 0x0649A },
	{ 0x0F992, 0x06F23 },
	{ 0x0F993, 0x07149 },
	{ 0x0F994, 0x07489 },
	{ 0x0F995, 0x079CA },
	{ 0x0F996, 0x07DF4 },
	{ 0x0F997, 0x0806F },
	{ 0x0F998, 0x08F26 },
	{ 0x0F999, 0x084EE },
	{ 0x0F99A, 0x09023 },
	{ 0x0F99B, 0x0934A },
	{ 0x0F99C, 0x05217 },
	{ 0x0F99D, 0x052A3 },
	{ 0x0F99E, 0x054BD },
	{ 0x0F99F, 0x070C8 },
	{ 0x0F9A0, 0x088C2 },
	{ 0x0F9A1, 0x08AAA },
	{ 0x0F9A2, 0x05EC9 },
	{ 0x0F9A3, 0x05FF5 },
	{ 0x0F9A4, 0x0637B },
	{ 0x0F9A5, 0x06BAE },
	{ 0x0F9A6, 0x07C3E },
	{ 0x0F9A7, 0x07375 },
	{ 0x0F9A8, 0x04EE4 },
	{ 0x0F9A9, 0x056F9 },
	{ 0x0F9AA, 0x05BE7 },
	{ 0x0F9AB, 0x05DBA },
	{ 0x0F9AC, 0x0601C },
	{ 0x0F9AD, 0x073B2 },
	{ 0x0F9AE, 0x07469 },
	{ 0x0F9AF, 0x07F9A },
	{ 0x0F9B0, 0x08046 },
	{ 0x0F9B1, 0x09234 },
	{ 0x0F9B2, 0x096F6 },
	{ 0x0F9B3, 0x09748 },
	{ 0x0F9B4, 0x09818 },
	{ 0x0F9B5, 0x04F8B },
	{ 0x0F9B6, 0x079AE },
	{ 0x0F9B7, 0x091B4 },
	{ 0x0F9B8, 0x096B8 },
	{ 0x0F9B9, 0x060E1 },
	{ 0x0F9BA, 0x04E86 },
	{ 0x0F9BB, 0x050DA },
	{ 0x0F9BC, 0x05BEE },
	{ 0x0F9BD, 0x05C3F },
	{ 0x0F9BE, 0x06599 },
	{ 0x0F9BF, 0x06A02 },
	{ 0x0F9C0, 0x071CE },
	{ 0x0F9C1, 0x07642 },
	{ 0x0F9C2, 0x084FC },
	{ 0x0F9C3, 0x0907C },
	{ 0x0F9C4, 0x09F8D },
	{ 0x0F9C5, 0x06688 },
	{ 0x0F9C6, 0x0962E },
	{ 0x0F9C7, 0x05289 },
	{ 0x0F9C8, 0x0677B },
	{ 0x0F9C9, 0x067F3 },
	{ 0x0F9CA, 0x06D41 },
	{ 0x0F9CB, 0x06E9C },
	{ 0x0F9CC, 0x07409 },
	{ 0x0F9CD, 0x07559 },
	{ 0x0F9CE, 0x0786B },
	{ 0x0F9CF, 0x07D10 },
	{ 0x0F9D0, 0x0985E },
	{ 0x0F9D1, 0x0516D },
	{ 0x0F9D2, 0x0622E },
	{ 0x0F9D3, 0x09678 },
	{ 0x0F9D4, 0x0502B },
	{ 0x0F9D5, 0x05D19 },
	{ 0x0F9D6, 0x06DEA },
	{ 0x0F9D7, 0x08F2A },
	{ 0x0F9D8, 0x05F8B },
	{ 0x0F9D9, 0x06144 },
	{ 0x0F9DA, 0x06817 },
	{ 0x0F9DB, 0x07387 },
	{ 0x0F9DC, 0x09686 },
	{ 0x0F9DD, 0x05229 },
	{ 0x0F9DE, 0x0540F },
	{ 0x0F9DF, 0x05C65 },
	{ 0x0F9E0, 0x06613 },
	{ 0x0F9E1, 0x0674E },
	{ 0x0F9E2, 0x068A8 },
	{ 0x0F9E3, 0x06CE5 },
	{ 0x0F9E4, 0x07406 },
	{ 0x0F9E5, 0x075E2 },
	{ 0x0F9E6, 0x07F79 },
	{ 0x0F9E7, 0x088CF },
	{ 0x0F9E8, 0x088E1 },
	{ 0x0F9E9, 0x091CC },
	{ 0x0F9EA, 0x096E2 },
	{ 0x0F9EB, 0x0533F },
	{ 0x0F9EC, 0x06EBA },
	{ 0x0F9ED, 0x0541D },
	{ 0x0F9EE, 0x071D0 },
	{ 0x0F9EF, 0x07498 },
	{ 0x0F9F0, 0x085FA },
	{ 0x0F9F1, 0x096A3 },
	{ 0x0F9F2, 0x09C57 },
	{ 0x0F9F3, 0x09E9F },
	{ 0x0F9F4, 0x06797 },
	{ 0x0F9F5, 0x06DCB },
	{ 0x0F9F6, 0x081E8 },
	{ 0x0F9F7, 0x07ACB },
	{ 0x0F9F8, 0x07B20 },
	{ 0x0F9F9, 0x07C92 },
	{ 0x0F9FA, 0x072C0 },
	{ 0x0F9FB, 0x07099 },
	{ 0x0F9FC, 0x08B58 },
	{ 0x0F9FD, 0x04EC0 },
	{ 0x0F9FE, 0x08336 },
	{ 0x0F9FF, 0x0523A },
	{ 0x0FA00, 0x05207 },
	{ 0x0FA01, 0x05EA6 },
	{ 0x0FA02, 0x062D3 },
	{ 0x0FA03, 0x07CD6 },
	{ 0x0FA04, 0x05B85 },
	{ 0x0FA05, 0x06D1E },
	{ 0x0FA06, 0x066B4 },
	{ 0x0FA07, 0x08F3B },
	{ 0x0FA08, 0x0884C },
	{ 0x0FA09, 0x0964D },
	{ 0x0FA0A, 0x0898B },
	{ 0x0FA0B, 0x05ED3 },
	{ 0x0FA0C, 0x05140 },
	{ 0x0FA0D, 0x055C0 },
	{ 0x0FA10, 0x0585A },
	{ 0x0FA12, 0x06674 },
	{ 0x0FA15, 0x051DE },
	{ 0x0FA16, 0x0732A },
	{ 0x0FA17, 0x076CA },
	{ 0x0FA18, 0x0793C },
	{ 0x0FA19, 0x0795E },
	{ 0x0FA1A, 0x07965 },
	{ 0x0FA1B, 0x0798F },
	{ 0x0FA1C, 0x09756 },
	{ 0x0FA1D, 0x07CBE },
	{ 0x0FA1E, 0x07FBD },
	{ 0x0FA20, 0x08612 },
	{ 0x0FA22, 0x08AF8 },
	{ 0x0FA25, 0x09038 },
	{ 0x0FA26, 0x090FD },
	{ 0x0FA2A, 0x098EF },
	{ 0x0FA2B, 0x098FC },
	{ 0x0FA2C, 0x09928 },
	{ 0x0FA2D, 0x09DB4 },
	{ 0x0FA30, 0x04FAE },
	{ 0x0FA31, 0x050E7 },
	{ 0x0FA32, 0x0514D },
	{ 0x0FA33, 0x052C9 },
	{ 0x0FA34, 0x052E4 },
	{ 0x0FA35, 0x05351 },
	{ 0x0FA36, 0x0559D },
	{ 0x0FA37, 0x05606 },
	{ 0x0FA38, 0x05668 },
	{ 0x0FA39, 0x05840 },
	{ 0x0FA3A, 0x058A8 },
	{ 0x0FA3B, 0x05C64 },
	{ 0x0FA3C, 0x05C6E },
	{ 0x0FA3D, 0x06094 },
	{ 0x0FA3E, 0x06168 },
	{ 0x0FA3F, 0x0618E },
	{ 0x0FA40, 0x061F2 },
	{ 0x0FA41, 0x0654F },
	{ 0x0FA42, 0x065E2 },
	{ 0x0FA43, 0x06691 },
	{ 0x0FA44, 0x06885 },
	{ 0x0FA45, 0x06D77 },
	{ 0x0FA46, 0x06E1A },
	{ 0x0FA47, 0x06F22 },
	{ 0x0FA48, 0x0716E },
	{ 0x0FA49, 0x0722B },
	{ 0x0FA4A, 0x07422 },
	{ 0x0FA4B, 0x07891 },
	{ 0x0FA4C, 0x0793E },
	{ 0x0FA4D, 0x07949 },
	{ 0x0FA4E, 0x07948 },
	{ 0x0FA4F, 0x07950 },
	{ 0x0FA50, 0x07956 },
	{ 0x0FA51, 0x0795D },
	{ 0x0FA52, 0x0798D },
	{ 0x0FA53, 0x0798E },
	{ 0x0FA54, 0x07A40 },
	{ 0x0FA55, 0x07A81 },
	{ 0x0FA56, 0x07BC0 },
	{ 0x0FA57, 0x07DF4 },
	{ 0x0FA58, 0x07E09 },
	{ 0x0FA59, 0x07E41 },
	{ 0x0FA5A, 0x07F72 },
	{ 0x0FA5B, 0x08005 },
	{ 0x0FA5C, 0x081ED },
	{ 0x0FA5D, 0x08279 },
	{ 0x0FA5E, 0x08279 },
	{ 0x0FA5F, 0x08457 },
	{ 0x0FA60, 0x08910 },
	{ 0x0FA61, 0x08996 },
	{ 0x0FA62, 0x08B01 },
	{ 0x0FA63, 0x08B39 },
	{ 0x0FA64, 0x08CD3 },
	{ 0x0FA65, 0x08D08 },
	{ 0x0FA66, 0x08FB6 },
	{ 0x0FA67, 0x09038 },
	{ 0x0FA68, 0x096E3 },
	{ 0x0FA69, 0x097FF },
	{ 0x0FA6A, 0x0983B },
	{ 0x0FB1D, 0x005D9, 0x005B4 },
	{ 0x0FB1F, 0x005F2, 0x005B7 },
	{ 0x0FB2A, 0x005E9, 0x005C1 },
	{ 0x0FB2B, 0x005E9, 0x005C2 },
	{ 0x0FB2C, 0x005E9, 0x005BC, 0x005C1 },
	{ 0x0FB2D, 0x005E9, 0x005BC, 0x005C2 },
	{ 0x0FB2E, 0x005D0, 0x005B7 },
	{ 0x0FB2F, 0x005D0, 0x005B8 },
	{ 0x0FB30, 0x005D0, 0x005BC },
	{ 0x0FB31, 0x005D1, 0x005BC },
	{ 0x0FB32, 0x005D2, 0x005BC },
	{ 0x0FB33, 0x005D3, 0x005BC },
	{ 0x0FB34, 0x005D4, 0x005BC },
	{ 0x0FB35, 0x005D5, 0x005BC },
	{ 0x0FB36, 0x005D6, 0x005BC },
	{ 0x0FB38, 0x005D8, 0x005BC },
	{ 0x0FB39, 0x005D9, 0x005BC },
	{ 0x0FB3A, 0x005DA, 0x005BC },
	{ 0x0FB3B, 0x005DB, 0x005BC },
	{ 0x0FB3C, 0x005DC, 0x005BC },
	{ 0x0FB3E, 0x005DE, 0x005BC },
	{ 0x0FB40, 0x005E0, 0x005BC },
	{ 0x0FB41, 0x005E1, 0x005BC },
	{ 0x0FB43, 0x005E3, 0x005BC },
	{ 0x0FB44, 0x005E4, 0x005BC },
	{ 0x0FB46, 0x005E6, 0x005BC },
	{ 0x0FB47, 0x005E7, 0x005BC },
	{ 0x0FB48, 0x005E8, 0x005BC },
	{ 0x0FB49, 0x005E9, 0x005BC },
	{ 0x0FB4A, 0x005EA, 0x005BC },
	{ 0x0FB4B, 0x005D5, 0x005B9 },
	{ 0x0FB4C, 0x005D1, 0x005BF },
	{ 0x0FB4D, 0x005DB, 0x005BF },
	{ 0x0FB4E, 0x005E4, 0x005BF },
	{ 0x1D15E, 0x1D157, 0x1D165 },
	{ 0x1D15F, 0x1D158, 0x1D165 },
	{ 0x1D160, 0x1D158, 0x1D165, 0x1D16E },
	{ 0x1D161, 0x1D158, 0x1D165, 0x1D16F },
	{ 0x1D162, 0x1D158, 0x1D165, 0x1D170 },
	{ 0x1D163, 0x1D158, 0x1D165, 0x1D171 },
	{ 0x1D164, 0x1D158, 0x1D165, 0x1D172 },
	{ 0x1D1BB, 0x1D1B9, 0x1D165 },
	{ 0x1D1BC, 0x1D1BA, 0x1D165 },
	{ 0x1D1BD, 0x1D1B9, 0x1D165, 0x1D16E },
	{ 0x1D1BE, 0x1D1BA, 0x1D165, 0x1D16E },
	{ 0x1D1BF, 0x1D1B9, 0x1D165, 0x1D16F },
	{ 0x1D1C0, 0x1D1BA, 0x1D165, 0x1D16F },
	{ 0x2F800, 0x04E3D },
	{ 0x2F801, 0x04E38 },
	{ 0x2F802, 0x04E41 },
	{ 0x2F803, 0x20122 },
	{ 0x2F804, 0x04F60 },
	{ 0x2F805, 0x04FAE },
	{ 0x2F806, 0x04FBB },
	{ 0x2F807, 0x05002 },
	{ 0x2F808, 0x0507A },
	{ 0x2F809, 0x05099 },
	{ 0x2F80A, 0x050E7 },
	{ 0x2F80B, 0x050CF },
	{ 0x2F80C, 0x0349E },
	{ 0x2F80D, 0x2063A },
	{ 0x2F80E, 0x0514D },
	{ 0x2F80F, 0x05154 },
	{ 0x2F810, 0x05164 },
	{ 0x2F811, 0x05177 },
	{ 0x2F812, 0x2051C },
	{ 0x2F813, 0x034B9 },
	{ 0x2F814, 0x05167 },
	{ 0x2F815, 0x0518D },
	{ 0x2F816, 0x2054B },
	{ 0x2F817, 0x05197 },
	{ 0x2F818, 0x051A4 },
	{ 0x2F819, 0x04ECC },
	{ 0x2F81A, 0x051AC },
	{ 0x2F81B, 0x051B5 },
	{ 0x2F81C, 0x291DF },
	{ 0x2F81D, 0x051F5 },
	{ 0x2F81E, 0x05203 },
	{ 0x2F81F, 0x034DF },
	{ 0x2F820, 0x0523B },
	{ 0x2F821, 0x05246 },
	{ 0x2F822, 0x05272 },
	{ 0x2F823, 0x05277 },
	{ 0x2F824, 0x03515 },
	{ 0x2F825, 0x052C7 },
	{ 0x2F826, 0x052C9 },
	{ 0x2F827, 0x052E4 },
	{ 0x2F828, 0x052FA },
	{ 0x2F829, 0x05305 },
	{ 0x2F82A, 0x05306 },
	{ 0x2F82B, 0x05317 },
	{ 0x2F82C, 0x05349 },
	{ 0x2F82D, 0x05351 },
	{ 0x2F82E, 0x0535A },
	{ 0x2F82F, 0x05373 },
	{ 0x2F830, 0x0537D },
	{ 0x2F831, 0x0537F },
	{ 0x2F832, 0x0537F },
	{ 0x2F833, 0x0537F },
	{ 0x2F834, 0x20A2C },
	{ 0x2F835, 0x07070 },
	{ 0x2F836, 0x053CA },
	{ 0x2F837, 0x053DF },
	{ 0x2F838, 0x20B63 },
	{ 0x2F839, 0x053EB },
	{ 0x2F83A, 0x053F1 },
	{ 0x2F83B, 0x05406 },
	{ 0x2F83C, 0x0549E },
	{ 0x2F83D, 0x05438 },
	{ 0x2F83E, 0x05448 },
	{ 0x2F83F, 0x05468 },
	{ 0x2F840, 0x054A2 },
	{ 0x2F841, 0x054F6 },
	{ 0x2F842, 0x05510 },
	{ 0x2F843, 0x05553 },
	{ 0x2F844, 0x05563 },
	{ 0x2F845, 0x05584 },
	{ 0x2F846, 0x05584 },
	{ 0x2F847, 0x05599 },
	{ 0x2F848, 0x055AB },
	{ 0x2F849, 0x055B3 },
	{ 0x2F84A, 0x055C2 },
	{ 0x2F84B, 0x05716 },
	{ 0x2F84C, 0x05606 },
	{ 0x2F84D, 0x05717 },
	{ 0x2F84E, 0x05651 },
	{ 0x2F84F, 0x05674 },
	{ 0x2F850, 0x05207 },
	{ 0x2F851, 0x058EE },
	{ 0x2F852, 0x057CE },
	{ 0x2F853, 0x057F4 },
	{ 0x2F854, 0x0580D },
	{ 0x2F855, 0x0578B },
	{ 0x2F856, 0x05832 },
	{ 0x2F857, 0x05831 },
	{ 0x2F858, 0x058AC },
	{ 0x2F859, 0x214E4 },
	{ 0x2F85A, 0x058F2 },
	{ 0x2F85B, 0x058F7 },
	{ 0x2F85C, 0x05906 },
	{ 0x2F85D, 0x0591A },
	{ 0x2F85E, 0x05922 },
	{ 0x2F85F, 0x05962 },
	{ 0x2F860, 0x216A8 },
	{ 0x2F861, 0x216EA },
	{ 0x2F862, 0x059EC },
	{ 0x2F863, 0x05A1B },
	{ 0x2F864, 0x05A27 },
	{ 0x2F865, 0x059D8 },
	{ 0x2F866, 0x05A66 },
	{ 0x2F867, 0x036EE },
	{ 0x2F868, 0x036FC },
	{ 0x2F869, 0x05B08 },
	{ 0x2F86A, 0x05B3E },
	{ 0x2F86B, 0x05B3E },
	{ 0x2F86C, 0x219C8 },
	{ 0x2F86D, 0x05BC3 },
	{ 0x2F86E, 0x05BD8 },
	{ 0x2F86F, 0x05BE7 },
	{ 0x2F870, 0x05BF3 },
	{ 0x2F871, 0x21B18 },
	{ 0x2F872, 0x05BFF },
	{ 0x2F873, 0x05C06 },
	{ 0x2F874, 0x05F53 },
	{ 0x2F875, 0x05C22 },
	{ 0x2F876, 0x03781 },
	{ 0x2F877, 0x05C60 },
	{ 0x2F878, 0x05C6E },
	{ 0x2F879, 0x05CC0 },
	{ 0x2F87A, 0x05C8D },
	{ 0x2F87B, 0x21DE4 },
	{ 0x2F87C, 0x05D43 },
	{ 0x2F87D, 0x21DE6 },
	{ 0x2F87E, 0x05D6E },
	{ 0x2F87F, 0x05D6B },
	{ 0x2F880, 0x05D7C },
	{ 0x2F881, 0x05DE1 },
	{ 0x2F882, 0x05DE2 },
	{ 0x2F883, 0x0382F },
	{ 0x2F884, 0x05DFD },
	{ 0x2F885, 0x05E28 },
	{ 0x2F886, 0x05E3D },
	{ 0x2F887, 0x05E69 },
	{ 0x2F888, 0x03862 },
	{ 0x2F889, 0x22183 },
	{ 0x2F88A, 0x0387C },
	{ 0x2F88B, 0x05EB0 },
	{ 0x2F88C, 0x05EB3 },
	{ 0x2F88D, 0x05EB6 },
	{ 0x2F88E, 0x05ECA },
	{ 0x2F88F, 0x2A392 },
	{ 0x2F890, 0x05EFE },
	{ 0x2F891, 0x22331 },
	{ 0x2F892, 0x22331 },
	{ 0x2F893, 0x08201 },
	{ 0x2F894, 0x05F22 },
	{ 0x2F895, 0x05F22 },
	{ 0x2F896, 0x038C7 },
	{ 0x2F897, 0x232B8 },
	{ 0x2F898, 0x261DA },
	{ 0x2F899, 0x05F62 },
	{ 0x2F89A, 0x05F6B },
	{ 0x2F89B, 0x038E3 },
	{ 0x2F89C, 0x05F9A },
	{ 0x2F89D, 0x05FCD },
	{ 0x2F89E, 0x05FD7 },
	{ 0x2F89F, 0x05FF9 },
	{ 0x2F8A0, 0x06081 },
	{ 0x2F8A1, 0x0393A },
	{ 0x2F8A2, 0x0391C },
	{ 0x2F8A3, 0x06094 },
	{ 0x2F8A4, 0x226D4 },
	{ 0x2F8A5, 0x060C7 },
	{ 0x2F8A6, 0x06148 },
	{ 0x2F8A7, 0x0614C },
	{ 0x2F8A8, 0x0614E },
	{ 0x2F8A9, 0x0614C },
	{ 0x2F8AA, 0x0617A },
	{ 0x2F8AB, 0x0618E },
	{ 0x2F8AC, 0x061B2 },
	{ 0x2F8AD, 0x061A4 },
	{ 0x2F8AE, 0x061AF },
	{ 0x2F8AF, 0x061DE },
	{ 0x2F8B0, 0x061F2 },
	{ 0x2F8B1, 0x061F6 },
	{ 0x2F8B2, 0x06210 },
	{ 0x2F8B3, 0x0621B },
	{ 0x2F8B4, 0x0625D },
	{ 0x2F8B5, 0x062B1 },
	{ 0x2F8B6, 0x062D4 },
	{ 0x2F8B7, 0x06350 },
	{ 0x2F8B8, 0x22B0C },
	{ 0x2F8B9, 0x0633D },
	{ 0x2F8BA, 0x062FC },
	{ 0x2F8BB, 0x06368 },
	{ 0x2F8BC, 0x06383 },
	{ 0x2F8BD, 0x063E4 },
	{ 0x2F8BE, 0x22BF1 },
	{ 0x2F8BF, 0x06422 },
	{ 0x2F8C0, 0x063C5 },
	{ 0x2F8C1, 0x063A9 },
	{ 0x2F8C2, 0x03A2E },
	{ 0x2F8C3, 0x06469 },
	{ 0x2F8C4, 0x0647E },
	{ 0x2F8C5, 0x0649D },
	{ 0x2F8C6, 0x06477 },
	{ 0x2F8C7, 0x03A6C },
	{ 0x2F8C8, 0x0654F },
	{ 0x2F8C9, 0x0656C },
	{ 0x2F8CA, 0x2300A },
	{ 0x2F8CB, 0x065E3 },
	{ 0x2F8CC, 0x066F8 },
	{ 0x2F8CD, 0x06649 },
	{ 0x2F8CE, 0x03B19 },
	{ 0x2F8CF, 0x06691 },
	{ 0x2F8D0, 0x03B08 },
	{ 0x2F8D1, 0x03AE4 },
	{ 0x2F8D2, 0x05192 },
	{ 0x2F8D3, 0x05195 },
	{ 0x2F8D4, 0x06700 },
	{ 0x2F8D5, 0x0669C },
	{ 0x2F8D6, 0x080AD },
	{ 0x2F8D7, 0x043D9 },
	{ 0x2F8D8, 0x06717 },
	{ 0x2F8D9, 0x0671B },
	{ 0x2F8DA, 0x06721 },
	{ 0x2F8DB, 0x0675E },
	{ 0x2F8DC, 0x06753 },
	{ 0x2F8DD, 0x233C3 },
	{ 0x2F8DE, 0x03B49 },
	{ 0x2F8DF, 0x067FA },
	{ 0x2F8E0, 0x06785 },
	{ 0x2F8E1, 0x06852 },
	{ 0x2F8E2, 0x06885 },
	{ 0x2F8E3, 0x2346D },
	{ 0x2F8E4, 0x0688E },
	{ 0x2F8E5, 0x0681F },
	{ 0x2F8E6, 0x06914 },
	{ 0x2F8E7, 0x03B9D },
	{ 0x2F8E8, 0x06942 },
	{ 0x2F8E9, 0x069A3 },
	{ 0x2F8EA, 0x069EA },
	{ 0x2F8EB, 0x06AA8 },
	{ 0x2F8EC, 0x236A3 },
	{ 0x2F8ED, 0x06ADB },
	{ 0x2F8EE, 0x03C18 },
	{ 0x2F8EF, 0x06B21 },
	{ 0x2F8F0, 0x238A7 },
	{ 0x2F8F1, 0x06B54 },
	{ 0x2F8F2, 0x03C4E },
	{ 0x2F8F3, 0x06B72 },
	{ 0x2F8F4, 0x06B9F },
	{ 0x2F8F5, 0x06BBA },
	{ 0x2F8F6, 0x06BBB },
	{ 0x2F8F7, 0x23A8D },
	{ 0x2F8F8, 0x21D0B },
	{ 0x2F8F9, 0x23AFA },
	{ 0x2F8FA, 0x06C4E },
	{ 0x2F8FB, 0x23CBC },
	{ 0x2F8FC, 0x06CBF },
	{ 0x2F8FD, 0x06CCD },
	{ 0x2F8FE, 0x06C67 },
	{ 0x2F8FF, 0x06D16 },
	{ 0x2F900, 0x06D3E },
	{ 0x2F901, 0x06D77 },
	{ 0x2F902, 0x06D41 },
	{ 0x2F903, 0x06D69 },
	{ 0x2F904, 0x06D78 },
	{ 0x2F905, 0x06D85 },
	{ 0x2F906, 0x23D1E },
	{ 0x2F907, 0x06D34 },
	{ 0x2F908, 0x06E2F },
	{ 0x2F909, 0x06E6E },
	{ 0x2F90A, 0x03D33 },
	{ 0x2F90B, 0x06ECB },
	{ 0x2F90C, 0x06EC7 },
	{ 0x2F90D, 0x23ED1 },
	{ 0x2F90E, 0x06DF9 },
	{ 0x2F90F, 0x06F6E },
	{ 0x2F910, 0x23F5E },
	{ 0x2F911, 0x23F8E },
	{ 0x2F912, 0x06FC6 },
	{ 0x2F913, 0x07039 },
	{ 0x2F914, 0x0701E },
	{ 0x2F915, 0x0701B },
	{ 0x2F916, 0x03D96 },
	{ 0x2F917, 0x0704A },
	{ 0x2F918, 0x0707D },
	{ 0x2F919, 0x07077 },
	{ 0x2F91A, 0x070AD },
	{ 0x2F91B, 0x20525 },
	{ 0x2F91C, 0x07145 },
	{ 0x2F91D, 0x24263 },
	{ 0x2F91E, 0x0719C },
	{ 0x2F91F, 0x243AB },
	{ 0x2F920, 0x07228 },
	{ 0x2F921, 0x07235 },
	{ 0x2F922, 0x07250 },
	{ 0x2F923, 0x24608 },
	{ 0x2F924, 0x07280 },
	{ 0x2F925, 0x07295 },
	{ 0x2F926, 0x24735 },
	{ 0x2F927, 0x24814 },
	{ 0x2F928, 0x0737A },
	{ 0x2F929, 0x0738B },
	{ 0x2F92A, 0x03EAC },
	{ 0x2F92B, 0x073A5 },
	{ 0x2F92C, 0x03EB8 },
	{ 0x2F92D, 0x03EB8 },
	{ 0x2F92E, 0x07447 },
	{ 0x2F92F, 0x0745C },
	{ 0x2F930, 0x07471 },
	{ 0x2F931, 0x07485 },
	{ 0x2F932, 0x074CA },
	{ 0x2F933, 0x03F1B },
	{ 0x2F934, 0x07524 },
	{ 0x2F935, 0x24C36 },
	{ 0x2F936, 0x0753E },
	{ 0x2F937, 0x24C92 },
	{ 0x2F938, 0x07570 },
	{ 0x2F939, 0x2219F },
	{ 0x2F93A, 0x07610 },
	{ 0x2F93B, 0x24FA1 },
	{ 0x2F93C, 0x24FB8 },
	{ 0x2F93D, 0x25044 },
	{ 0x2F93E, 0x03FFC },
	{ 0x2F93F, 0x04008 },
	{ 0x2F940, 0x076F4 },
	{ 0x2F941, 0x250F3 },
	{ 0x2F942, 0x250F2 },
	{ 0x2F943, 0x25119 },
	{ 0x2F944, 0x25133 },
	{ 0x2F945, 0x0771E },
	{ 0x2F946, 0x0771F },
	{ 0x2F947, 0x0771F },
	{ 0x2F948, 0x0774A },
	{ 0x2F949, 0x04039 },
	{ 0x2F94A, 0x0778B },
	{ 0x2F94B, 0x04046 },
	{ 0x2F94C, 0x04096 },
	{ 0x2F94D, 0x2541D },
	{ 0x2F94E, 0x0784E },
	{ 0x2F94F, 0x0788C },
	{ 0x2F950, 0x078CC },
	{ 0x2F951, 0x040E3 },
	{ 0x2F952, 0x25626 },
	{ 0x2F953, 0x07956 },
	{ 0x2F954, 0x2569A },
	{ 0x2F955, 0x256C5 },
	{ 0x2F956, 0x0798F },
	{ 0x2F957, 0x079EB },
	{ 0x2F958, 0x0412F },
	{ 0x2F959, 0x07A40 },
	{ 0x2F95A, 0x07A4A },
	{ 0x2F95B, 0x07A4F },
	{ 0x2F95C, 0x2597C },
	{ 0x2F95D, 0x25AA7 },
	{ 0x2F95E, 0x25AA7 },
	{ 0x2F95F, 0x07AEE },
	{ 0x2F960, 0x04202 },
	{ 0x2F961, 0x25BAB },
	{ 0x2F962, 0x07BC6 },
	{ 0x2F963, 0x07BC9 },
	{ 0x2F964, 0x04227 },
	{ 0x2F965, 0x25C80 },
	{ 0x2F966, 0x07CD2 },
	{ 0x2F967, 0x042A0 },
	{ 0x2F968, 0x07CE8 },
	{ 0x2F969, 0x07CE3 },
	{ 0x2F96A, 0x07D00 },
	{ 0x2F96B, 0x25F86 },
	{ 0x2F96C, 0x07D63 },
	{ 0x2F96D, 0x04301 },
	{ 0x2F96E, 0x07DC7 },
	{ 0x2F96F, 0x07E02 },
	{ 0x2F970, 0x07E45 },
	{ 0x2F971, 0x04334 },
	{ 0x2F972, 0x26228 },
	{ 0x2F973, 0x26247 },
	{ 0x2F974, 0x04359 },
	{ 0x2F975, 0x262D9 },
	{ 0x2F976, 0x07F7A },
	{ 0x2F977, 0x2633E },
	{ 0x2F978, 0x07F95 },
	{ 0x2F979, 0x07FFA },
	{ 0x2F97A, 0x08005 },
	{ 0x2F97B, 0x264DA },
	{ 0x2F97C, 0x26523 },
	{ 0x2F97D, 0x08060 },
	{ 0x2F97E, 0x265A8 },
	{ 0x2F97F, 0x08070 },
	{ 0x2F980, 0x2335F },
	{ 0x2F981, 0x043D5 },
	{ 0x2F982, 0x080B2 },
	{ 0x2F983, 0x08103 },
	{ 0x2F984, 0x0440B },
	{ 0x2F985, 0x0813E },
	{ 0x2F986, 0x05AB5 },
	{ 0x2F987, 0x267A7 },
	{ 0x2F988, 0x267B5 },
	{ 0x2F989, 0x23393 },
	{ 0x2F98A, 0x2339C },
	{ 0x2F98B, 0x08201 },
	{ 0x2F98C, 0x08204 },
	{ 0x2F98D, 0x08F9E },
	{ 0x2F98E, 0x0446B },
	{ 0x2F98F, 0x08291 },
	{ 0x2F990, 0x0828B },
	{ 0x2F991, 0x0829D },
	{ 0x2F992, 0x052B3 },
	{ 0x2F993, 0x082B1 },
	{ 0x2F994, 0x082B3 },
	{ 0x2F995, 0x082BD },
	{ 0x2F996, 0x082E6 },
	{ 0x2F997, 0x26B3C },
	{ 0x2F998, 0x082E5 },
	{ 0x2F999, 0x0831D },
	{ 0x2F99A, 0x08363 },
	{ 0x2F99B, 0x083AD },
	{ 0x2F99C, 0x08323 },
	{ 0x2F99D, 0x083BD },
	{ 0x2F99E, 0x083E7 },
	{ 0x2F99F, 0x08457 },
	{ 0x2F9A0, 0x08353 },
	{ 0x2F9A1, 0x083CA },
	{ 0x2F9A2, 0x083CC },
	{ 0x2F9A3, 0x083DC },
	{ 0x2F9A4, 0x26C36 },
	{ 0x2F9A5, 0x26D6B },
	{ 0x2F9A6, 0x26CD5 },
	{ 0x2F9A7, 0x0452B },
	{ 0x2F9A8, 0x084F1 },
	{ 0x2F9A9, 0x084F3 },
	{ 0x2F9AA, 0x08516 },
	{ 0x2F9AB, 0x273CA },
	{ 0x2F9AC, 0x08564 },
	{ 0x2F9AD, 0x26F2C },
	{ 0x2F9AE, 0x0455D },
	{ 0x2F9AF, 0x04561 },
	{ 0x2F9B0, 0x26FB1 },
	{ 0x2F9B1, 0x270D2 },
	{ 0x2F9B2, 0x0456B },
	{ 0x2F9B3, 0x08650 },
	{ 0x2F9B4, 0x0865C },
	{ 0x2F9B5, 0x08667 },
	{ 0x2F9B6, 0x08669 },
	{ 0x2F9B7, 0x086A9 },
	{ 0x2F9B8, 0x08688 },
	{ 0x2F9B9, 0x0870E },
	{ 0x2F9BA, 0x086E2 },
	{ 0x2F9BB, 0x08779 },
	{ 0x2F9BC, 0x08728 },
	{ 0x2F9BD, 0x0876B },
	{ 0x2F9BE, 0x08786 },
	{ 0x2F9BF, 0x045D7 },
	{ 0x2F9C0, 0x087E1 },
	{ 0x2F9C1, 0x08801 },
	{ 0x2F9C2, 0x045F9 },
	{ 0x2F9C3, 0x08860 },
	{ 0x2F9C4, 0x08863 },
	{ 0x2F9C5, 0x27667 },
	{ 0x2F9C6, 0x088D7 },
	{ 0x2F9C7, 0x088DE },
	{ 0x2F9C8, 0x04635 },
	{ 0x2F9C9, 0x088FA },
	{ 0x2F9CA, 0x034BB },
	{ 0x2F9CB, 0x278AE },
	{ 0x2F9CC, 0x27966 },
	{ 0x2F9CD, 0x046BE },
	{ 0x2F9CE, 0x046C7 },
	{ 0x2F9CF, 0x08AA0 },
	{ 0x2F9D0, 0x08AED },
	{ 0x2F9D1, 0x08B8A },
	{ 0x2F9D2, 0x08C55 },
	{ 0x2F9D3, 0x27CA8 },
	{ 0x2F9D4, 0x08CAB },
	{ 0x2F9D5, 0x08CC1 },
	{ 0x2F9D6, 0x08D1B },
	{ 0x2F9D7, 0x08D77 },
	{ 0x2F9D8, 0x27F2F },
	{ 0x2F9D9, 0x20804 },
	{ 0x2F9DA, 0x08DCB },
	{ 0x2F9DB, 0x08DBC },
	{ 0x2F9DC, 0x08DF0 },
	{ 0x2F9DD, 0x208DE },
	{ 0x2F9DE, 0x08ED4 },
	{ 0x2F9DF, 0x08F38 },
	{ 0x2F9E0, 0x285D2 },
	{ 0x2F9E1, 0x285ED },
	{ 0x2F9E2, 0x09094 },
	{ 0x2F9E3, 0x090F1 },
	{ 0x2F9E4, 0x09111 },
	{ 0x2F9E5, 0x2872E },
	{ 0x2F9E6, 0x0911B },
	{ 0x2F9E7, 0x09238 },
	{ 0x2F9E8, 0x092D7 },
	{ 0x2F9E9, 0x092D8 },
	{ 0x2F9EA, 0x0927C },
	{ 0x2F9EB, 0x093F9 },
	{ 0x2F9EC, 0x09415 },
	{ 0x2F9ED, 0x28BFA },
	{ 0x2F9EE, 0x0958B },
	{ 0x2F9EF, 0x04995 },
	{ 0x2F9F0, 0x095B7 },
	{ 0x2F9F1, 0x28D77 },
	{ 0x2F9F2, 0x049E6 },
	{ 0x2F9F3, 0x096C3 },
	{ 0x2F9F4, 0x05DB2 },
	{ 0x2F9F5, 0x09723 },
	{ 0x2F9F6, 0x29145 },
	{ 0x2F9F7, 0x2921A },
	{ 0x2F9F8, 0x04A6E },
	{ 0x2F9F9, 0x04A76 },
	{ 0x2F9FA, 0x097E0 },
	{ 0x2F9FB, 0x2940A },
	{ 0x2F9FC, 0x04AB2 },
	{ 0x2F9FD, 0x29496 },
	{ 0x2F9FE, 0x0980B },
	{ 0x2F9FF, 0x0980B },
	{ 0x2FA00, 0x09829 },
	{ 0x2FA01, 0x295B6 },
	{ 0x2FA02, 0x098E2 },
	{ 0x2FA03, 0x04B33 },
	{ 0x2FA04, 0x09929 },
	{ 0x2FA05, 0x099A7 },
	{ 0x2FA06, 0x099C2 },
	{ 0x2FA07, 0x099FE },
	{ 0x2FA08, 0x04BCE },
	{ 0x2FA09, 0x29B30 },
	{ 0x2FA0A, 0x09B12 },
	{ 0x2FA0B, 0x09C40 },
	{ 0x2FA0C, 0x09CFD },
	{ 0x2FA0D, 0x04CCE },
	{ 0x2FA0E, 0x04CED },
	{ 0x2FA0F, 0x09D67 },
	{ 0x2FA10, 0x2A0CE },
	{ 0x2FA11, 0x04CF8 },
	{ 0x2FA12, 0x2A105 },
	{ 0x2FA13, 0x2A20E },
	{ 0x2FA14, 0x2A291 },
	{ 0x2FA15, 0x09EBB },
	{ 0x2FA16, 0x04D56 },
	{ 0x2FA17, 0x09EF9 },
	{ 0x2FA18, 0x09EFE },
	{ 0x2FA19, 0x09F05 },
	{ 0x2FA1A, 0x09F0F },
	{ 0x2FA1B, 0x09F16 },
	{ 0x2FA1C, 0x09F3B },
	{ 0x2FA1D, 0x2A600 },
};

/**
 * Are the first bytes of string `s' forming a valid UTF-8 character?
 *
 * @returns amount of bytes used to encode that character, or 0 if invalid.
 */
gint
utf8_is_valid_char(const gchar *s)
{
	const guchar u = (guchar) *s;
	gint len;
	gint slen;
	guint32 v;
	guint32 ov;

	if (UTF8_IS_ASCII(u))
		return 1;

	if (!UTF8_IS_START(u))
		return 0;

	len = UTF8_SKIP(s);

	if (len < 2 || !UTF8_IS_CONTINUATION(s[1]))
		return 0;

	for (slen = len - 1, s++, ov = v = u; slen; slen--, s++, ov = v) {
		if (!UTF8_IS_CONTINUATION(*s))
			return 0;
		v = UTF8_ACCUMULATE(v, *s);
		if (v < ov)
			return 0;
	}

	if (UNISKIP(v) < len)
		return 0;

	return len;
}

/*
 * utf8_is_valid_string
 *
 * Returns amount of UTF-8 chars when first `len' bytes of the given string
 * `s' form valid a UTF-8 string, 0 meaning the string is not valid UTF-8.
 *
 * If `len' is 0, the length is computed with strlen().
 */
gint utf8_is_valid_string(const gchar *s, gint len)
{
	const gchar *x = s;
	const gchar *s_end;
	gint n = 0;

	if (!len)
		len = strlen(s);
	g_assert(len >= 0);
	s_end = s + len;

	while (x < s_end) {
		gint clen = utf8_is_valid_char(x);
		if (clen == 0)
			return 0;
		x += clen;
		n++;
	}

	if (x != s_end)
		return 0;

	return n;
}

/**
 * @param uc the unicode character to encode.
 * @param buf the destination buffer. MUST BE at least 6 bytes long.
 * @returns 0 if the unicode character is invalid. Otherwise the
 *          length of the UTF-8 character is returned.
 */
gint
utf8_encode_char(guint32 uc, gchar *buf)
{
	guint len;
	gchar *p;
	
	g_assert(buf);

	if (UNICODE_IS_SURROGATE(uc)) {
		return 0;
	}

	len = UNISKIP(uc);
	if (len > 4) {
		len = 2;
		uc = UNI_REPLACEMENT;
	}
	
	g_assert(len > 0 && len <= 6);
	
	p = &buf[len];
	while (--p > buf) {
		*p = (uc | UTF8_BYTE_MARK) & UTF8_BYTE_MASK;
		uc >>= UTF8_ACCU_SHIFT;
	}
	*p = uc | UTF8_LENGTH_MARK(len);
	return len;
}

/*
 * utf8_decode_char
 *
 * Returns the character value of the first character in the string `s',
 * which is assumed to be in UTF-8 encoding and no longer than `len'.
 * `retlen' will be set to the length, in bytes, of that character.
 *
 * If `s' does not point to a well-formed UTF-8 character, the behaviour
 * is dependent on the value of `warn'.  When FALSE, it is assumed that
 * the caller will raise a warning, and this function will silently just
 * set `retlen' to -1 and return zero.
 */
guint32 utf8_decode_char(const gchar *s, gint len, gint *retlen, gboolean warn)
{
	guint32 v = *s;
	guint32 ov = 0;
	gint clen = 1;
	gint expectlen = 0;
	gint warning = -1;
	char msg[128];

	g_assert(s);

#define UTF8_WARN_EMPTY				0
#define UTF8_WARN_CONTINUATION		1
#define UTF8_WARN_NON_CONTINUATION	2
#define UTF8_WARN_FE_FF				3
#define UTF8_WARN_SHORT				4
#define UTF8_WARN_OVERFLOW			5
#define UTF8_WARN_SURROGATE			6
#define UTF8_WARN_BOM				7
#define UTF8_WARN_LONG				8
#define UTF8_WARN_FFFF				9

	if (len == 0) {
		warning = UTF8_WARN_EMPTY;
		goto malformed;
	}

	if (UTF8_IS_ASCII(v)) {
		if (retlen)
			*retlen = 1;
		return *s;
	}

	if (UTF8_IS_CONTINUATION(v)) {
		warning = UTF8_WARN_CONTINUATION;
		goto malformed;
	}

	if (UTF8_IS_START(v) && len > 1 && !UTF8_IS_CONTINUATION(s[1])) {
		warning = UTF8_WARN_NON_CONTINUATION;
		goto malformed;
	}

	if (v == 0xfe || v == 0xff) {
		warning = UTF8_WARN_FE_FF;
		goto malformed;
	}

	if      (!(v & 0x20)) { clen = 2; v &= 0x1f; }
	else if (!(v & 0x10)) { clen = 3; v &= 0x0f; }
	else if (!(v & 0x08)) { clen = 4; v &= 0x07; }
	else if (!(v & 0x04)) { clen = 5; v &= 0x03; }
	else if (!(v & 0x02)) { clen = 6; v &= 0x01; }
	else if (!(v & 0x01)) { clen = 7; v = 0; }

	if (retlen)
		*retlen = clen;

	expectlen = clen;

	if (len < expectlen) {
		warning = UTF8_WARN_SHORT;
		goto malformed;
	}

	for (clen--, s++, ov = v; clen; clen--, s++, ov = v) {
		if (!UTF8_IS_CONTINUATION(*s)) {
			s--;
			warning = UTF8_WARN_NON_CONTINUATION;
			goto malformed;
		} else
			v = UTF8_ACCUMULATE(v, *s);

		if (v < ov) {
			warning = UTF8_WARN_OVERFLOW;
			goto malformed;
		} else if (v == ov) {
			warning = UTF8_WARN_LONG;
			goto malformed;
		}
	}

	if (UNICODE_IS_SURROGATE(v)) {
		warning = UTF8_WARN_SURROGATE;
		goto malformed;
	} else if (UNICODE_IS_BYTE_ORDER_MARK(v)) {
		warning = UTF8_WARN_BOM;
		goto malformed;
	} else if (expectlen > UNISKIP(v)) {
		warning = UTF8_WARN_LONG;
		goto malformed;
	} else if (UNICODE_IS_ILLEGAL(v)) {
		warning = UTF8_WARN_FFFF;
		goto malformed;
	}

	return v;

malformed:

	if (!warn) {
		if (retlen)
			*retlen = -1;
		return 0;
	}

	switch (warning) {
	case UTF8_WARN_EMPTY:
		gm_snprintf(msg, sizeof(msg), "empty string");
		break;
	case UTF8_WARN_CONTINUATION:
		gm_snprintf(msg, sizeof(msg),
			"unexpected continuation byte 0x%02lu", (gulong) v);
		break;
	case UTF8_WARN_NON_CONTINUATION:
		gm_snprintf(msg, sizeof(msg),
			"unexpected non-continuation byte 0x%02lu "
			"after start byte 0x%02ld", (gulong) s[1], (gulong) v);
		break;
	case UTF8_WARN_FE_FF:
		gm_snprintf(msg, sizeof(msg), "byte 0x%02lu", (gulong) v);
		break;
	case UTF8_WARN_SHORT:
		gm_snprintf(msg, sizeof(msg), "%d byte%s, need %d",
			len, len == 1 ? "" : "s", expectlen);
		break;
	case UTF8_WARN_OVERFLOW:
		gm_snprintf(msg, sizeof(msg), "overflow at 0x%02lu, byte 0x%02lu",
			(gulong) ov, (gulong) *s);
		break;
	case UTF8_WARN_SURROGATE:
		gm_snprintf(msg, sizeof(msg), "UTF-16 surrogate 0x04%lu", (gulong) v);
		break;
	case UTF8_WARN_BOM:
		gm_snprintf(msg, sizeof(msg), "byte order mark 0x%04lu", (gulong) v);
		break;
	case UTF8_WARN_LONG:
		gm_snprintf(msg, sizeof(msg), "%d byte%s, need %d",
			expectlen, expectlen == 1 ? "" : "s", UNISKIP(v));
		break;
	case UTF8_WARN_FFFF:
		gm_snprintf(msg, sizeof(msg), "character 0x%04lu", (gulong) v);
		break;
	default:
		gm_snprintf(msg, sizeof(msg), "unknown reason");
		break;
	}

	g_warning("malformed UTF-8 character: %s", msg);

	if (retlen)
		*retlen = expectlen ? expectlen : len;

	return 0;
}

/*
 * utf8_to_iso8859
 *
 * Convert UTF-8 string to ISO-8859-1 inplace.  If `space' is TRUE, all
 * characters outside the U+0000 .. U+00FF range are turned to space U+0020.
 * Otherwise, we stop at the first out-of-range character.
 *
 * If `len' is 0, the length of the string is computed with strlen().
 *
 * Returns length of decoded string.
 */
gint utf8_to_iso8859(gchar *s, gint len, gboolean space)
{
	gchar *x = s;
	gchar *xw = s;			/* Where we write back ISO-8859 chars */
	gchar *s_end;

	if (!len)
		len = strlen(s);
	s_end = s + len;

	while (x < s_end) {
		gint clen;
		guint32 v = utf8_decode_char(x, len, &clen, FALSE);

		if (clen == -1)
			break;

		g_assert(clen >= 1);

		if (v & 0xffffff00) {	/* Not an ISO-8859-1 character */
			if (!space)
				break;
			v = 0x20;
		}

		*xw++ = (guchar) v;
		x += clen;
		len -= clen;
	}

	*xw = '\0';

	return xw - s;
}


#if defined(USE_GLIB2)

static const char *
get_locale_charset(void)
{
	const char *cs = NULL;
	
	g_get_charset(&cs);
	return cs;
}
	
#else /* !USE_GLIB2 */

#if defined(I_LIBCHARSET)
#define get_locale_charset() locale_charset()
#else /* !I_LIBCHARSET */

/* List of known codesets. The first word of each string is the alias to be
 * returned. The words are seperated by whitespaces.
 */
static const char *codesets[] = {
 "ASCII ISO_646.IRV:1983 646 C US-ASCII la_LN.ASCII lt_LN.ASCII",
 "BIG5 big5 big5 zh_TW.BIG5 zh_TW.Big5",
 "CP1046 IBM-1046",
 "CP1124 IBM-1124",
 "CP1129 IBM-1129",
 "CP1252 IBM-1252",
 "CP437 en_NZ en_US",
 "CP775 lt lt_LT lv lv_LV",
 "CP850 IBM-850 cp850 ca ca_ES de de_AT de_CH de_DE en en_AU en_CA en_GB "
	"en_ZA es es_AR es_BO es_CL es_CO es_CR es_CU es_DO es_EC es_ES es_GT "
	"es_HN es_MX es_NI es_PA es_PY es_PE es_SV es_UY es_VE et et_EE eu eu_ES "
	"fi fi_FI fr fr_BE fr_CA fr_CH fr_FR ga ga_IE gd gd_GB gl gl_ES id id_ID " 
	"it it_CH it_IT nl nl_BE nl_NL pt pt_BR pt_PT sv sv_SE mt mt_MT eo eo_EO",
 "CP852 cs cs_CZ hr hr_HR hu hu_HU pl pl_PL ro ro_RO sk sk_SK sl sl_SI "
	"sq sq_AL sr sr_YU",
 "CP856 IBM-856",
 "CP857 tr tr_TR",
 "CP861 is is_IS",
 "CP862 he he_IL",
 "CP864 ar ar_AE ar_DZ ar_EG ar_IQ ar_IR ar_JO ar_KW ar_MA ar_OM ar_QA "
	"ar_SA ar_SY",
 "CP865 da da_DK nb nb_NO nn nn_NO no no_NO",
 "CP866 ru_RU.CP866 ru_SU.CP866 be be_BE bg bg_BG mk mk_MK ru ru_RU ",
 "CP869 el el_GR",
 "CP874 th th_TH",
 "CP922 IBM-922",
 "CP932 IBM-932 ja ja_JP",
 "CP943 IBM-943",
 "CP949 KSC5601 kr kr_KR",
 "CP950 zh_TW",
 "DEC-HANYU dechanyu",
 "DEC-KANJI deckanji",
 "EUC-JP IBM-eucJP eucJP eucJP sdeckanji ja_JP.EUC",
 "EUC-KR IBM-eucKR eucKR eucKR deckorean 5601 ko_KR.EUC",
 "EUC-TW IBM-eucTW eucTW eucTW cns11643",
 "GB2312 IBM-eucCN hp15CN eucCN dechanzi gb2312 zh_CN.EUC",
 "GBK zh_CN",
 "HP-ARABIC8 arabic8",
 "HP-GREEK8 greek8",
 "HP-HEBREW8 hebrew8",
 "HP-KANA8 kana8",
 "HP-ROMAN8 roman8 ",
 "HP-TURKISH8 turkish8 ",
 "ISO-8859-1 ISO8859-1 iso88591 da_DK.ISO_8859-1 de_AT.ISO_8859-1 "
	"de_CH.ISO_8859-1 de_DE.ISO_8859-1 en_AU.ISO_8859-1 en_CA.ISO_8859-1 "
	"en_GB.ISO_8859-1 en_US.ISO_8859-1 es_ES.ISO_8859-1 fi_FI.ISO_8859-1 "
	"fr_BE.ISO_8859-1 fr_CA.ISO_8859-1 fr_CH.ISO_8859-1 fr_FR.ISO_8859-1 "
	"is_IS.ISO_8859-1 it_CH.ISO_8859-1 it_IT.ISO_8859-1 la_LN.ISO_8859-1 "
	"lt_LN.ISO_8859-1 nl_BE.ISO_8859-1 nl_NL.ISO_8859-1 no_NO.ISO_8859-1 "
	"pt_PT.ISO_8859-1 sv_SE.ISO_8859-1",
 "ISO-8859-13 IBM-921",
 "ISO-8859-14 ISO_8859-14 ISO_8859-14:1998 iso-ir-199 latin8 iso-celtic l8",
 "ISO-8859-15 ISO8859-15 iso885915 da_DK.DIS_8859-15 de_AT.DIS_8859-15 "
	"de_CH.DIS_8859-15 de_DE.DIS_8859-15 en_AU.DIS_8859-15 en_CA.DIS_8859-15 "
	"en_GB.DIS_8859-15 en_US.DIS_8859-15 es_ES.DIS_8859-15 fi_FI.DIS_8859-15 "
	"fr_BE.DIS_8859-15 fr_CA.DIS_8859-15 fr_CH.DIS_8859-15 fr_FR.DIS_8859-15 "
	"is_IS.DIS_8859-15 it_CH.DIS_8859-15 it_IT.DIS_8859-15 la_LN.DIS_8859-15 "
	"lt_LN.DIS_8859-15 nl_BE.DIS_8859-15 nl_NL.DIS_8859-15 no_NO.DIS_8859-15 "
	"pt_PT.DIS_8859-15 sv_SE.DIS_8859-15",
 "ISO-8859-2 ISO8859-2 iso88592 cs_CZ.ISO_8859-2 hr_HR.ISO_8859-2 "
	"hu_HU.ISO_8859-2 la_LN.ISO_8859-2 lt_LN.ISO_8859-2 pl_PL.ISO_8859-2 "
	"sl_SI.ISO_8859-2",
 "ISO-8859-4 ISO8859-4 la_LN.ISO_8859-4 lt_LT.ISO_8859-4",
 "ISO-8859-5 ISO8859-5 iso88595 ru_RU.ISO_8859-5 ru_SU.ISO_8859-5",
 "ISO-8859-6 ISO8859-6 iso88596",
 "ISO-8859-7 ISO8859-7 iso88597",
 "ISO-8859-8 ISO8859-8 iso88598",
 "ISO-8859-9 ISO8859-9 iso88599",
 "KOI8-R koi8-r ru_RU.KOI8-R ru_SU.KOI8-R",
 "KOI8-U uk_UA.KOI8-U",
 "SHIFT_JIS SJIS PCK ja_JP.SJIS ja_JP.Shift_JIS",
 "TIS-620 tis620 TACTIS TIS620.2533",
 "UTF-8 utf8 *",
 NULL
};

/*
 * locale_charset:
 *
 * Returns a string representing the current locale as an alias which is
 * understood by GNU iconv. The returned pointer points to a static buffer.
 */
const char *get_locale_charset(void)
{
	int i = 0;
	const char *cs;
	const char *start = codesets[0]; 
	const char *first_end = NULL;
	size_t cs_len;

	cs = nl_langinfo(CODESET);
	if (NULL == cs || '\0' == *cs)
		return NULL;

	cs_len = strlen(cs);

	while (NULL != codesets[i]) {
		static char buf[64];
		const char *end;
		size_t len;
		
		end = strchr(start, ' ');
		if (NULL == end)
			end = strchr(start, '\0');
		if (NULL == first_end)
			first_end = end;

 		len = end - start;
		if (len > 0 && 0 == g_ascii_strncasecmp(cs, start, cs_len)) {
			len = first_end - codesets[i] + 1;
			g_strlcpy(buf, codesets[i], MIN(len, sizeof(buf)));
			return buf;
		}
		if ('\0' == *end) {
			first_end = NULL;
			start = codesets[++i];
		} else
			start = end + 1;
		
	}
	return NULL;
}
#endif /* I_LIBCHARSET */

#endif /* USE_GLIB2 */

const gchar *locale_get_charset(void)
{
	return charset;
}

static void textdomain_init(const char *codeset)
{
#ifdef ENABLE_NLS
	bindtextdomain(PACKAGE, LOCALE_EXP);

#ifdef HAS_BIND_TEXTDOMAIN_CODESET

#ifdef USE_GLIB2	
	codeset = "UTF-8";
#endif /* USE_GLIB2*/

	bind_textdomain_codeset(PACKAGE, codeset);

#endif /* HAS_BIND_TEXTDOMAIN_CODESET */

	textdomain(PACKAGE);
	
#else /* !NLS */
	(void) codeset;
#endif /* NLS */
}

void locale_init(void)
{

	unicode_decompose_init();

	setlocale(LC_ALL, "");
	charset = get_locale_charset();
	
	if (charset == NULL) {
		/* Default locale codeset */
		charset = "ISO-8859-1";
		g_warning("locale_init: Using default codeset %s as fallback.",
			charset);
	}

	g_message("using charset \"%s\"", charset);
	textdomain_init(charset);

	if ((GIConv)-1 == (cd_latin_to_utf8 = g_iconv_open("UTF-8", "ISO-8859-1")))
		g_warning("g_iconv_open(\"UTF-8\", \"ISO-8859-1\") failed.");
	if (0 != strcmp("ISO-8859-1", charset)) {
		if ((GIConv)-1 == (cd_locale_to_utf8 = g_iconv_open("UTF-8", charset)))
			g_warning("g_iconv_open(\"UTF-8\", \"%s\") failed.", charset);
	} else {
		cd_locale_to_utf8 = cd_latin_to_utf8;
	}
	if ((GIConv)-1 == (cd_utf8_to_locale = g_iconv_open(charset, "UTF-8")))
		g_warning("g_iconv_open(\"%s\", \"UTF-8\") failed.", charset);

#ifdef USE_ICU
	{
		UErrorCode errorCode = U_ZERO_ERROR;

		/* set up the locale converter */
		conv_icu_locale = ucnv_open(charset, &errorCode);
		if (U_FAILURE(errorCode))
			g_warning("ucnv_open for locale failed with %d", errorCode);

		/* set up the UTF-8 converter */
		conv_icu_utf8 = ucnv_open("utf8", &errorCode);
		if (U_FAILURE(errorCode))
			g_warning("ucnv_open for utf-8 failed with %d", errorCode);
	}
#endif
}

/*
 * locale_close
 *
 * Called at shutdown time.
 */
void locale_close(void)
{
#ifdef USE_ICU
	if (conv_icu_locale) {
	  ucnv_close(conv_icu_locale);
	  conv_icu_locale=NULL;
	}
	if (conv_icu_utf8) {
	  ucnv_close(conv_icu_utf8);
	  conv_icu_utf8=NULL;
	}
#endif
}

static inline char *g_iconv_complete(GIConv cd,
	const char *inbuf, size_t inbytes_left,
	char *outbuf, size_t outbytes_left)
{
#if 0
	/* This is the appropriate replacement unicode character 0xFFFD but
	 * it looks awkward (a modified question mark) in a non-unicode
	 * context. So rather use a underscore for filenames.
	 */
	static const gchar replacement[] = { 0xEF, 0xBF, 0xBD };
#else
	static const gchar replacement[] = { '_' };
#endif
	gchar *result = outbuf;

	if ((GIConv) -1 == cd)
		return NULL;

	if (outbytes_left > 0)
		outbuf[0] = '\0';

	while (inbytes_left > 0 && outbytes_left > 1) {
		size_t ret;

		ret = g_iconv(cd, (gpointer) &inbuf, &inbytes_left,
		    &outbuf, &outbytes_left);

		if ((size_t) -1 == ret) {
			switch (errno) {
			case EILSEQ:
			case EINVAL:
				if (common_dbg > 1)
					g_warning("g_iconv_complete: g_iconv() failed soft: %s",
						g_strerror(errno));

				if (outbytes_left > sizeof replacement) {
					inbuf++;
					inbytes_left--;
					memcpy(outbuf, replacement, sizeof replacement);
					outbuf += sizeof replacement;
					outbytes_left -= sizeof replacement;
				} else {
					outbytes_left = 1;
				}
				break;
			default:
				if (common_dbg > 1)
					g_warning("g_iconv_complete(): g_iconv() failed hard: %s",
						g_strerror(errno));
				return NULL;
			}
		}
	}
	*outbuf = '\0';
	return result;
}

/*
 * locale_to_utf8
 *
 * If ``len'' is 0 the length will be calculated using strlen(), otherwise
 * only ``len'' characters will be converted.
 * If the string is already valid UTF-8 it will be returned "as-is".
 * The function might return a pointer to a STATIC buffer! If the output
 * string is longer than 4095 characters it will be truncated.
 * Non-convertible characters will be replaced by '_'. The returned string
 * WILL be NUL-terminated in any case.
 *
 * In case of an unrecoverable error, NULL is returned.
 *
 * ATTENTION:	Don't use this function for anything but *uncritical*
 *				strings	e.g., to view strings in the GUI. The conversion
 *				MAY be inappropriate!
 */
gchar *locale_to_utf8(const gchar *str, size_t len)
{
	static gchar outbuf[4096 + 6]; /* an UTF-8 char is max. 6 bytes large */

	g_assert(NULL != str);

	if (0 == len)
		len = strlen(str);
	if (utf8_is_valid_string(str, len))
		return (gchar *) str; /* Override const */
	else
		return g_iconv_complete(cd_locale_to_utf8,
				str, len, outbuf, sizeof(outbuf) - 7);
}

/**
 * Converts a string from the current locale encoding to UTF-8 encoding
 * with all characters decomposed.
 *
 * @param str the string to convert.
 * @param len the length of ``str''. May be set to zero if ``str'' is 
 *		  NUL-terminated.
 *
 * @returns a newly allocated string.
 */
gchar *
locale_to_utf8_nfd(const gchar *str, size_t len)
{
	char sbuf[4096];
	const gchar *s;
	gchar *ret;
	size_t utf8_len;
	
	g_assert(NULL != str);

	if (0 == len)
		len = strlen(str);
	if (0 == len)
		return g_strdup("");

   	utf8_len = len * 6 + 1;
	if (utf8_is_valid_string(str, len)) {
		s = str;
	} else {
		gchar *p;

		g_assert((utf8_len - 1) / 6 == len);
		p = len < sizeof sbuf ? sbuf : g_malloc(utf8_len);
		s = g_iconv_complete(cd_locale_to_utf8, str, len, p, utf8_len);
	}

	/*
	 * Do a dry run first to determine the length. The output buffer won't
	 * be touched, but it must be valid. So just pass ``s'' not NULL.
	 */
	len = utf8_decompose_nfd(s, /* fake */ (gchar *) s, 0);
	utf8_len = len + 1;
	ret = g_malloc(utf8_len);
	len = utf8_decompose_nfd(s, ret, utf8_len);
	g_assert(len < utf8_len);
	
	if (s != str && s != sbuf) {
		g_free((gchar *) s); /* Override const */
		s = NULL;
	}

	return ret;
}


gchar *utf8_to_locale(const gchar *str, size_t len)
{
	static gchar outbuf[4096 + 6]; /* a multibyte char is max. 6 bytes large */

	g_assert(NULL != str);

	return g_iconv_complete(cd_utf8_to_locale,
				str, len != 0 ? len : strlen(str), outbuf, sizeof(outbuf) - 7);
}

gboolean is_ascii_string(const gchar *str)
{
	gint c;

	while ((c = (guchar) *str++))
		if (c & 0x80)
	        return FALSE;

    return TRUE;
}

gchar *iso_8859_1_to_utf8(const gchar *fromstr)
{
	static gchar outbuf[4096 + 6]; /* a multibyte char is max. 6 bytes large */

	g_assert(NULL != fromstr);
 
	return g_iconv_complete(cd_latin_to_utf8,
				fromstr, strlen(fromstr), outbuf, sizeof(outbuf) - 7);
}

gchar *lazy_utf8_to_locale(const gchar *str, size_t len)
{
	gchar *t = utf8_to_locale(str, len);
	return NULL != t ? t : "<Cannot convert to locale>";
}

/**
 * Converts the supplied string ``str'' from the current locale encoding
 * to a UTF-8 string.  If compiled for GTK+ 2.x this function does also
 * enforce NFC.
 *
 * @param str the string to convert.
 * @param len the length of ``str''. May be set to zero if ``str'' is 
 *		  NUL-terminated.
 *
 * @returns the converted string or ``str'' if no conversion was necessary.
 */
gchar *
lazy_locale_to_utf8(const gchar *str, size_t len)
{
	const gchar *s;
	
	/* Let's assume that most of the supplied strings are pure ASCII. */
	if (is_ascii_string(str))
		return (gchar *) str; /* Override const */

	s = locale_to_utf8(str, len);
	if (!s)
		return "<Cannot convert to UTF-8>";

#ifdef USE_GLIB2
	{
		static gchar buf[4096 + 6];
		gchar *s_nfc;

		/* Enforce UTF-8 NFC because GTK+ would render a decomposed string
		 * otherwise (if the string is not in NFC). This is a known GTK+ bug.
		 */

		s_nfc = g_utf8_normalize(s, (gssize) -1, G_NORMALIZE_NFC);
		if (0 != strcmp(s, s_nfc)) {
			g_strlcpy(buf, s_nfc, sizeof buf);
			s = buf;
		}
		G_FREE_NULL(s_nfc);
	}
#endif	/* USE_GLIB2 */

	return (gchar *) s; /* Override const */
}

static void
unicode_decompose_init(void)
{
#if 0 && defined(USE_GLIB2)
	guint i;

	/* Check all single Unicode characters */
	for (i = 0; i <= 0x10FFFD; i++) {
		guint size;
		gchar buf[256];
		gchar utf8_char[7];
		gchar *s;

		if (
			UNICODE_IS_SURROGATE(i) ||
			UNICODE_IS_BYTE_ORDER_MARK(i) ||
			UNICODE_IS_ILLEGAL(i)
		) {
			continue;
		}

		size = g_unichar_to_utf8(i, utf8_char);
		g_assert((gint) size >= 0 && size < sizeof utf8_char);
		utf8_char[size] = '\0';
		utf8_decompose_nfd(utf8_char, buf, G_N_ELEMENTS(buf));
		s = g_utf8_normalize(utf8_char, -1, G_NORMALIZE_NFD);
		if (strcmp(s, buf)) {
			g_message("\n0x%04X\nbuf=\"%s\"\ns=\"%s\"", i, buf, s);

#if (GLIB_MAJOR_VERSION > 2) || (GLIB_MINOR_VERSION >= 4) /* Glib >= 2.4.0 */
			/*
			 * The normalized strings should be identical. However, older
			 * versions of GLib do not normalize some characters properly.
			 */
			G_BREAKPOINT();
#endif /* GLib >= 2.4.0 */
			
		}
		G_FREE_NULL(s);
	}
#endif /* USE_GLIB2 */
}

/**
 * This is a highly specialized function (read: don't use it if you don't
 * understand what it does and how it's used) to be used with
 * utf8_decode_char().
 * It's purpose is to determine the maximum possible length in bytes of
 * current UTF-8 character that ``s'' points to.
 *
 * @param s a UTF-8 encoded string.
 * @param len number of bytes pending to be decoded.
 *
 * @returns the maximum length in bytes of the current UTF-8 character.
 */
static inline size_t
utf8_decode_lookahead(const gchar *s, size_t len)
{
	while (len < 6 && s[len] != '\0')
		len++;
	return len;
}

/**
 * Converts a UTF-8 encoded string to a UTF-32 encoded string. The
 * target string ``out'' is always be zero-terminated unless ``size''
 * is zero.
 *
 * @param in the UTF-8 input string.
 * @param out the target buffer for converted UTF-32 string.
 * @param size the length of the outbuf buffer - characters not bytes!
 *        Whether the buffer was too small can be checked by comparing
 *        ``size'' with the return value. The value of ``size'' MUST NOT
 *        exceed INT_MAX.
 *
 * @returns the length of completely converted string.
 */
size_t
utf8_to_utf32(const gchar *in, guint32 *out, size_t size)
{
	const gchar *s = in;
	guint32 *p = out;
	guint retlen;
	size_t len = 0;

	g_assert(in != NULL);	
	g_assert(out != NULL);	
	g_assert(size <= INT_MAX);

	if (size > 0) {
		while (*s != '\0' && --size > 0) {
			len = utf8_decode_lookahead(s, len);
			*p++ = utf8_decode_char(s, len, &retlen, TRUE);
			s += retlen;
			len -= retlen;
		}
		*p = 0x0000;
	}

	while (*s != '\0') {
		len = utf8_decode_lookahead(s, len);
		utf8_decode_char(s, len, &retlen, TRUE);
		s += retlen;
		len -= retlen;
		p++;
	}
	
	return p - out;
}

/**
 * Converts a UTF-32 encoded string to a UTF-8 encoded string. The
 * target string ``out'' is always be zero-terminated unless ``size''
 * is zero.
 *
 * @param in the UTF-32 input string.
 * @param out the target buffer for converted UTF-8 string.
 * @param size the length of the outbuf buffer - characters not bytes!
 *        Whether the buffer was too small can be checked by comparing
 *        ``size'' with the return value. The value of ``size'' MUST NOT
 *        exceed INT_MAX.
 *
 * @returns the length in bytes of completely converted string.
 */
size_t
utf32_to_utf8(const guint32 *in, gchar *out, size_t size)
{
	const guint32 *s = in;
	gchar *p = out;
	guint retlen;
	guint32 uc;
	gchar utf8_buf[7];

	g_assert(in != NULL);
	g_assert(out != NULL);	
	g_assert(size <= INT_MAX);

	if (size > 0) {
		do {
			uc = *s;
			retlen = utf8_encode_char(uc, utf8_buf);
			if (uc == 0x0000 || retlen > size)
				break;
			memcpy(p, utf8_buf, retlen);
			p += retlen;
			size -= retlen;
			s++;
		} while (size > 0);
		*p = '\0';
	}

	while ((uc = *s) != 0x0000) {
		retlen = utf8_encode_char(uc, utf8_buf);
		p += retlen;
	}
	
	return p - out;
}

/**
 * Looks up the decomposed string for a UTF-32 character.
 *
 * @param uc the unicode character to look up.
 *
 * @returns NULL if the character is not in decomposition table. Otherwise,
 *          the returned pointer points to a possibly unterminated UTF-32
 *			string of maximum UTF32_NFD_REPLACE_MAXLEN characters. The result
 *			is constant.
 */
static inline const guint32 *
utf32_nfd_lookup(guint32 uc)
{
	size_t i, j = 0, k = G_N_ELEMENTS(utf32_nfd_lut);
	guint32 ch;

	/* Perform a binary search to find ``uc'' */
	do {
		i = j + (k >> 1);
		ch = utf32_nfd_lut[i][0];
		if (ch == uc)
			return &utf32_nfd_lut[i][1];

		if (ch < uc) {
			j = i + 1;
			k--;
		}
		k >>= 1;
	} while (k != 0);

	return NULL;
}

/**
 * Decomposes a single UTF-32 character.
 *
 * @param uc the UTF-32 to decompose.
 * @param len the variable ``len'' points to will be set to
 *        length in characters (not bytes!) of decomposed string. This is
 *        important because the decomposed string is not zero-terminated.
 *
 * @returns a pointer to a buffer holding the decomposed string.
 *			The buffer is unterminated. The maximum length is
 *			UTF32_NFD_REPLACE_MAXLEN characters. The returned pointer points
 *			to a static buffer which might get overwritten by subsequent
 *			calls to this function.
 */
static inline const guint32 *
utf32_decompose_nfd_char(guint32 uc, size_t *len)
{
	static guint32 buf[3];
	guint32 *p = buf;
	const guint32 *q;

	if (UNICODE_IS_ASCII(uc)) {
		*p++ = uc;
	} else if (UNICODE_IS_HANGUL(uc)) {
		/*
		 * Take advantage of algorithmic Hangul decomposition to reduce
		 * the size of the lookup table drastically. See also:
		 *
		 * 		http://www.unicode.org/reports/tr15/#Hangul
		 */
#define T_COUNT 28
#define V_COUNT 21
#define N_COUNT (T_COUNT * V_COUNT)
		static const guint32 l_base = 0x1100;
		static const guint32 v_base = 0x1161;
		static const guint32 t_base = 0x11A7;
		const guint32 i = uc - UNI_HANGUL_FIRST;
		guint32 l = l_base + i / N_COUNT;
		guint32 v = v_base + (i % N_COUNT) / T_COUNT;
		guint32 t = t_base + i % T_COUNT;

		*p++ = l;
		*p++ = v;

		if (t != t_base) {
			*p++ = t;
		}
#undef N_COUNT
#undef V_COUNT
#undef T_COUNT
	} else if (NULL != (q = utf32_nfd_lookup(uc))) {
		*len = utf32_strmaxlen(q, UTF32_NFD_REPLACE_MAXLEN);
		return q;
	} else {
		*p++ = uc;
	}

	g_assert(p > buf && p < &buf[sizeof buf]);
	*len = p - buf;
	return buf;
}

/**
 * Determines the length of a UTF-32 string.
 *
 * @param s a UTF-32 string.
 * @returns the length in characters (not bytes!) of the string ``s''.
 */
size_t
utf32_strlen(const guint32 *s)
{
	const guint32 *p = s;

	g_assert(s != NULL);

	while (*p != 0x0000)
		p++;

	return p - s;
}

/**
 * Determines the length of a UTF-32 string inspecting at most ``maxlen''
 * characters (not bytes!). This can safely be used with unterminated UTF-32
 * strings if ``maxlen'' has an appropriate value.
 *
 * To detect whether the actual string is longer than ``maxlen'' characters,
 * just check if ``string[maxlen]'' is 0x0000, if and only if the returned
 * value equals maxlen. Otherwise, the returned value is indeed the
 * complete length of the UTF-32 string.
 *
 * @param s a UTF-32 string.
 * @param maxlen the maximum number of characters to inspect.
 *
 * @returns the length in characters (not bytes!) of the string ``s''.
 */
size_t
utf32_strmaxlen(const guint32 *s, size_t maxlen)
{
	const guint32 *p = s;

	g_assert(s != NULL);
	g_assert(maxlen <= INT_MAX);

	while (maxlen-- > 0 && *p != 0x0000) {
		p++;
	}

	return p - s;
}

/**
 * Decomposes a UTF-32 encoded string.
 *
 */
size_t
utf32_decompose_nfd(const guint32 *in, guint32 *out, size_t size)
{
	const guint32 *d, *s = in;
	guint32 *p = out;
	guint32 uc;
	size_t d_len;

	g_assert(in != NULL);
	g_assert(out != NULL);
	g_assert(size <= INT_MAX);

	if (size-- > 0) {
		while ((uc = *s) != 0x0000) {
			d = utf32_decompose_nfd_char(uc, &d_len);
			if (d_len > size)
				break;
			while (d_len-- > 0) {
				*p++ = *d++;
			}
			s++;
		}
		*p = 0x0000;
	}
	
	while ((uc = *s) != 0x0000) {
		d = utf32_decompose_nfd_char(uc, &d_len);
		p += d_len;
	}

	return p - out;
}

/**
 * Decomposes a UTF-8 encoded string.
 *
 * The UTF-8 string written to ``out'' is always NUL-terminated unless
 * ``size'' is zero. If the size of ``out'' is too small to hold the
 * complete decomposed string, the resulting string will be truncated but
 * the validity of the UTF-8 encoding will be preserved. Truncation is
 * indicated by the return value being equal to or greater than ``size''.
 *
 * @param in a UTF-8 encoded string.
 * @param out a pointer to a buffer which will hold the decomposed string.
 * @param size the number of bytes ``out'' can hold.
 * 
 * @returns the length in bytes (not characters!) of completely decomposed
 *			string.
 */
size_t
utf8_decompose_nfd(const gchar *in, gchar *out, size_t size)
{
	const guint32 *d;
	const gchar *s = in;
	gchar *p = out;
	guint32 uc;
	size_t len = 0, d_len;
	guint retlen;
	gchar utf8_buf[7];

	g_assert(in != NULL);
	g_assert(out != NULL);
	g_assert(size <= INT_MAX);
	
	if (size-- > 0) {
		while (*s != '\0') {
			size_t utf8_len;
			gchar buf[256], *q;
			
			len = utf8_decode_lookahead(s, len);
			uc = utf8_decode_char(s, len, &retlen, TRUE);
			d = utf32_decompose_nfd_char(uc, &d_len);
			q = buf;
			while (d_len-- > 0) {
				uc = *d++;
				utf8_len = utf8_encode_char(uc, utf8_buf);
				g_assert((size_t) (&buf[sizeof buf] - q) >= utf8_len);
				memcpy(q, utf8_buf, utf8_len);
				q += utf8_len;
			}
			
			utf8_len = q - buf;
			if (utf8_len > size)
				break;

			memcpy(p, buf, utf8_len);
			p+= utf8_len;

			s += retlen;
			len -= retlen;
		}
		*p = '\0';
	}
	
	while (*s != '\0') {
		len = utf8_decode_lookahead(s, len);
		uc = utf8_decode_char(s, len, &retlen, TRUE);
		d = utf32_decompose_nfd_char(uc, &d_len);
		while (d_len-- > 0) {
			uc = *d++;
			p += utf8_encode_char(uc, utf8_buf);
		}
		
		s += retlen;
		len -= retlen;
	}

	return p - out;
}

#ifdef USE_ICU

/*
 * to_icu_conv
 *
 * Convert a string from the locale encoding to internal ICU encoding (UTF32)
 *
 */
int to_icu_conv(const gchar *in, int lenin, UChar *out, int lenout)
{
	UErrorCode error = U_ZERO_ERROR;
	int r;

	r = ucnv_toUChars(conv_icu_locale, out, lenout, in, lenin, &error);

	return (error != U_ZERO_ERROR && error != U_BUFFER_OVERFLOW_ERROR) ? 0 : r;
}

/*
 * to_icu_conv
 *
 * Convert a string from ICU encoding (UTF32) to UTF8 encoding (fast)
 *
 */
int icu_to_utf8_conv(const UChar *in, int lenin, gchar *out, int lenout)
{
	UErrorCode error = U_ZERO_ERROR;
	int r;

	r = ucnv_fromUChars(conv_icu_utf8, out, lenout, in, lenin, &error);

	return (error != U_ZERO_ERROR && error != U_BUFFER_OVERFLOW_ERROR &&
			error != U_STRING_NOT_TERMINATED_WARNING) ? 0 : r;
}

/*
 * unicode_NFC
 *
 * Compact a string as specified in unicode
 *
 */
int unicode_NFC(const UChar *source, gint32 len, UChar *result, gint32 rlen)
{
	UErrorCode error = U_ZERO_ERROR;
	int r;

	r = unorm_normalize(source, len, UNORM_NFC, 0, result, rlen, &error);

	return (error != U_ZERO_ERROR && error != U_BUFFER_OVERFLOW_ERROR) ? 0 : r;
}

/*
 * unicode_NFC
 *
 * Expand and K a string as specified in unicode
 * K will transform special character in the standard form
 * for instance : The large japanese space will be transform to a normal space
 *
 */
int unicode_NFKD(const UChar *source, gint32 len, UChar *result, gint32 rlen)
{
	UErrorCode error = U_ZERO_ERROR;
	int r;

	r = unorm_normalize (source, len, UNORM_NFKD, 0, result, rlen, &error);

	return (error != U_ZERO_ERROR && error != U_BUFFER_OVERFLOW_ERROR) ? 0 : r;
}

/*
 * unicode_upper
 *
 * Upper case a string
 * This is usefull to transorm the german sset to SS
 * Note : this will not transform hiragana to katakana
 *
 */
int unicode_upper(const UChar *source, gint32 len, UChar *result, gint32 rlen)
{
	UErrorCode error = U_ZERO_ERROR;
	int r;

	r = u_strToUpper(result, rlen, source, len, NULL, &error);
	
	return (error != U_ZERO_ERROR && error != U_BUFFER_OVERFLOW_ERROR) ? 0 : r;
}

/*
 * unicode_lower
 *
 * Lower case a string
 *
 */
int unicode_lower(const UChar *source, gint32 len, UChar *result, gint32 rlen)
{
	UErrorCode error = U_ZERO_ERROR;
	int r;

	r = u_strToLower(result, rlen, source, len, NULL, &error);
	
	return (error != U_ZERO_ERROR && error != U_BUFFER_OVERFLOW_ERROR) ? 0 : r;
}

/*
 * unicode_filters
 *
 * Remove all the non letter and non digit by looking the unicode symbol type
 * all other characters will be reduce to normal space
 * try to merge continues spaces in the same time
 * keep the important non spacing marks
 */
int unicode_filters(const UChar *source, gint32 len, UChar *result)
{
	int i, j;
	int space = 0;

	for (i = 0, j = 0; i < len; i++) {
		switch (u_charType(source[i])) {
		case U_LOWERCASE_LETTER :
		case U_OTHER_LETTER :
		case U_MODIFIER_LETTER :
		case U_DECIMAL_DIGIT_NUMBER :
			result[j++] = source[i];
			space = 0;
			break;

		case U_CONTROL_CHAR :
			if (source[i] == '\n')
				result[j++] = source[i];
			break;

		case U_NON_SPACING_MARK :
			/* Do not skip the japanese " and  kana marks and so on */

			switch (source[i]) {
				/* Japanese voiced sound marks */
			case 0x3099:
			case 0x309A:
				/* Virama signs */
			case 0x0BCD:
			case 0x094D:
			case 0x09CD:
			case 0x0A4D:
			case 0x0ACD:
			case 0x0B4D:
			case 0x0CCD:
			case 0x1039:
			case 0x1714:
			case 0x0C4D:
				/* Nukta signs */
			case 0x093C:
			case 0x09BC:
			case 0x0A3C:
			case 0x0ABC:
			case 0x0B3C:
			case 0x0CBC:
				/* Greek Ypogegrammeni */
			case 0x0345:
				/* Tibetan */
			case 0x0F71:
			case 0x0F72:
			case 0x0F7A:
			case 0x0F7B:
			case 0x0F7C:
			case 0x0F7D:
			case 0x0F80:
			case 0x0F74:
			case 0x0F39:
			case 0x0F18:
			case 0x0F19:
			case 0x0F35:
			case 0x0F37:
			case 0x0FC6:
			case 0x0F82:
			case 0x0F83:
			case 0x0F84:
			case 0x0F86:
			case 0x0F87:

		/* Others : not very sure we must keep them or not ... */

				/* Myanmar */
			case 0x1037:
				/* Sinhala */
			case 0x0DCA:
				/* Thai */
			case 0x0E3A:
				/* Hanundo */
			case 0x1734:
				/* Devanagari */
			case 0x0951:
			case 0x0952:
				/* Lao */
			case 0x0EB8:
			case 0x0EB9:
				/* Limbu */
			case 0x193B:
			case 0x1939:
			case 0x193A:
				/* Mongolian */
			case 0x18A9:
				result[j++] = source[i];
			}
			break;

		case U_UPPERCASE_LETTER :
		case U_TITLECASE_LETTER :
		case U_PARAGRAPH_SEPARATOR :
		case U_UNASSIGNED :
		case U_COMBINING_SPACING_MARK :
		case U_LINE_SEPARATOR :
		case U_LETTER_NUMBER :
		case U_OTHER_NUMBER :
		case U_SPACE_SEPARATOR :
		case U_FORMAT_CHAR :
		case U_PRIVATE_USE_CHAR :
		case U_SURROGATE :
		case U_DASH_PUNCTUATION :
		case U_START_PUNCTUATION :
		case U_END_PUNCTUATION :
		case U_CONNECTOR_PUNCTUATION :
		case U_OTHER_PUNCTUATION :
		case U_MATH_SYMBOL :
		case U_CURRENCY_SYMBOL :
		case U_MODIFIER_SYMBOL :
		case U_OTHER_SYMBOL :
		case U_INITIAL_PUNCTUATION :
		case U_FINAL_PUNCTUATION :
		case U_CHAR_CATEGORY_COUNT :
			if (0 == space)
				result[j++] = 0x0020;
			space = 1;
			break;
		}
	}
	return j;
}

/*
 * unicode_canonize
 *
 * Apply the NFKD/NFC algo to have nomalized keywords
 * The string `in' MUST be valid UTF-8 or that function would return rubbish.
 */
gchar *unicode_canonize(const gchar *in)
{
	UChar *qtmp1;
	UChar *qtmp2;
	int	len, maxlen;
	gchar *out;
	gboolean latin_locale = is_latin_locale();

	len = strlen(in);
	maxlen = len * 6; /* Max 6 bytes for one char in utf8 */

	g_assert(utf8_is_valid_string(in, len));

	qtmp1 = (UChar *) g_malloc(maxlen*sizeof(UChar));
	qtmp2 = (UChar *) g_malloc(maxlen*sizeof(UChar));

	if (latin_locale) {
		len = to_icu_conv(in, len, qtmp1, maxlen);
		len = unicode_NFC(qtmp1, len, qtmp2, maxlen);
	} else
		len = to_icu_conv(in, len, qtmp2, maxlen);

	len = unicode_NFKD(qtmp2, len, qtmp1, maxlen);
	len = unicode_upper(qtmp1, len, qtmp2, maxlen);
	len = unicode_lower(qtmp2, len, qtmp1, maxlen);
	len = unicode_filters(qtmp1, len, qtmp2);
	len = unicode_NFC(qtmp2, len, qtmp1, maxlen);

	out = g_malloc(len + 1);

	if (latin_locale) /* Should be pure ASCII now */
		len = icu_to_utf8_conv(qtmp2, len, out, len);
	else
		len = icu_to_utf8_conv(qtmp2, len, out, len * 6);

	out[len] = 0;

	g_free(qtmp1);
	g_free(qtmp2);

	return out;
}

/* vi: set ts=4 sw=4 cindent: */
#endif	/* USE_ICU */
