/* GTS - Library for the manipulation of triangulated surfaces
 * Copyright (C) 1999 Stphane Popinet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include "gts.h"

static void prepend_triangle_bbox (GtsTriangle * t, GSList ** bboxes)
{
  *bboxes = g_slist_prepend (*bboxes, 
			     gts_bbox_triangle (gts_bbox_class (), t));
}

static gboolean segment_is_ok (GtsSegment * s)
{
  g_return_val_if_fail (s != NULL, FALSE);
  g_return_val_if_fail (s->v1 != s->v2, FALSE);
  g_return_val_if_fail (GTS_OBJECT (s)->reserved == NULL, FALSE);
  return TRUE;
}

static gboolean vertex_is_ok (GtsVertex * v)
{
  g_return_val_if_fail (v != NULL, FALSE);
  g_return_val_if_fail (GTS_OBJECT (v)->reserved == NULL, FALSE);
  return TRUE;
}

static gboolean triangle_is_ok (GtsTriangle * t)
{
  g_return_val_if_fail (t != NULL, FALSE);
  g_return_val_if_fail (t->e1 != NULL, FALSE);
  g_return_val_if_fail (t->e2 != NULL, FALSE);
  g_return_val_if_fail (t->e3 != NULL, FALSE);
  g_return_val_if_fail (t->e1 != t->e2 && t->e1 != t->e3 && t->e2 != t->e3, 
			FALSE);
  g_return_val_if_fail (gts_segments_touch (GTS_SEGMENT (t->e1), 
					    GTS_SEGMENT (t->e2)),
			FALSE);
  g_return_val_if_fail (gts_segments_touch (GTS_SEGMENT (t->e1), 
					    GTS_SEGMENT (t->e3)), 
			FALSE);
  g_return_val_if_fail (gts_segments_touch (GTS_SEGMENT (t->e2), 
					    GTS_SEGMENT (t->e3)), 
			FALSE);
  g_return_val_if_fail (GTS_SEGMENT (t->e1)->v1 != GTS_SEGMENT (t->e1)->v2, 
			FALSE);
  g_return_val_if_fail (GTS_SEGMENT (t->e2)->v1 != GTS_SEGMENT (t->e2)->v2, 
			FALSE);
  g_return_val_if_fail (GTS_SEGMENT (t->e3)->v1 != GTS_SEGMENT (t->e3)->v2, 
			FALSE);
  g_return_val_if_fail (GTS_OBJECT (t)->reserved == NULL, FALSE);
  return TRUE;
}

int main (int argc, char * argv[])
{
  GtsSurface * s1, * s2;
  GtsSurfaceInter * si;
  GNode * tree1, * tree2;
  GSList * bboxes;
  FILE * fp;
  guint line;

  if (argc != 3) {
    fprintf (stderr, 
	     "%s: check set operations between surfaces\n"
	     "usage: %s FILE1 FILE2\n",
	     argv[0], argv[0]);
    return 1;
  }

  /* open first file */
  if ((fp = fopen (argv[1], "rt")) == NULL) {
    fprintf (stderr, "%s: can not open file `%s'\n", argv[0], argv[2]);
    return 1;
  }
  /* reads in first surface file */
  s1 = gts_surface_new (gts_surface_class (),
			GTS_FACE_CLASS (gts_nface_class ()),
			GTS_EDGE_CLASS (gts_nedge_class ()),
			GTS_VERTEX_CLASS (gts_nvertex_class ()));
  if ((line = gts_surface_read (s1, fp))) {
    fprintf (stderr, "%s: `%s' is not a valid GTS surface file\n", 
	     argv[0], argv[1]);
    fprintf (stderr, "Format error at line %d\n", line);
    return 1;
  }
  fclose (fp);

 /* open second file */
  if ((fp = fopen (argv[2], "rt")) == NULL) {
    fprintf (stderr, "%s: can not open file `%s'\n", argv[0], argv[3]);
    return 1;
  }
  /* reads in second surface file */
  s2 = gts_surface_new (gts_surface_class (),
			GTS_FACE_CLASS (gts_nface_class ()),
			GTS_EDGE_CLASS (gts_nedge_class ()),
			GTS_VERTEX_CLASS (gts_nvertex_class ()));
  if ((line = gts_surface_read (s2, fp))) {
    fprintf (stderr, "%s: `%s' is not a valid GTS surface file\n", 
	     argv[0], argv[2]);
    fprintf (stderr, "Format error at line %d\n", line);
    return 1;
  }
  fclose (fp);

  /* display summary information about both surfaces */
#if 0
  gts_surface_print_stats (s1, stderr);
  gts_surface_print_stats (s2, stderr);
#endif

  /* build bounding boxes for first surface */
  bboxes = NULL;
  gts_surface_foreach_face (s1, (GtsFunc) prepend_triangle_bbox, &bboxes);
  /* build bounding box tree for first surface */
  tree1 = gts_bb_tree_new (bboxes);
  /* free list of bboxes */
  g_slist_free (bboxes);

  /* build bounding boxes for second surface */
  bboxes = NULL;
  gts_surface_foreach_face (s2, (GtsFunc) prepend_triangle_bbox, &bboxes);
  /* build bounding box tree for second surface */
  tree2 = gts_bb_tree_new (bboxes);
  /* free list of bboxes */
  g_slist_free (bboxes);

  {
    GSList * i;
    GtsSurface * s = gts_surface_new (gts_surface_class (),
				      gts_face_class (),
				      gts_edge_class (),
				      gts_vertex_class ());

    si = gts_surface_inter_new (gts_surface_inter_class (), 
				s1, s2, tree1, tree2);

    gts_surface_merge (s, si->s1);
    gts_surface_merge (s, si->s2);
    gts_surface_print_stats (s, stderr);
    gts_object_destroy (GTS_OBJECT (s));

    printf ("(geometry \"inter\" { = LIST {\n");
    i = si->edges;
    while (i) {
      GtsSegment * s = i->data;
      printf ("# %s->%s\n"
	      "VECT 1 2 0 2 0 %g %g %g %g %g %g\n",
	      GTS_NVERTEX (s->v1)->name,
	      GTS_NVERTEX (s->v2)->name,
	      GTS_POINT (s->v1)->x, 
	      GTS_POINT (s->v1)->y, 
	      GTS_POINT (s->v1)->z, 
	      GTS_POINT (s->v2)->x, 
	      GTS_POINT (s->v2)->y, 
	      GTS_POINT (s->v2)->z);
      i = i->next;
    }
    printf ("}})\n"
	    "(normalization \"inter\" none)\n");
    printf ("(geometry \"s1\" { =\n");
    gts_surface_write_oogl (si->s1, stdout);
    printf ("})\n"
	    "(normalization \"s1\" none)\n");
    printf ("(geometry \"s2\" { =\n");
    gts_surface_write_oogl (si->s2, stdout);
    printf ("})\n"
	    "(normalization \"s2\" none)\n");

    gts_surface_foreach_face (s1, (GtsFunc) triangle_is_ok, NULL);
    gts_surface_foreach_edge (s1, (GtsFunc) segment_is_ok, NULL);
    gts_surface_foreach_edge (s1, (GtsFunc) vertex_is_ok, NULL);

    gts_surface_foreach_face (s2, (GtsFunc) triangle_is_ok, NULL);
    gts_surface_foreach_edge (s2, (GtsFunc) segment_is_ok, NULL);
    gts_surface_foreach_edge (s2, (GtsFunc) vertex_is_ok, NULL);

    gts_surface_foreach_face (si->s1, (GtsFunc) triangle_is_ok, NULL);
    gts_surface_foreach_edge (si->s1, (GtsFunc) segment_is_ok, NULL);
    gts_surface_foreach_edge (si->s1, (GtsFunc) vertex_is_ok, NULL);

    gts_surface_foreach_face (si->s2, (GtsFunc) triangle_is_ok, NULL);
    gts_surface_foreach_edge (si->s2, (GtsFunc) segment_is_ok, NULL);
    gts_surface_foreach_edge (si->s2, (GtsFunc) vertex_is_ok, NULL);

    g_assert (gts_surface_inter_check (si));
  }

  /* destroy surfaces and intersection */
  gts_object_destroy (GTS_OBJECT (s1));
  gts_object_destroy (GTS_OBJECT (s2));
  gts_object_destroy (GTS_OBJECT (si));

  /* destroy bounding box trees (including bounding boxes) */
  gts_bb_tree_destroy (tree1, TRUE);
  gts_bb_tree_destroy (tree2, TRUE);  

  return 0;
}
