/* preferences.c - Preferences handling for gupsc

   Copyright (C) 2000  Henning Kulander <hennikul@ifi.uio.no>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <pthread.h>

#include "support.h"
#include "interface.h"
#include "ups_fetch.h"
#include "dial.h"
#include "ups_dials.h"
#include "ups_canvas.h"
#include "preferences.h"
#define _MIT_POSIX_THREADS 1
#include <pthread.h>

static  pthread_t thread;
static  pthread_attr_t  threadattr;
pthread_mutex_t sync_mutex=PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t ups_mutex=PTHREAD_MUTEX_INITIALIZER;
int res=-1;

typedef struct _InfoType InfoType;

struct _InfoType {
	gchar *name;
	gchar *description;
};

InfoType *info_types[NINFO_TYPES];
int info_types_index = 0;

InfoDial info_dials[NDIALS];

gchar *fontname_dials;
gchar *hostname=NULL;
gint delay;
guint timeout=0;
gboolean udp; 
GtkWidget *window;

gchar *session;

/* Clear all the dials when they are first made */
void 
preferences_infodials_clear (void)
{
	gint i;
	
	for (i = 0; i < NDIALS ; i++) {
		info_dials[i].infotype = NULL;
		info_dials[i].dial = NULL;
	}
}

/* Set an infodial to display a certain type of information */
static void 
preferences_infodial_set (gint num, gchar *infotype)
{
	gchar *last_info;
	
	last_info = info_dials[num].infotype;
	info_dials[num].infotype = g_strdup (infotype);
	if (last_info != NULL)
		if (!strcmp (last_info, infotype))
			info_dials[num].changed = FALSE;
		else
			info_dials[num].changed = TRUE;
	else
		info_dials[num].changed = TRUE;
	g_free (last_info);
}

/* Get a stringvariable from current session's config */
gchar *
preferences_get_string_variable (gchar *variable)
{
	gchar *var;
	gchar *content;
	
	var = g_strdup_printf (variable, session);
	content = gnome_config_get_string (var);
	g_free (var);
	
	return content;
}

/* Set a stringvariable in current session's config */
void 
preferences_set_string_variable (gchar *variable, gchar *value)
{
	gchar *var;
	
	var = g_strdup_printf (variable, session);
	gnome_config_set_string (var, value);
	
	g_free (var);
}

/* Get an intvariable from current session's config */
static gint 
preferences_get_int_variable (gchar *variable)
{
	gchar *var;
	gint content;
	
	var = g_strdup_printf (variable, session);
	content = gnome_config_get_int (var);
	g_free (var);
	
	return content;
}

/* Set an intvariable in current session's config */
void 
preferences_set_int_variable (gchar *variable, gint value)
{
	gchar *var;
	
	var = g_strdup_printf (variable, session);
	gnome_config_set_int (var, value);
	
	g_free(var);
}

/* Get a boolean variable from current session's config */
gboolean 
preferences_get_bool_variable (gchar *variable)
{
	gchar *var;
	gboolean content;
	
	var = g_strdup_printf (variable, session);
	content = gnome_config_get_bool (var);
	g_free (var);
	
	return content;  
}

/* Set a boolean variable in current session's config */
void 
preferences_set_bool_variable (gchar *variable, gboolean value)
{
	gchar *var;
	
	var = g_strdup_printf (variable, session);
	gnome_config_set_bool (var, value);
	
	g_free (var);
}

/* Load preferences from gnome-config database */
gboolean 
preferences_load (GtkWidget *widget)
{
	gint i;
	gint col, row;
	GtkWidget *canvas;
	gchar *oldfontname_dials;
	gchar *oldhostname;
	gboolean oldconnecttype;
	gboolean connected=FALSE;
	gchar *tmpstr;
  
	/* Get the variables from config-database */
	oldhostname = hostname;
	oldconnecttype = udp;
	hostname = preferences_get_string_variable ("/gupsc/%s/hostname="\
						    "localhost");
	udp = preferences_get_bool_variable ("/gupsc/%s/udp=TRUE");

	delay = preferences_get_int_variable ("/gupsc/%s/delay=1000.0");

	tmpstr = preferences_get_string_variable ("/gupsc/%s/dial1=ACFREQ");
	preferences_infodial_set (0, tmpstr);
	g_free (tmpstr);
	tmpstr = preferences_get_string_variable ("/gupsc/%s/dial2=UTILITY");
	preferences_infodial_set (1, tmpstr);
	g_free (tmpstr);
	tmpstr = preferences_get_string_variable ("/gupsc/%s/dial3=LOADPCT");
	preferences_infodial_set (2, tmpstr);
	g_free (tmpstr);
	tmpstr = preferences_get_string_variable("/gupsc/%s/dial4=NONE");
	preferences_infodial_set (3, tmpstr);
	g_free (tmpstr);

	lock_ups_mutex ();

	/* Check if hostname has changed */
	if (oldhostname != NULL) {
		if (strcmp (oldhostname,hostname) || oldconnecttype != udp) { 
			for (i = 0 ; i < NDIALS ; i++)
				info_dials[i].changed = TRUE;
			if (udp) {
				connected = connect_ups_udp (hostname);
			} else {
				connected = connect_ups_tcp (hostname);
			}
		}
		g_free (oldhostname);
	} else if (udp) {
		connected = connect_ups_udp (hostname);
	} else	{
		connected = connect_ups_tcp (hostname);
	}

	/* Check if font has changed */
	oldfontname_dials = fontname_dials;
	fontname_dials = preferences_get_string_variable ("/gupsc/%s/dialfont=-adobe-helvetica-medium-r-normal-*-*-100-*-*-p-*-iso8859-1");
	if (oldfontname_dials != NULL && fontname_dials != NULL) {
		ups_battery_set_font (fontname_dials);
		if (strcmp (oldfontname_dials, fontname_dials))
			for (i = 0; i < NDIALS; i++)
				info_dials[i].changed = TRUE;
		g_free (oldfontname_dials);
	}

	set_current_limits ();
	/* Remove dials and make new ones */
	for (i = 0; i < NDIALS ; i++) {
		if (info_dials[i].changed == TRUE) {
			/* Remove dial and canvas if it exists... */
			if (info_dials[i].canvas) {
				gtk_container_remove ( 
					GTK_CONTAINER ( 
						lookup_widget (widget, 
							       "dial_table")),
					info_dials[i].canvas);
				g_free (info_dials[i].dial);
				info_dials[i].dial = NULL;
				gtk_widget_destroy ( 
					GTK_WIDGET (info_dials[i].canvas));
				info_dials[i].canvas = NULL;
			}
			 
			/* Make new canvas and dial. */
			if (info_types_has (info_dials[i].infotype)) {
				/* find position in table */
				/* 0 and 2 goes left,1 and 3 right: */
				col = ( (i%2) == 0 ? 0 : 1 ); 
                                /* 0 and 1 goes top, 2 and 3 bottom: */
				row = ( i < 2      ? 0 : 1 ); 
				  
				/* Make canvas */
				gtk_widget_push_visual (
					gdk_imlib_get_visual ());
				gtk_widget_push_colormap (
					gdk_imlib_get_colormap ());
				canvas = gnome_canvas_new ();
				gtk_widget_pop_colormap ();
				gtk_widget_pop_visual ();
				gtk_widget_ref (canvas);
				gtk_widget_show (canvas);
				gtk_table_attach( 
					GTK_TABLE ( lookup_widget( 
						widget, "dial_table")),
					canvas, col, col+1, row, row+1,
					(GtkAttachOptions) ( 0 ),
					(GtkAttachOptions) (GTK_FILL), 0, 0);
				gtk_widget_set_usize (canvas, 130, 130);
				gnome_canvas_set_scroll_region (
					GNOME_CANVAS (canvas), 0, 0, 130, 130);
				info_dials[i].canvas = canvas;
				
				/* Make new dial */
				info_dials[i].dial = 
					ups_make_dial (info_dials[i].infotype
						       , canvas);
				}
			info_dials[i].changed = FALSE;
		}
	}
	
	unlock_ups_mutex ();
 
	if (connected)
		start_fetching ();

	display_ups_data (global_window_get ());

	return TRUE;
}

/* Shows a optionmenu with infotypes avaliable for the UPS */
static void 
preferences_box_showmenu (GtkWidget *preferences, gchar *menuname, 
			  InfoDial *info_dial)
{
	GtkWidget *menu, *menuitem;
	gint selected = 0;
	gint i;

	/* Make new menu */
	menu = gtk_menu_new();
	menuitem = gtk_menu_item_new_with_label (_("None"));
	gtk_widget_show (menuitem);
	gtk_menu_append (GTK_MENU (menu), menuitem);
	for (i = 0; i < info_types_index; i++) {
		menuitem = gtk_menu_item_new_with_label ( 
			info_types[i]->description);
		if (!strcmp (info_dial->infotype, info_types[i]->name))
			selected = i+1; /* Find item to be autoselected */
		gtk_widget_show (menuitem);		  
		info_dial->menu_items[i] = menuitem;
		gtk_menu_append (GTK_MENU (menu), menuitem);		
	}
	info_dial->n_items = i;

	/* Select item found in configuration, or none if not avaliable
	 * in UPS */
	gtk_menu_set_active (GTK_MENU (menu), selected);

	gtk_option_menu_remove_menu /* Remove old menu */
		(GTK_OPTION_MENU (lookup_widget (GTK_WIDGET (preferences),
						 menuname)));
	gtk_option_menu_set_menu /* Make a new one */
		(GTK_OPTION_MENU (lookup_widget (GTK_WIDGET (preferences),
						 menuname)), menu);
}

/* Show all the option menus */
void 
preferences_box_showmenus (GtkWidget *preferences)
{
	preferences_box_showmenu (preferences, "itemmenu1", &info_dials[0]);
	preferences_box_showmenu (preferences, "itemmenu2", &info_dials[1]);
	preferences_box_showmenu (preferences, "itemmenu3", &info_dials[2]);
	preferences_box_showmenu (preferences, "itemmenu4", &info_dials[3]);  
}

/* Show the preferences box with apropriate values */
gboolean 
preferences_box_make (GtkWidget *main_widget)
{
	GtkWidget *preferences;

	preferences_load (main_widget);
	preferences = create_propertybox ();
  
	gtk_entry_set_text
		(GTK_ENTRY (lookup_widget (GTK_WIDGET (preferences), 
					   "combo_entry_hostname")),
		 hostname);
	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (lookup_widget (GTK_WIDGET (preferences),
						   "radio_udp")), 
		 udp);
	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (lookup_widget (GTK_WIDGET (preferences),
						   "radio_tcp")), 
		 !udp);
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (lookup_widget (GTK_WIDGET (preferences),
						 "spinbutton_delay")),
		 (gfloat) delay);

	preferences_box_showmenus (preferences);
	
	gnome_font_picker_set_font_name
		(GNOME_FONT_PICKER (lookup_widget (GTK_WIDGET (preferences),
						   "fontpicker_dials")),
		 fontname_dials);

	gnome_property_box_set_state (GNOME_PROPERTY_BOX (preferences), 
				      FALSE);
	
	gtk_widget_show (preferences);

	return TRUE;
}

/* Get the hostname where the UPS is connected */
gchar *
preferences_hostname_get (void)
{
	return g_strdup (hostname);
}

/* Set wether or not to use udp transfer.. */
gboolean 
preferences_udp_set (gboolean val)
{
	udp = val;
	gnome_config_set_bool ("/gupsc/basic/udp", udp);

	return TRUE;
}

/* Check if we are using UDP */
gboolean 
preferences_udp_get (void)
{
	return udp;
}

/* Get the sampling delay */
gint 
preferences_delay_get (void)
{
	return delay;
}

/* Get the fontname used in the dials */
gchar *
preferences_get_fontname_dials (void)
{
	return fontname_dials;
}

/* Clear the info-types array */
void 
info_types_clear (void)
{
	int i;

	for (i = 0; i < 10; i++) {
		if (info_types[i]) {
			if (info_types[i]->description)
				g_free (info_types[i]->description);
			g_free (info_types[i]);
			info_types[i] = NULL;
		}
	}
	info_types_index = 0;
}

/* Add an info-type with name and description */
void 
info_types_add (gchar *name, gchar *description)
{
	info_types[info_types_index] = g_new (InfoType, 1);
	info_types[info_types_index]->name = name;
	info_types[info_types_index]->description = description;
	info_types_index++;
}

/* Get name for infotype with given number */
gchar *
info_types_get_name (gint number)
{
	return info_types[number]->name;
}

/* Get infodial at index */
InfoDial *
info_dial_get (gint index)
{
	return &info_dials[index];
}

/* Is "name" a valid info-type on this UPS? */
gboolean 
info_types_has (gchar *name)
{
	gint i;

	for (i = 0 ; i < info_types_index ; i++)
		if (!strcmp (name, info_types[i]->name))
			return TRUE;

	return FALSE;
}

/* Set the timeout identification */
void 
global_timeout_set (guint id)
{
	timeout = id;
}

/* Get the identification of the timeout */
guint 
global_timeout_get (void)
{
	return timeout;
}

/* fetch variables in here, then put them somewhere nice */
static void *
fetch_thread (void *ptr)
{
	while (42) {
		lock_sync_mutex ();
		if (!fetch_ups_data ())
			printf ("Failed to read data from UPS...\n");
	}  
}

/* Function called by the timeout, updates dials. */
gboolean 
global_updatetimeout_func (gpointer data)
{
	gboolean ret;
	
	ret = display_ups_data (window);
	unlock_sync_mutex ();

	return ret;  
}

/* Set the global pointer to the window */
void 
global_window_set (GtkWidget *w)
{
	window = w;
}

/* Get the global pointer to the window */
GtkWidget *
global_window_get (void)
{
	return window;
}

/* Set the global session-name */
void 
global_session_set (gchar *ups_session)
{
	session = ups_session;
}

void 
unlock_sync_mutex (void)
{
	pthread_mutex_unlock (&sync_mutex);
}

void 
lock_sync_mutex (void)
{
	pthread_mutex_lock (&sync_mutex);
}

void 
unlock_ups_mutex (void)
{
	pthread_mutex_unlock (&ups_mutex);
}

void 
lock_ups_mutex (void)
{
	pthread_mutex_lock (&ups_mutex);
}

/* Make a new sampling thread */
gboolean 
make_thread (void)
{
	lock_sync_mutex ();
  
	pthread_attr_init (&threadattr);
	pthread_attr_setscope (&threadattr, PTHREAD_SCOPE_SYSTEM);
	pthread_attr_setdetachstate (&threadattr, PTHREAD_CREATE_DETACHED);
	
	res = pthread_create (&thread, &threadattr, fetch_thread, NULL);

	return TRUE;
}

/* Start the sampling */
guint 
start_fetching (void)
{
	guint timeout;

	unlock_sync_mutex ();

	if (global_timeout_get () != 0 )
		g_source_remove (global_timeout_get ());
	timeout = g_timeout_add_full (G_PRIORITY_HIGH_IDLE, 
				      preferences_delay_get (), 
				      global_updatetimeout_func,
				      NULL,
				      NULL);
	global_timeout_set (timeout);
	
	return timeout;
}
