/*  -*-objc-*-
 *  BrowserViewer.m: Implementation of the BrowserViewer Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "BrowserViewer.h"
#include "GWBrowser.h"
#include "GWBrowserCell.h"
#include "ColumnIcon.h"
#include "Functions.h"
#include "GNUstep.h"

/* Notifications */
#ifndef GNUSTEP
	NSString *GWFileSystemWillChangeNotification = @"GWFileSystemWillChangeNotification";
	NSString *GWFileSystemDidChangeNotification = @"GWFileSystemDidChangeNotification";
	NSString *GWDidSetFileAttributesNotification = @"GWDidSetFileAttributesNotification";
	NSString *GWSortTypeDidChangeNotification = @"GWSortTypeDidChangeNotification";
#else
	extern NSString *GWFileSystemWillChangeNotification;
	extern NSString *GWFileSystemDidChangeNotification;
	extern NSString *GWDidSetFileAttributesNotification;
	extern NSString *GWSortTypeDidChangeNotification;
#endif

/* File Operations */
#ifndef GNUSTEP
	NSString *GWorkspaceRenameOperation = @"GWorkspaceRenameOperation";
	NSString *GWorkspaceCreateFileOperation = @"GWorkspaceCreateFileOperation";
	NSString *GWorkspaceCreateDirOperation = @"GWorkspaceCreateDirOperation";
	NSString *GWorkspaceRecycleOutOperation = @"GWorkspaceRecycleOutOperation";
	NSString *GWorkspaceEmptyRecyclerOperation = @"GWorkspaceEmptyRecyclerOperation";
#else
	extern NSString *GWorkspaceRenameOperation;
	extern NSString *GWorkspaceCreateFileOperation;
	extern NSString *GWorkspaceCreateDirOperation;
	extern NSString *GWorkspaceRecycleOutOperation;
	extern NSString *GWorkspaceEmptyRecyclerOperation;
#endif

/* File Watcher Strings */
#ifndef GNUSTEP
	NSString *GWFileWatcherFileDidChangeNotification = @"GWFileWatcherFileDidChangeNotification";
	NSString *GWWatchedDirectoryDeleted = @"GWWatchedDirectoryDeleted";
	NSString *GWFileDeletedInWatchedDirectory = @"GWFileDeletedInWatchedDirectory";
	NSString *GWFileCreatedInWatchedDirectory = @"GWFileCreatedInWatchedDirectory";
#else
	extern NSString *GWFileWatcherFileDidChangeNotification;
	extern NSString *GWWatchedDirectoryDeleted;
	extern NSString *GWFileDeletedInWatchedDirectory;
	extern NSString *GWFileCreatedInWatchedDirectory;
#endif

NSFileManager *fmanager = nil;

@implementation BrowserViewer

- (void)dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  TEST_RELEASE (rootPath);
  TEST_RELEASE (selectedPaths);
  TEST_RELEASE (watchedPaths);
  TEST_RELEASE (browser);
  [super dealloc];
}

- (id)init
{
	self = [super initWithFrame: NSZeroRect];
	browser = nil;
	rootPath = nil;
	selectedPaths = nil;
	watchedPaths = nil;
	fmanager = [NSFileManager defaultManager];
	return self;
}

- (void)setPath:(NSString *)path 
       delegate:(id)adelegate
       viewApps:(BOOL)canview
{
	int colswidth, winwidth;

  fm = [NSFileManager defaultManager];

	[self setDelegate: adelegate];
  ASSIGN (rootPath, path);
	TEST_RELEASE (selectedPaths);
	selectedPaths = [[NSArray alloc] initWithObjects: rootPath, nil]; 
  viewsapps = canview;
  autoSynchronize = YES; 

	colswidth = [delegate getBrowserColumnsWidth];
	resizeIncrement = colswidth;			
	winwidth = [delegate getWindowFrameWidth];			
  columns = (int)winwidth / resizeIncrement;      
  columnsWidth = (winwidth - 16) / columns;

  [self setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];

	if (browser != nil) {
		[browser removeFromSuperview];
		RELEASE (browser);
	}

	browser = [[GWBrowser alloc] initWithFrame: NSZeroRect basePath: rootPath]; 
	[browser setDelegate: self]; 
  [browser setAutoresizingMask: NSViewWidthSizable | NSViewHeightSizable];    
  [browser setMinColumnWidth: columnsWidth];    
	[browser setMaxVisibleColumns: columns];
  [self addSubview: browser]; 

	[[NSNotificationCenter defaultCenter] removeObserver: self];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(fileSystemDidChange:) 
                					  name: GWFileSystemDidChangeNotification
                					object: nil];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(sortTypeDidChange:) 
                					  name: GWSortTypeDidChangeNotification
                					object: nil];

	if (watchedPaths != nil) {
		[self unsetWatchers];
		RELEASE (watchedPaths);
		watchedPaths = nil;
	}

	[self setCurrentSelection: [NSArray arrayWithObject: rootPath]];
}

- (NSString *)menuName
{
	return @"Browser";
}

- (NSString *)shortCut
{
	return @"b";
}

- (BOOL)usesShelf
{
	return YES;
}

- (NSImage *)preview
{
	NSBundle *bundle = [NSBundle bundleForClass: [self class]];
	NSString *imgpath = [bundle pathForResource: @"preview" ofType: @"tiff"];
	NSImage *img = [[NSImage alloc] initWithContentsOfFile: imgpath];
	
	return AUTORELEASE (img);
}

- (NSImage *)miniicon
{
	NSBundle *bundle = [NSBundle bundleForClass: [self class]];
	NSString *imgpath = [bundle pathForResource: @"miniwindow" ofType: @"tiff"];
	NSImage *img = [[NSImage alloc] initWithContentsOfFile: imgpath];	
	return AUTORELEASE (img);
}

- (void)renewAll
{
	NSArray *spats = RETAIN (selectedPaths);
	[self setPath: rootPath delegate: delegate viewApps: viewsapps];
	[self setCurrentSelection: [NSArray arrayWithObject: rootPath]];
	[self resizeWithOldSuperviewSize: [self frame].size];
	[self setCurrentSelection: spats];
	[self resizeWithOldSuperviewSize: [self frame].size];
	RELEASE (spats);
}

- (void)validateRootPathAfterOperation:(NSDictionary *)opdict
{
  if ([rootPath isEqualToString: @"/"] == YES) {
    return;

  } else {
    NSString *operation = [opdict objectForKey: @"operation"];
    NSString *source = [opdict objectForKey: @"source"];
    NSArray *files = [opdict objectForKey: @"files"];
    int i;
    
    if ([operation isEqualToString: NSWorkspaceMoveOperation] 
        || [operation isEqualToString: NSWorkspaceDestroyOperation]
        || [operation isEqualToString: GWorkspaceRenameOperation]
				|| [operation isEqualToString: NSWorkspaceRecycleOperation]
				|| [operation isEqualToString: GWorkspaceRecycleOutOperation]
				|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) { 

      if ([operation isEqualToString: GWorkspaceRenameOperation]) {      
        files = [NSArray arrayWithObject: [source lastPathComponent]]; 
        source = [source stringByDeletingLastPathComponent];            
      } 
      
      for (i = 0; i < [files count]; i++) {
        NSString *fpath = [source stringByAppendingPathComponent: [files objectAtIndex: i]];

        if (subPathOfPath(fpath, rootPath)
                        			|| [fpath isEqualToString: rootPath]) {  
          [self closeNicely];      
          break;
        }
      }
      
    }
  }  
}

- (void)fileSystemDidChange:(NSNotification *)notification
{
  NSDictionary *dict;
  NSString *operation, *source, *destination;
  NSArray *files;
	NSArray *selpaths;
	NSMutableArray *selnames;
  int i, column, count; 
    
  dict = (NSDictionary *)[notification object];  
  operation = [dict objectForKey: @"operation"];
  source = [dict objectForKey: @"source"];
  destination = [dict objectForKey: @"destination"];
  files = [dict objectForKey: @"files"];

  if ([operation isEqualToString: NSWorkspaceMoveOperation]   
     || [operation isEqualToString: NSWorkspaceCopyOperation] 
        || [operation isEqualToString: NSWorkspaceLinkOperation]
           || [operation isEqualToString: NSWorkspaceDuplicateOperation]
						 || [operation isEqualToString: NSWorkspaceRecycleOperation]
							 || [operation isEqualToString: GWorkspaceRecycleOutOperation]) { 
                          
    column = [browser columnWithPath: destination]; 
		
    if (column != -1) {
			selpaths = [browser selectionInColumn: column];
			
			[browser removeCellsWithNames: files inColumn: column]; 			
			[browser addCellsWithNames: files inColumn: column dimmed: NO]; 			

			count = 0;
			if (selpaths != nil) {
				count = [selpaths count];
			}
		
			if (count > 0) {
				selnames = [NSMutableArray arrayWithCapacity: count];
				
				for (i = 0; i < count; i++) {
    			[selnames addObject: [[selpaths objectAtIndex: i] lastPathComponent]];
  			}

				[browser setMultipleSelection: selnames
                    				 inColumn: column 
													 sendAction: NO];
				
				[browser updateIconOfColumn: column + 1];
			}
    }
  }
  
  if ([operation isEqualToString: NSWorkspaceMoveOperation] 
        || [operation isEqualToString: NSWorkspaceDestroyOperation]
				|| [operation isEqualToString: NSWorkspaceRecycleOperation]
				|| [operation isEqualToString: GWorkspaceRecycleOutOperation]
				|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) {
               
    column = [browser columnWithPath: source];
    if (column != -1) {		
			selpaths = [browser selectionInColumn: column];
			
			[browser removeCellsWithNames: files inColumn: column]; 	
			
			count = 0;
			if (selpaths != nil) {
				count = [selpaths count];
			}		
		
			if (count > 0) {
				selnames = [NSMutableArray arrayWithCapacity: count];
				
				for (i = 0; i < count; i++) {
    			[selnames addObject: [[selpaths objectAtIndex: i] lastPathComponent]];
  			}

				for (i = 0; i < count; i++) {
					NSString *fname = [selnames objectAtIndex: i];

					if ([files containsObject: fname] == YES) {
  					[selnames removeObjectAtIndex: i];
  					count--;
  					i--;
					}
				}

				if ([selnames count] > 0) {
					[browser setMultipleSelection: selnames
                    					 inColumn: column 
														 sendAction: NO];
					[browser updateIconOfColumn: column + 1];
								
				} else {
  				if (column != 0) {
    				[browser selectCellWithTitle: [source lastPathComponent] 
														  	inColumn: column - 1 
													  	sendAction: YES];
  				} else {
    				[browser setLastColumn: column];
  				}
				
					[browser setLeafIconAtColumn: column];
				}
			}		
    }
  }
	
  if ([operation isEqualToString: GWorkspaceRenameOperation]) { 
    NSString *oldname = [source lastPathComponent];
    NSString *newname = [[destination lastPathComponent] copy]; 
    destination = [destination stringByDeletingLastPathComponent];                      
    column = [browser columnWithPath: destination]; 
		
    if (column != -1) {
			selpaths = [browser selectionInColumn: column];
			
			[browser removeCellsWithNames: [NSArray arrayWithObject: oldname] 
													 inColumn: column];

			[browser addCellsWithNames: [NSArray arrayWithObject: newname] 
										  	inColumn: column 
													dimmed: NO];
													
			count = 0;
			if (selpaths != nil) {
				count = [selpaths count];
			}
		
			if (count > 0) {
				selnames = [NSMutableArray arrayWithCapacity: 1];
				
				for (i = 0; i < count; i++) {
					NSString *name = [[selpaths objectAtIndex: i] lastPathComponent];
				
					if ([name isEqualToString: oldname]) {
						[selnames addObject: newname];
					} else {
    				[selnames addObject: name];
					}
  			}

				[browser setMultipleSelection: selnames
                    				 inColumn: column 
													 sendAction: NO];
													 
				[browser updateIconOfColumn: column + 1]; 
				
      	if ([[[browser selectedCell] stringValue] isEqualToString: newname]) {    
        	[browser selectCellWithTitle: newname inColumn: column sendAction: YES];    
      	}
			}
    }
		RELEASE (newname);
  }
    
  if ([operation isEqualToString: GWorkspaceCreateFileOperation] 
              || [operation isEqualToString: GWorkspaceCreateDirOperation]) {  
		NSString *fileName = [files objectAtIndex: 0];    
    column = [browser columnWithPath: destination];

    if (column != -1) {
			selpaths = [browser selectionInColumn: column];

			[browser removeCellsWithNames: files inColumn: column]; 			
			[browser addCellsWithNames: files inColumn: column dimmed: NO]; 			
		
			count = 0;
			if (selpaths != nil) {
				count = [selpaths count];
			}
		
			if (column != [browser lastValidColumn]) {
				if (count > 0) {
					selnames = [NSMutableArray arrayWithCapacity: count];

					for (i = 0; i < count; i++) {
    				[selnames addObject: [[selpaths objectAtIndex: i] lastPathComponent]];
  				}

					[browser setMultipleSelection: selnames
                    					 inColumn: column 
														 sendAction: NO];
				}
				
			} else if ([[self window] isKeyWindow] == YES) {
      	[browser selectCellWithTitle: fileName inColumn: column sendAction: YES]; 
      	[[browser lastIcon] selectToEdit];	
			}
    }
  }
	
  [self validateRootPathAfterOperation: dict];
	
	[delegate updateTheInfoString];
}
   
- (void)sortTypeDidChange:(NSNotification *)notification
{  
	NSString *notifPath = [notification object];
  
	if (notifPath != nil) {
  	int column = [browser columnWithPath: notifPath];
	
  	if (column != -1) {
    	id matrix = [browser matrixInColumn: column];
    	[browser emptyMatrix: matrix inColumn: column];
    	[self createCellsInMatrix: matrix ofColumn: column isNew: NO];
    	[matrix sizeToCells];
    	[matrix setNeedsDisplay: YES];
  	}
	} else {	
		[self renewAll];
	}
}

- (void)watcherNotification:(NSNotification *)notification
{
  NSDictionary *notifdict = (NSDictionary *)[notification object];
  NSString *path = [notifdict objectForKey: @"path"];
  
  if ([watchedPaths containsObject: path] == NO) {
    return;    

  } else {
    NSString *event = [notifdict objectForKey: @"event"];
    int i;
				
    if (event == GWWatchedDirectoryDeleted) {
      if ((subPathOfPath(path, rootPath) == YES)
                            || ([path isEqualToString: rootPath] == YES)) {  
        [self closeNicely];      
        return;
      } else {
        NSString *s = [path stringByDeletingLastPathComponent];
        [self unsetWatcherForPath: path]; 
        [self setCurrentSelection: [NSArray arrayWithObject: s]];
        return;
      }
    }

    if (event == GWFileDeletedInWatchedDirectory) {
      NSArray *files;
      int column;
		
      if (subPathOfPath(path, rootPath) == YES) {
        return; 
      } 
             
      files = [notifdict objectForKey: @"files"];
      column = [browser columnWithPath: path];

      if (column != -1) {
				NSArray *selpaths;
				NSMutableArray *selnames;
				int count;
				
				selpaths = [browser selectionInColumn: column];

				[browser removeCellsWithNames: files inColumn: column]; 			

				count = 0;
				if (selpaths != nil) {
					count = [selpaths count];
				}		
			
				if (count > 0) {
					selnames = [NSMutableArray arrayWithCapacity: count];

					for (i = 0; i < count; i++) {
    				[selnames addObject: [[selpaths objectAtIndex: i] lastPathComponent]];
  				}
	
					for (i = 0; i < count; i++) {
						NSString *fname = [selnames objectAtIndex: i];

						if ([files containsObject: fname] == YES) {
  						[selnames removeObjectAtIndex: i];
  						count--;
  						i--;
						}
					}
	
					if ([selnames count] > 0) {
						[browser setMultipleSelection: selnames
                    						 inColumn: column 
															 sendAction: NO];
						[browser updateIconOfColumn: column + 1];

					} else {
  					if (column != 0) {		
							if ((column - 1) >= [browser firstVisibleColumn]) {
	   						[browser selectCellWithTitle: [path lastPathComponent]
														  			inColumn: column - 1 
													  			sendAction: YES];
							}
  					} else {
    					[browser setLastColumn: column];
  					}

						[browser setLeafIconAtColumn: column];
					}		
				}
      }    

      return;
    }
       
    if (event == GWFileCreatedInWatchedDirectory) {
    
      if (subPathOfPath(path, rootPath) == YES) {
        return; 
      } else {
        NSArray *files = [notifdict objectForKey: @"files"];
        int column = [browser columnWithPath: path];
        
        if (column != -1) {
          [browser addCellsWithNames: files inColumn: column dimmed: NO];
        } 
      }
    }
	
  } 
}

- (void)setWatchersFromTimer:(id)sender
{
  [self setWatchers];
}

- (void)setWatchers
{
  int i;
  
  for (i = 0; i < [watchedPaths count]; i++) {
    [self setWatcherForPath: [watchedPaths objectAtIndex: i]];  
  }

  [[NSNotificationCenter defaultCenter] addObserver: self 
                		selector: @selector(watcherNotification:) 
                				name: GWFileWatcherFileDidChangeNotification
                			object: nil];
}

- (void)setWatcherForPath:(NSString *)path
{
	[delegate addWatcherForPath: path];
}

- (void)unsetWatchers
{
  int i;

  [[NSNotificationCenter defaultCenter] removeObserver: self 
                name: GWFileWatcherFileDidChangeNotification object: nil];

  for (i = 0; i < [watchedPaths count]; i++) {
    [self unsetWatcherForPath: [watchedPaths objectAtIndex: i]];  
  }
}

- (void)unsetWatcherForPath:(NSString *)path
{
	[delegate removeWatcherForPath: path];
}

- (void)closeNicely
{
  NSTimer *t;
  
  [self unsetWatchers]; 
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  t = [NSTimer timerWithTimeInterval: 2 target: self 
          selector: @selector(close:) userInfo: nil repeats: NO];                                             
  [[NSRunLoop currentRunLoop] addTimer: t forMode: NSDefaultRunLoopMode];
}

- (void)close:(id)sender
{
  [[self window] performClose: nil];
}


//
// ViewersProtocol
//
- (void)setSelectedPaths:(NSArray *)paths
{
  NSString *fullPath;
  NSArray *components;
  NSMutableArray *wpaths;
  NSString *s;
  BOOL isDir;
  int i, j;

	if ((paths == nil) || ([paths count] == 0) 
									|| ([paths isEqualToArray: selectedPaths])) {
		return;
	}
	
  ASSIGN (selectedPaths, paths);

	[delegate setTheSelectedPaths: paths];
    
  fullPath = [paths objectAtIndex: 0]; 
  [fm fileExistsAtPath: fullPath isDirectory: &isDir];
  if ((isDir == NO) || ([paths count] > 1)) {
    fullPath = [fullPath stringByDeletingLastPathComponent];
  }

  components = [fullPath pathComponents];  
  wpaths = [NSMutableArray arrayWithCapacity: 1];  
  s = [NSString string];        
   
  for (i = 0; i < [components count]; i++) {  
    s = [s stringByAppendingPathComponent: [components objectAtIndex: i]];
    [wpaths addObject: s];            
  }  
 
  if (watchedPaths == nil) {          
    ASSIGN (watchedPaths, wpaths);  
    [self setWatchers];

  } else {  
    int count = [wpaths count];

    for (i = 0; i < [watchedPaths count]; i++) {
      NSString *s1, *s2;

      s1 = [watchedPaths objectAtIndex: i];

      if (count > i) {
        s2 = [wpaths objectAtIndex: i];  
      } else {
        i = count;
        break;
      }

      if ([s1 isEqualToString: s2] == NO) {
        break;
      }    
    }

    for (j = i; j < [watchedPaths count]; j++) {  
      [self unsetWatcherForPath: [watchedPaths objectAtIndex: j]];
    }

    for (j = i; j < [wpaths count]; j++) {  
      [self setWatcherForPath: [wpaths objectAtIndex: j]];
    }

    ASSIGN (watchedPaths, wpaths);
  }
}

- (void)setCurrentSelection:(NSArray *)paths
{	
  if (paths == nil) {
    return;
  }

	[browser setPath: [paths objectAtIndex: 0]];

	if ([paths count] > 1) {
  	NSMutableArray *names = [NSMutableArray arrayWithCapacity: [paths count]];
		int i, column;
	
  	for (i = 0; i < [paths count]; i++) {
    	[names addObject: [[paths objectAtIndex: i] lastPathComponent]];
  	}

		column = [browser lastColumn];

		if (column > 0) {
			[browser setMultipleSelection: names 
													 inColumn: column - 1 sendAction: YES];
		}
	}
	
	[self setSelectedPaths: paths]; 
	
	[delegate updateTheInfoString];
}

- (NSArray *)selectedPaths
{
  return selectedPaths;
}

- (NSPoint)locationOfIconForPath:(NSString *)path
{
	if ([selectedPaths containsObject: path]) {
		NSPoint p = [browser lastIconPosition];
		return [self convertPoint: p toView: nil];
	}
	
	return NSZeroPoint;
}

- (void)setResizeIncrement:(int)increment
{
  resizeIncrement = increment;
  columnsWidth = ([[self window] frame].size.width - 16) / columns;
  [browser setMinColumnWidth: columnsWidth];
  [self resizeWithOldSuperviewSize: [self frame].size];
  [self setNeedsDisplay: YES];  
}

- (void)setAutoSynchronize:(BOOL)value
{
  autoSynchronize = value;
}

- (id)viewerView
{
  return browser;
}

- (BOOL)viewsApps
{
  return viewsapps;
}

- (void)selectAll
{
  [browser selectAllInLastColumn];
}

- (id)delegate
{
  return delegate;
}

- (void)setDelegate:(id)anObject
{	
  delegate = anObject;
}


- (void)createCellsInMatrix:(NSMatrix *)matrix 
                   ofColumn:(int)column 
                      isNew:(BOOL)isnew
{
  NSString *ptc;
  NSArray *files;
	NSArray *selectedCells;
  NSMutableArray *sortfiles;
  NSArray *hiddenFiles;
  NSMutableArray *paths; 
	NSUserDefaults *defaults;
	BOOL gsFileBrowserHideDotFiles;
  NSString *s, *h; 
  BOOL exists = NO, is_dir = NO; 
  NSString *defApp, *type;
  int i, count;
  int stype;
	
	selectedCells = [browser selectedCells];
	if (selectedCells != nil) {
		if (([selectedCells count] > 1) || ([selectedCells count] == 0)) {
			return;
		}
	}
		
  ptc = [browser pathToColumn: column];
  files = [fm directoryContentsAtPath: ptc];

	h = [ptc stringByAppendingPathComponent: @".hidden"];
	h = [NSString stringWithContentsOfFile: h];
	hiddenFiles = [h componentsSeparatedByString: @"\n"];
	
	defaults = [NSUserDefaults standardUserDefaults];
	gsFileBrowserHideDotFiles = [defaults boolForKey: @"GSFileBrowserHideDotFiles"];
	
	if (hiddenFiles != nil  ||  gsFileBrowserHideDotFiles) {	
		NSMutableArray *mutableFiles = AUTORELEASE ([files mutableCopy]);
	
		if (hiddenFiles != nil) {
	    [mutableFiles removeObjectsInArray: hiddenFiles];
	  }
	
		if (gsFileBrowserHideDotFiles) {
	    int j = [mutableFiles count] - 1;
	    
	    while (j >= 0) {
				NSString *file = (NSString *)[mutableFiles objectAtIndex: j];

				if ([file hasPrefix: @"."]) {
		    	[mutableFiles removeObjectAtIndex: j];
		  	}
				j--;
			}
	  }
		
		files = mutableFiles;
	}

  count = [files count];
  if (count == 0) {
		return;
	}
  
  paths = [NSMutableArray arrayWithCapacity: 1];
  
  for (i = 0; i < count; ++i) {
    s = [ptc stringByAppendingPathComponent: [files objectAtIndex: i]];
    [paths addObject: s];
  }
        	
	stype = [delegate sortTypeForDirectoryAtPath: ptc]; 
	
  paths = AUTORELEASE ([[paths sortedArrayUsingFunction: (int (*)(id, id, void*))compPaths
                                      context: (void *)stype] mutableCopy]);
       
  sortfiles = [NSMutableArray arrayWithCapacity: 1];
  for (i = 0; i < count; ++i) {
    [sortfiles addObject: [[paths objectAtIndex: i] lastPathComponent]];
  }  
  
  if (isnew == YES) {
    [matrix addColumn]; 
  }
  
  for (i = 0; i < count; ++i) {
    id cell;

    if (isnew == YES) {
      if (i != 0) {
		    [matrix insertRow: i];
      } 
    } else {
      [matrix insertRow: i];
    }
    
    cell = [matrix cellAtRow: i column: 0];    
    [cell setStringValue: [sortfiles objectAtIndex: i]];
    s = [ptc stringByAppendingPathComponent: [sortfiles objectAtIndex: i]];
    
    exists = [fm fileExistsAtPath: s isDirectory: &is_dir];

    if (is_dir == YES) {    
      [[NSWorkspace sharedWorkspace] getInfoForFile: s application: &defApp type: &type]; 
      [cell setLeaf: ((type == NSApplicationFileType) ? !viewsapps : NO)];
    } else {
		  [cell setLeaf: YES];
    }
        
  }
}

- (GWBrowser *)browser
{
  return browser;
}

- (void)resizeWithOldSuperviewSize:(NSSize)oldFrameSize
{
  NSRect r = [self frame];
  float w = r.size.width;
  float h = r.size.height; 
  int col = columns;
  
  [browser setFrame: NSMakeRect(0, 0, w, h - 2)];       

  if (autoSynchronize == YES) {
    columns = (int)[[self window] frame].size.width / resizeIncrement;
  
    if (col != columns) {
      [self renewAll];
    }
  }
}

- (BOOL)fileManager:(NSFileManager*)fileManager
				  shouldProceedAfterError:(NSDictionary*)errorDictionary
{
  return YES;
}

- (void)fileManager:(NSFileManager*)fileManager
	  willProcessPath:(NSString*)path
{

}

@end


//
// GWBrowser Delegate Methods
//

@implementation BrowserViewer (GWBrowserDelegate)

- (void)gwbrowser:(GWBrowser *)sender createRowsForColumn:(int)column
                       inMatrix:(NSMatrix *)matrix
{
  NSString *ptc = [sender pathToColumn: column];
  BOOL exists = NO, is_dir = NO; 

  exists = [fm fileExistsAtPath: ptc isDirectory: &is_dir];
  if (is_dir == NO) {	
    return;
  } else {
		NSString *defApp, *type;
		
		[[NSWorkspace sharedWorkspace] getInfoForFile: ptc application: &defApp type: &type]; 
		if ((type == NSApplicationFileType) && (viewsapps == NO)) {
    	return;
		}
  }
	
  [self createCellsInMatrix: matrix ofColumn: column isNew: YES];
  
  columns = [sender maxVisibleColumns];
}

- (void)currentSelectedPaths:(NSArray *)paths
{
  if (autoSynchronize == YES) {		
    [self setSelectedPaths: paths];
		[delegate addPathToHistory: paths];
		[delegate updateTheInfoString];
  }
}

- (void)openSelectedPaths:(NSArray *)paths newViewer:(BOOL)isnew
{
	[self setSelectedPaths: paths];	
	[delegate openTheSelectedPaths: paths newViewer: isnew]; 
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[delegate performFileOperationWithDictionary: opdict]; 
}

- (NSString *)getTrashPath
{
	return [delegate getTrashPath];
}																								

- (int)getSortTypeForDirectoryAtPath:(NSString *)path
{
	return [delegate sortTypeForDirectoryAtPath: path];
}

@end
