/*  -*-objc-*-
 *  IconsViewerIcon.m: Implementation of the IconsViewerIcon Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "IconsViewerIcon.h"
#include "Functions.h"
#include "GNUstep.h"

extern NSString *GWorkspaceRecycleOutOperation;

@implementation IconsViewerIcon

- (void)dealloc
{
  RELEASE (paths);
  TEST_RELEASE (fullPath);
  RELEASE (name);
	TEST_RELEASE (hostname);
  RELEASE (type);
	RELEASE (namelabel);
  RELEASE (icon);
  RELEASE (highlight);
  [super dealloc];
}

- (id)initForPaths:(NSArray *)fpaths delegate:(id)adelegate
{
  self = [super init];
  if (self) {
    NSFont *font;
    NSString *defApp = nil, *t = nil;
    int count;

    fm = [NSFileManager defaultManager];
		
    [self setFrame: NSMakeRect(0, 0, 64, 52)];
		TEST_RELEASE (paths);
    paths = [fpaths mutableCopy];
		[self setDelegate: adelegate];

		labelWidth = [delegate getCellsWidth] - 4;	
    font = [NSFont systemFontOfSize: 12];
    isSelect = NO; 
    dimmed = NO;
    count = [paths count];                    

    if (count == 1) {
      singlepath = YES;
      ASSIGN (fullPath, [paths objectAtIndex: 0]);
			
			if ([fullPath isEqualToString: @"/"]) {
				ASSIGN (name, fullPath);
				isRootIcon = YES;
			} else {
    		ASSIGN (name, [fullPath lastPathComponent]);
				isRootIcon = NO;
			}
			
      [[NSWorkspace sharedWorkspace] getInfoForFile: fullPath application: &defApp type: &t];      
      ASSIGN (type, t);
    } else {
      fullPath = nil;
      singlepath = NO;
      type = nil;
			isRootIcon = NO;
      name = [[NSString alloc] initWithFormat: @"%i items", count];
    }

    if (singlepath == YES) {
      ASSIGN (icon, [[NSWorkspace sharedWorkspace] iconForFile: fullPath]);    
    } else {
      ASSIGN (icon, [NSImage imageNamed: @"MultipleSelection.tiff"]);
    }
    
    ASSIGN (highlight, [NSImage imageNamed: @"CellHighlight.tiff"]);

		if (isRootIcon == YES) {
			NSHost *host = [NSHost currentHost];
			NSString *hname = [host name];
			NSRange range = [hname rangeOfString: @"."];

			if (range.length != 0) {	
				hname = [hname substringToIndex: range.location];
			} 			
			ASSIGN (hostname, hname);			
  	} else {
			hostname = nil;
		}

    namelabel = [NSTextField new];    
		[namelabel setFont: font];
		[namelabel setBezeled: NO];
		[namelabel setEditable: NO];
		[namelabel setSelectable: NO];
		[namelabel setAlignment: NSCenterTextAlignment];
	  [namelabel setBackgroundColor: [NSColor windowBackgroundColor]];
	  [namelabel setTextColor: [NSColor blackColor]];
		[self setLabelWidth]; 
    
    [self registerForDraggedTypes: [NSArray arrayWithObjects: NSFilenamesPboardType, nil]];
    
    dragdelay = 0;
    isDragTarget = NO;
    onSelf = NO;
  }
  return self;
}

- (void)setPaths:(NSArray *)fpaths
{
  NSString *defApp = nil, *t = nil;
  int count;

	TEST_RELEASE (paths);
  paths = [fpaths mutableCopy];
  count = [paths count];                    

  if (count == 1) {
    singlepath = YES;
    ASSIGN (fullPath, [paths objectAtIndex: 0]);
		if ([fullPath isEqualToString: @"/"]) {
			ASSIGN (name, fullPath);
			isRootIcon = YES;
		} else {
    	ASSIGN (name, [fullPath lastPathComponent]);
			isRootIcon = NO;
		}
    [[NSWorkspace sharedWorkspace] getInfoForFile: fullPath application: &defApp type: &t];      
    ASSIGN (type, t);
  } else {
    TEST_RELEASE (fullPath);
    fullPath = nil;
    singlepath = NO;
    type = nil;
		isRootIcon = NO;
    name = [[NSString alloc] initWithFormat: @"%i items", count];
  }

  if (singlepath == YES) {
    ASSIGN (icon, [[NSWorkspace sharedWorkspace] iconForFile: fullPath]);    
  } else {
    ASSIGN (icon, [NSImage imageNamed: @"MultipleSelection.tiff"]);
  }

	if (isRootIcon == YES) {
		NSHost *host = [NSHost currentHost];
		NSString *hname = [host name];
		NSRange range = [hname rangeOfString: @"."];

		if (range.length != 0) {	
			hname = [hname substringToIndex: range.location];
		} 			
		ASSIGN (hostname, hname);			
  } else {
		TEST_RELEASE (hostname);
		hostname = nil;
	}

  [self setLabelWidth]; 
}

- (void)select
{
  if (dimmed || isSelect) {
    return;
  }
  
  isSelect = YES;
	[namelabel setBackgroundColor: [NSColor whiteColor]];
  [self setLabelWidth]; 
	[delegate setLabelFrameOfIcon: self];	
	[self setNeedsDisplay: YES];
	[delegate unselectIconsDifferentFrom: self];	
	[delegate setTheCurrentSelection: paths];	
}

- (void)unselect
{
  if (isSelect == NO) {
    return;
  }

	isSelect = NO;
	[namelabel setBackgroundColor: [NSColor windowBackgroundColor]];
  [self setLabelWidth]; 
	[delegate setLabelFrameOfIcon: self];
	[self setNeedsDisplay: YES];
}

- (void)setLabelWidth
{
  NSFont *font = [NSFont systemFontOfSize: 12];
  NSRect rect = [namelabel frame];
	NSString *nstr = isRootIcon ? hostname : name;
  	  
	labelWidth = [delegate getCellsWidth] - 4;	
	
  if (isSelect == YES) {
    [namelabel setFrame: NSMakeRect(0, 0, [font widthOfString: nstr] + 8, 14)];
    [namelabel setStringValue: nstr];
  } else {
    int width = (int)[[namelabel font] widthOfString: nstr] + 8;
    if (width > labelWidth) {
      width = labelWidth;
    }
    [namelabel setFrame: NSMakeRect(0, 0, width, 14)];  
    [namelabel setStringValue: cutLabel(nstr, namelabel, width - 8)];  
  }

  [(NSView *)delegate setNeedsDisplayInRect: rect];
}

- (NSTextField *)myLabel
{
  return namelabel;
}

- (NSString *)type
{
  return type;
}

- (NSArray *)paths
{
  return paths;
}

- (NSString *)name
{
  return name;
}

- (NSString *)hostname
{
	return hostname;
}

- (NSSize)iconShift
{
	NSRect r = [self frame];
	NSSize s = [icon size];
	
	return NSMakeSize((r.size.width - s.width) / 2, (r.size.height - s.height) / 2);	
}

- (BOOL)isSinglePath
{
  return singlepath;
}

- (BOOL)isSelect
{
  return isSelect;
}

- (void)setDimmed:(BOOL)value
{
  dimmed = value;
  
  if (dimmed == YES) {
    [namelabel setTextColor: [NSColor grayColor]];
  } else {
    [namelabel setTextColor: [NSColor blackColor]];
  }
  
  [namelabel setNeedsDisplay: YES];
}

- (BOOL)isDimmed
{
  return dimmed;
}

- (BOOL)isRootIcon
{
	return isRootIcon;
}

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent 
{
  return YES;
}

- (void)mouseDown:(NSEvent *)theEvent
{
	int count = [theEvent clickCount];    
  
  if (dimmed == YES) {
    return;
  }
  
	if(count == 1) {
		if([theEvent modifierFlags] == 2)  {
			[delegate setShiftClickValue: YES];     
			if (isSelect == YES) {
				[self unselect];
				return;
      } else {
				[self select];
			}
		} else {
			[delegate setShiftClickValue: NO];
      if (isSelect == NO) {       
				[self select];
			}
		}
	} else {
		unsigned int modifier = [theEvent modifierFlags];
		
		[delegate openTheCurrentSelection: paths newViewer: (modifier == NSControlKeyMask)];
	}  
}

- (void)mouseDragged:(NSEvent *)theEvent
{
  if (dimmed == YES) {
    return;
  }

	if(dragdelay < 5) {
    dragdelay++;
    return;
  }
  
  [self startExternalDragOnEvent: theEvent];
}

- (id)delegate
{
  return delegate;
}

- (void)setDelegate:(id)aDelegate
{
	delegate = aDelegate;
}

- (void)drawRect:(NSRect)rect
{
	NSRect r = [self bounds];
  NSSize s = [icon size];
	NSPoint p = NSMakePoint((r.size.width - s.width) / 2, (r.size.height - s.height) / 2);  
	    	
	if(isSelect) {
		[highlight compositeToPoint: NSZeroPoint operation: NSCompositeSourceOver];
	}

	[icon compositeToPoint: p operation: NSCompositeSourceOver];
}

@end

@implementation IconsViewerIcon (DraggingSource)

- (void)startExternalDragOnEvent:(NSEvent *)event
{
	NSEvent *nextEvent;
  NSPoint dragPoint;
  NSPasteboard *pb;
  NSImage *dragIcon;
  
	nextEvent = [[self window] nextEventMatchingMask:
    							NSLeftMouseUpMask | NSLeftMouseDraggedMask];

  if([nextEvent type] != NSLeftMouseDragged) {
   	return;
  }
  
  dragPoint = [nextEvent locationInWindow];
  dragPoint = [self convertPoint: dragPoint fromView: nil];

	pb = [NSPasteboard pasteboardWithName: NSDragPboard];	
  [self declareAndSetShapeOnPasteboard: pb];
	
  if (singlepath == YES) {
    NSArray *selection = [delegate getTheCurrentSelection];
		
    if ([selection count] > 1) {
      dragIcon = [NSImage imageNamed: @"MultipleSelection.tiff"];
    } else {
      dragIcon = [[NSWorkspace sharedWorkspace] iconForFile: fullPath]; 
    }   
  } else {
    dragIcon = [NSImage imageNamed: @"MultipleSelection.tiff"];
  }

  [self dragImage: dragIcon
               at: dragPoint 
           offset: NSZeroSize
            event: event
       pasteboard: pb
           source: self
        slideBack: NO];
}

- (void)draggedImage:(NSImage *)anImage 
						 endedAt:(NSPoint)aPoint 
					 deposited:(BOOL)flag
{
	dragdelay = 0;
  onSelf = NO;
}

- (void)declareAndSetShapeOnPasteboard:(NSPasteboard *)pb
{
  NSArray *dndtypes;
  NSArray *selection;
  NSData *pbData;

  dndtypes = [NSArray arrayWithObject: NSFilenamesPboardType];
  [pb declareTypes: dndtypes owner: nil];
	selection = [delegate getTheCurrentSelection];
  pbData = [NSArchiver archivedDataWithRootObject: selection];
  [pb setData: pbData forType: NSFilenamesPboardType];
}

- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)flag
{
  return NSDragOperationAll;
}

@end

@implementation IconsViewerIcon (DraggingDestination)

- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
	NSString *fromPath;
  NSString *buff;
	int i, count;

	pb = [sender draggingPasteboard];
  if([[pb types] indexOfObject: NSFilenamesPboardType] != NSNotFound) {

    pbData = [pb dataForType: NSFilenamesPboardType];
    sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];  

    if ([paths isEqualToArray: sourcePaths]) {
      onSelf = YES;
      isDragTarget = YES;
      return NSDragOperationAll;
    }

    if (([type isEqualToString: NSDirectoryFileType] == NO)
            && ([type isEqualToString: NSFilesystemFileType] == NO)) {
      return NSDragOperationNone;
    }
    
	  count = [sourcePaths count];
	  fromPath = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];
    
	  if (count == 0) {
		  return NSDragOperationNone;
    } 
  
	  if ([fm isWritableFileAtPath: fullPath] == NO) {
		  return NSDragOperationNone;
	  }
  
	  if ([fullPath isEqualToString: fromPath]) {
		  return NSDragOperationNone;
    }  
  
	  for (i = 0; i < count; i++) {
		  if ([fullPath isEqualToString: [sourcePaths objectAtIndex: i]]) {
		    return NSDragOperationNone;
		  }
	  }
       
	  buff = [NSString stringWithString: fullPath];
	  while (1) {
		  for (i = 0; i < count; i++) {
			  if ([buff isEqualToString: [sourcePaths objectAtIndex: i]]) {
 		      return NSDragOperationNone;
			  }
		  }
      if ([buff isEqualToString: @"/"] == YES) {
        break;
      }            
		  buff = [buff stringByDeletingLastPathComponent];
	  }

    isDragTarget = YES;
		
    ASSIGN (icon, [NSImage imageNamed: @"FileIcon_Directory_Open.tiff"]);
    [self setNeedsDisplay: YES];
		
		sourceDragMask = [sender draggingSourceOperationMask];
	
		if (sourceDragMask == NSDragOperationCopy) {
			return NSDragOperationCopy;
		} else if (sourceDragMask == NSDragOperationLink) {
			return NSDragOperationLink;
		} else {
			return NSDragOperationAll;
		}
  }
      
  return NSDragOperationNone;
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender
{
  NSDragOperation sourceDragMask;
	
	if (isDragTarget == NO) {
		return NSDragOperationNone;
	}

	sourceDragMask = [sender draggingSourceOperationMask];
	
	if (sourceDragMask == NSDragOperationCopy) {
		return NSDragOperationCopy;
	} else if (sourceDragMask == NSDragOperationLink) {
		return NSDragOperationLink;
	} else {
		return NSDragOperationAll;
	}

	return NSDragOperationNone;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender
{
  if(isDragTarget == YES) {
    isDragTarget = NO;
    if (onSelf == NO) {      
      ASSIGN (icon, [[NSWorkspace sharedWorkspace] iconForFile: fullPath]);
      [self setNeedsDisplay: YES];
    }
    onSelf = NO;
  }
}

- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender
{
	return isDragTarget;
}

- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender
{
	return YES;
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender
{
	NSPasteboard *pb;
  NSDragOperation sourceDragMask;
	NSData *pbData;
	NSArray *sourcePaths;
  NSString *operation, *source;
  NSMutableArray *files;
	NSMutableDictionary *opDict;
	NSString *trashPath;
  int i;

  isDragTarget = NO;

  if (onSelf == YES) {
    onSelf = NO;
    return;
  }
  
  ASSIGN (icon, [[NSWorkspace sharedWorkspace] iconForFile: fullPath]);
  [self setNeedsDisplay: YES];

	sourceDragMask = [sender draggingSourceOperationMask];  
  pb = [sender draggingPasteboard];
  pbData = [pb dataForType: NSFilenamesPboardType];
  sourcePaths = [NSUnarchiver unarchiveObjectWithData: pbData];
  
  source = [[sourcePaths objectAtIndex: 0] stringByDeletingLastPathComponent];

	trashPath = [delegate getTrashPath];

	if ([source isEqualToString: trashPath]) {
		operation = GWorkspaceRecycleOutOperation;
	} else {
		if (sourceDragMask == NSDragOperationCopy) {
			operation = NSWorkspaceCopyOperation;
		} else if (sourceDragMask == NSDragOperationLink) {
			operation = NSWorkspaceLinkOperation;
		} else {
			operation = NSWorkspaceMoveOperation;
		}
  }
  
  files = [NSMutableArray arrayWithCapacity: 1];    
  for(i = 0; i < [sourcePaths count]; i++) {    
    [files addObject: [[sourcePaths objectAtIndex: i] lastPathComponent]];
  }  

	opDict = [NSMutableDictionary dictionaryWithCapacity: 4];
	[opDict setObject: operation forKey: @"operation"];
	[opDict setObject: source forKey: @"source"];
	[opDict setObject: fullPath forKey: @"destination"];
	[opDict setObject: files forKey: @"files"];
	
	[delegate performFileOperationWithDictionary: opDict];	
}

@end
